//	Exerter.h - something that can exert forces, impulses, and torques.
//
//  Copyright (C) 2002 Sam Varner
//
//  This file is part of Vamos Automotive Simulator.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef _EXERTER_H_
#define _EXERTER_H_

#include <vamos/geometry/Three_Vector.h>

namespace Vamos_Body
{
  //* An Exerter is capable of producing forces, impulses, and
  //torques.  These exertions cannot be applied to a rigid body
  //because an exerter has no location or orientation.  See Particle.
  class Exerter
  {
  protected:
	// The resultant force exerted by the component.
	Vamos_Geometry::Three_Vector m_force;

	// The impulse exerted by the component.
	Vamos_Geometry::Three_Vector m_impulse;

	// The resultant torque exerted by the component.
	Vamos_Geometry::Three_Vector m_torque;

  public:
	//** Constructor
	Exerter () {};

	//** Destructor
	virtual ~Exerter () {};

	// Find and store the forces, impulses, and torques for the
	// current configuration.
	virtual void find_forces () {};

	// Propagate the Particle forward in time by TIME.
	virtual void propagate (double time) {};

	// Undo the last propagation.
	virtual void rewind () {};

	// Do any necessary cleanup at the end of a time step.
	virtual void end_timestep () {};

	// Return the force exerted on the rigid body in the body's frame.
	virtual Vamos_Geometry::Three_Vector force () const { return m_force; }

	// Return the impulse exerted on the rigid body in the body's
	// frame.
	virtual Vamos_Geometry::Three_Vector impulse () const { return m_impulse; }

	// Return the torque exerted on the rigid body in the body's frame.
	virtual Vamos_Geometry::Three_Vector torque () const { return m_torque; }

	// Set the force, impulse and torque to zero;
	virtual void reset ();
  };
}

#endif // not _EXERTER_H_
