//  Aerodynamic_Device.h - a particle that produces drag and lift.
//
//  Copyright (C) 2002 Sam Varner
//
//  This file is part of Vamos Automotive Simulator.
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#ifndef _AERODYNAMIC_DEVICE_H_
#define _AERODYNAMIC_DEVICE_H_

#include <vamos/body/Particle.h>
#include <vamos/geometry/Three_Vector.h>

namespace Vamos_Body
{
  //* An Aerodynamic_Device is a particle that may produce drag or lift.
  class Aerodynamic_Device : public Particle
  {
  protected:
	// The current wind velocity vector.
	Vamos_Geometry::Three_Vector m_wind_vector;

	// The current air density.
	double m_density;

  public:
	//** Constructor
	Aerodynamic_Device (const Vamos_Geometry::Three_Vector& position);

	// Calculate the drag and lift due to WIND_VECTOR.  WIND_VECTOR is
	// supplied by the Body in the Body's frame.  DENSITY is the denisty
	// of the atmosphere.
	void wind (const Vamos_Geometry::Three_Vector& wind_vector, 
			   double density);

	// Find and store the forces, impulses, and torques for the
	// current configuration.
	virtual void find_forces () = 0;
  };

  //* An Aerodynamic_Device that produces drag.
  class Drag : public Aerodynamic_Device
  {
  protected:
	// The frontal area of the particle.
	double m_frontal_area;

	// The coefficient of drag.
	double m_drag_coefficient;

  public:
	//** Constructor
	Drag (const Vamos_Geometry::Three_Vector& position,
		  double frontal_area, double drag_coefficient);

	// Find and store the forces, impulses, and torques for the
	// current configuration.
	virtual void find_forces ();
  };

  class Wing : public Drag
  {
	// The area of the surface of the wing.
	double m_surface_area;

	// The coefficient of lift.
	double m_lift_coefficient;

	// The efficiency of the wing (< 1.0, > 0).
	double m_efficiency;

  public:
	//** Constructor
	Wing (const Vamos_Geometry::Three_Vector& position,
		  double frontal_area, double drag_coefficient,
		  double surface_area, double lift_coefficient,
		  double efficiency);

	// Find and store the forces, impulses, and torques for the
	// current configuration.
	void find_forces ();
  };
}

#endif // !_AERODYNAMIC_DEVICE_H_
