/*
 * mp2_decoder.h
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * Or, point your browser to http://www.gnu.org/copyleft/gpl.html
 *
 * Copyright (C) 2003 Sven Goethel, <sven@jausoft.com>
 */

#ifndef __MP2_DECODER_H
#define __MP2_DECODER_H

#include <mad.h>
#include "shm_memory_tool.h"
#include "any_decoder.h"

#define HDR_SIZE        9
#define LPCM_SIZE       7
#define MP2_BUFFER_SIZE 0x4000

typedef struct {
  unsigned char PES[HDR_SIZE];
  unsigned char LPCM[LPCM_SIZE];
  unsigned char Data[SPDIF_BURST_SIZE];
} LPCMFrame ;

// ----------------------------------------------------------------

#define OUT_BITS 16                                          // output 16 bit samples to DVB driver
#define OUT_FACT (OUT_BITS/8*2)                              // output factor is 16 bit & 2 channels -> 4 bytes
// cResample
#define MAX_NSAMPLES (1152*7)                                // max. buffer for resampled frame
// cNormalize
#define MAX_GAIN   3.0                                       // max. allowed gain
#define LIM_ACC    12                                        // bit, accuracy for lookup table
#define F_LIM_MAX  (mad_fixed_t)((1<<(MAD_F_FRACBITS+2))-1)  // max. value covered by lookup table
#define LIM_SHIFT  (MAD_F_FRACBITS-LIM_ACC)                  // shift value for table lookup
#define F_LIM_JMP  (mad_fixed_t)(1<<LIM_SHIFT)               // lookup table jump between values
// cLevel
#define POW_WIN 100                                          // window width for smoothing power values
#define EPSILON 0.00000000001                                // anything less than EPSILON is considered zero
// cMP3Player
#define LEN_CORR        3
#define SPEEDCHECKSTART ((MP3BUFSIZE*1000/32000/2/2)+1000)   // play time to fill buffers before speed check starts (ms)
#define SPEEDCHECKTIME  3000                                 // play time for speed check (ms)

// --- cResample ----------------------------------------------------------------

class cResample {
private:
  mad_fixed_t ratio;
  mad_fixed_t step;
  mad_fixed_t last;
  mad_fixed_t resampled[MAX_NSAMPLES];
public:
  bool SetInputRate(unsigned int oldrate, unsigned int newrate);
  unsigned int ResampleBlock(unsigned int nsamples, const mad_fixed_t *old);
  const mad_fixed_t *Resampled(void) { return resampled; }
  };

// --- cScale ----------------------------------------------------------------

// The dither code has been adapted from the madplay project
// (audio.c) found in the libmad distribution

enum eAudioMode { amRound, amDither };

class cScale {
private:
  enum { MIN=-MAD_F_ONE, MAX=MAD_F_ONE - 1 };
#ifdef DEBUG
  // audio stats
  unsigned long clipped_samples;
  mad_fixed_t peak_clipping;
  mad_fixed_t peak_sample;
#endif
  // dither
  struct dither {
    mad_fixed_t error[3];
    mad_fixed_t random;
    } leftD, rightD;
  //
  inline mad_fixed_t Clip(mad_fixed_t sample, bool stats=true);
  inline signed long LinearRound(mad_fixed_t sample);
  inline unsigned long Prng(unsigned long state);
  inline signed long LinearDither(mad_fixed_t sample, struct dither *dither);
public:
  void Init(void);
  void Stats(void);
  unsigned int ScaleBlock(unsigned char *data, unsigned int size, unsigned int &nsamples, const mad_fixed_t * &left, const mad_fixed_t * &right, eAudioMode mode);
  };


//
// #define FILE_COPY
//

#ifdef FILE_COPY
	#include <stdio.h>
#endif

class MP2Decoder : public AnyDecoder 
{
    public:
	int       size;		  // set res. size
	int       pcm_netto_size;  // set res. size, just Data ..

	unsigned int remaining_samples;  // set remain samples .. 
	unsigned int remaining_mp2;      // set remain mp2 bytes .. 
	unsigned int digested_mp2;       // set eaten mp2 bytes .. 
        unsigned int sample_rate;        // set Hz

	int      channels;	  // set num

	MP2Decoder() ;

	virtual ~MP2Decoder() ;

	void  finish();
	void  init();
	void  empty();

	int   pushData ( const uint_8 *head, int len );

	bool  decode (int maxwritesize=0);

	void  setExternalOutBuffer ( unsigned char * ExternalBuffer,
				     size_t          ExternalBufferSize );

	LPCMFrame & lpcmFrame() { return *lpcmFrame_ptr; }

	/**
  	 * delay > 0 : audio is delay ms behind video
  	 * delay == 0: audio is in sync with video
	 */
	void setDelay(uint_32 d) { delay = d; }
	uint_32 getDelay() { return delay; }

    private:
	// priv. use ..
	LPCMFrame *lpcmFrame_ptr;	  // given mem

	unsigned char * ExternalPCMData;
	size_t          ExternalPCMDataSize;

	// we use double bouffering ..
	uint_8* Mp2InputBuffer_cur;
	uint_8* Mp2InputBuffer_nxt;

	struct mad_stream m_stream;
	struct mad_frame  m_frame;
	struct mad_synth  m_synth;
	mad_timer_t       m_timer;
	unsigned long     FrameCount;

        cResample resample[2];
        unsigned int nsamples[2];
        const mad_fixed_t *data[2];
        cScale scale;

	unsigned char * abs_mp2_pos;

	uint_32 delay;

	bool   firstTime;
	bool   madInit;

	void  pushSamples2PCM(int maxwritesize);

	int  Convert(unsigned char * Data, size_t DataSize);
	static unsigned short MadFixedToUshort(mad_fixed_t Fixed);

#ifdef FILE_COPY
	FILE *fptr_raw, *fptr_mp2;
	int  fnr;
#endif
	static int instancenr;

};


#endif // __MP2_H
