#!/usr/bin/python

# VBoxGtk: A VirtualBox GTK+ GUI
# Copyright (C) 2008 Francisco J. Vazquez-Araujo, Spain
# franjva at gmail dot com

# This file is part of VBoxGtk.

# VBoxGtk is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# VBoxGtk is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with VBoxGtk.  If not, see <http://www.gnu.org/licenses/>.



# Model (VOs). Declaration of VM, VDI, NIC, Shared and Snapshots classes.

## VM 

import copy

class VM:
    def __init__(self, name):
        self.name = name
        self.state = 'powered off'
        self.ostype = 'unknown'
        self.hw = Hardware()
        self.snapshots = []
        self.need_saving = False
        self.need_saving_vdi = False

    def start(self):
        if self.state == 'running': return
        self.state = 'running'
        self.hw.transient_dvd = self.hw.dvd
        
    def sleep(self):
        if self.state != 'running': return
        self.state = 'saved'
        
    def stop(self):
        if self.state == 'saved': return
        self.state = 'powered off'
        self.hw.transient_dvd = None
        sdup = self.hw.shared_folders[:]
        for s in sdup:
            if s.transient: self.hw.shared_folders.remove(s)
            
    def update_settings(self, ostype, hwvirtex, mem, vmem, dvdpasstrough):
        if not self.state in [ 'powered off', 'aborted' ]: return
        self.hw.ostype = ostype
        self.hw.mem = mem
        self.hw.vmem = vmem
        self.hw.hwvirtex = hwvirtex
        self.hw.dvdpassthrough = dvdpasstrough
        self.need_saving = True
        
    def update_hd(self, hd):
        if not self.state in [ 'powered off', 'aborted' ]: return
        if self.hw.hd != None and len(self.hw.hd.uuid) < 2: self.hw.hd.attached_to.remove(self)
        if hd != None and len(hd.uuid) < 2: hd.attached_to.append(self)
        self.hw.hd = hd
        self.need_saving = True
        self.need_saving_vdi = True
        
    def update_dvd(self, dvd): # FIXME: put together with hd
        if not self.state in [ 'powered off', 'aborted' ]: return
        if self.hw.dvd != None: self.hw.dvd.attached_to.remove(self)
        if dvd != None: dvd.attached_to.append(self)
        self.hw.dvd = dvd
        self.need_saving = True
        self.need_saving_vdi = True
        
    def update_devices(self, audiodev, nicnumber, mac, attachment, nicdevice, cable, trace, tracefile, intnet):
        if not self.state in [ 'powered off', 'aborted' ]: return
        self.hw.audiodev = audiodev
        if self.hw.nics[nicnumber] == None: 
            self.hw.nics[nicnumber] = NIC(mac, attachment, nicdevice, cable, trace, tracefile, intnet)
        else:
            self.hw.nics[nicnumber].mac = mac
            self.hw.nics[nicnumber].attachment = attachment
            self.hw.nics[nicnumber].device = nicdevice
            self.hw.nics[nicnumber].cable = cable
            self.hw.nics[nicnumber].trace = trace
            if trace: self.hw.nics[nicnumber].tracefile = tracefile
            if attachment == 'intnet': self.hw.nics[nicnumber].intnet = intnet
        self.need_saving = True
        
    def add_shared(self, shared):
        if self.state == 'saved': return
        self.hw.shared_folders.append(shared)

    def del_shared(self, shared_num):
        if self.state == 'saved': return
        del self.hw.shared_folders[shared_num]

    def take_snapshot(self, snapshot):
        snapshot.hw = copy.copy(self.hw)
        self.snapshots.append(snapshot)

    def discard_snapshot(self, snapshot_num):
        if not self.state in [ 'powered off', 'aborted' ]: return
        del self.snapshots[snapshot_num]

    def discard_current_state(self):
        if not self.state in [ 'powered off', 'aborted' ]: return
        if self.hw.hd != None and len(self.hw.hd.uuid) < 2: self.hw.hd.attached_to.remove(self)
        if self.hw.dvd != None: self.hw.dvd.attached_to.remove(self)
        self.hw = copy.copy(self.snapshots[-1].hw)
        if self.hw.hd != None and len(self.hw.hd.uuid) < 2 and not self in self.hw.hd.attached_to: self.hw.hd.attached_to.append(self)
        if self.hw.dvd != None and not self in self.hw.dvd.attached_to: self.hw.dvd.attached_to.append(self)

    def edit_snapshot(self, snapshot_num, name):
        self.snapshots[snapshot_num].name = name

        
## Hardware

class Hardware:
    def __init__(self):
        self.hwvirtex = True
        self.mem = 128
        self.vmem = 8
        self.hd = None
        self.dvd = None
        self.transient_dvd = None
        self.dvdpassthrough = False
        self.audiodev = 'none'
        self.nics = [None, None, None, None]
        self.shared_folders = []

## VDI 
    
class VDI:
    def __init__(self, path, vtype):
        self.path = path
        self.uuid = []
        self.type = vtype
        self.attached_to = []
        self.attached_to_in_db = None


## NIC 
    
class NIC:
    def __init__(self, mac, attachment, device, cable, trace, tracefile = '', intnet = ''):
        self.mac = mac
        self.attachment = attachment
        self.device = device
        self.cable = cable
        self.trace = trace
        self.tracefile = tracefile
        self.intnet = intnet


    
## Shared 
    
class Shared:
    def __init__(self, name, path, writable, transient = False):
        self.name = name
        self.path = path
        self.writable = writable
        self.transient = transient
    
    
        
## Snapshot 
    
class Snapshot:
    def __init__(self, name, uuid = 0, comment = None):
        self.name = name
        self.uuid = uuid
        self.comment = comment
        self.hw = Hardware()
    
