#!/usr/bin/python

# VBoxGtk: A VirtualBox GTK+ GUI
# Copyright (C) 2008 Francisco J. Vazquez-Araujo, Spain
# franjva at gmail dot com

# This file is part of VBoxGtk.

# VBoxGtk is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.

# VBoxGtk is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with VBoxGtk.  If not, see <http://www.gnu.org/licenses/>.



# VBoxMgr class. Manages VMs and VDIs.

import util
import os

import vboxdao_vboxmanage
import vboxdao_xml
import vmvdi

class VBoxMgr:
    def __init__(self, vboxrunner):
        self.vboxiface = None
        self.vboxrunner = vboxrunner
        self.vboxdao_xml = vboxdao_xml.VBoxDAOXML()
        vbox_properties = vboxdao_vboxmanage.get_system_properties()
        self.version, self.ose, self.min_ram, self.max_ram, self.max_vram, self.max_vdi, self.default_vdi_folder = vbox_properties
        self.nicdevices = [('Am79C970A','Am79C970A'), ('Am79C973','Am79C973')]
        self.audiodevs = [('none','None'), ('null','Null'), ('oss','OSS'), ('alsa','ALSA'), ('pulse','PulseAudio')]
        if self.version >= '1.6.0' or not self.ose:
            self.nicdevices.append(('82540EM','82540EM'))
        if self.version >= '1.6.2':
            self.nicdevices.append(('82543GC','82543GC'))
        self.attachments = [('none','Disabled'), ('null','Null'), ('nat','NAT'), ('hostif','Host interface'), ('intnet','Internal network')]
        self.ostypes = vboxdao_vboxmanage.get_ostypes()
        self.vdis = self.vboxdao_xml.get_vdis()
        self.vdis += vboxdao_vboxmanage.get_hostdvds()
        self.vms = self.vboxdao_xml.get_vms()
        self.vboxrunner.init()
        self.vboxrunner.query_vm_states(self.vms, self.vdis)
    
    def exit(self):
        stop_vms = self.vboxrunner.mgr_finished()
        for i, vm in enumerate(self.vms):
            if vm.state == 'running':
                if stop_vms: self.stop_vm(i)
                continue
            self.save_vm(i)
        
    ## VDI management

    def create_hd(self, hd_path, hd_size, dynamic):
        if hd_path == util.basename(hd_path): hd_path = self.default_vdi_folder + hd_path
        hd = vmvdi.VDI(hd_path, 'disk')
        if vboxdao_vboxmanage.create_hd(self.vboxiface, hd, hd_size, dynamic): return 1
        self.vdis.append(hd)
        return 0

    def add_vdi(self, vdi_path, vdi_type):
        vdi = vmvdi.VDI(vdi_path, vdi_type)
        if vboxdao_vboxmanage.add_vdi(self.vboxiface, vdi): return 1
        self.vdis.append(vdi)
        return 0

    def remove_vdi(self, vdi_number, delete_file = False):
        vdi = self.vdis[vdi_number]
        if vdi.attached_to_in_db != None: vboxdao_vboxmanage.empty_vdis(self.vboxiface, vdi.attached_to_in_db)
        if vboxdao_vboxmanage.remove_vdi(self.vboxiface, vdi): return 1
        if delete_file: os.remove(vdi.path)
        del self.vdis[vdi_number]
        return 0
        
    ## VM management
    
    def create_vm(self, vm_name, hd, ostype):
        if vboxdao_vboxmanage.create_vm(self.vboxiface, vm_name): return 1
        vm = vmvdi.VM(vm_name)
        vm.ostype = ostype
        vm.nics = [None, None, None, None]
        vm.update_hd(hd)
        self.vms.append(vm)
        return 0

    def delete_vm(self, vm_number):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.empty_vdis(self.vboxiface, vm): return 1
        if vm.hw.dvd != None: vm.hw.dvd.attached_to.remove(vm)
        if vm.hw.hd != None: vm.hw.hd.attached_to.remove(vm)
        for vdi in self.vdis:
            if vdi.attached_to_in_db != vm: continue
            vdi.attached_to_in_db = None
            break
        if vboxdao_vboxmanage.delete_vm(self.vboxiface, vm.name): return 1
        del self.vms[vm_number]
        return 0

    def execute_vm(self, vm_number):
        if self.save_vm(vm_number): return 1
        vm = self.vms[vm_number]
        self.vboxrunner.execute_vm(vm)
        vm.start()
        return 0
        
    def sleep_vm(self, vm_number):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.change_state(self.vboxiface, vm, 'savestate'): return 1
        vm.sleep()
        return 0

    def stop_vm(self, vm_number):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.change_state(self.vboxiface, vm, 'poweroff'): return 1
        vm.stop()
        return 0
        
    def save_vm(self, vm_number):
        vm = self.vms[vm_number]
        hd = vm.hw.hd
        if hd != None: att_to = hd.attached_to_in_db
        else: att_to = None
        if att_to != None and att_to != vm and len(hd.uuid) < 2:
            if vboxdao_vboxmanage.empty_vdis(self.vboxiface, att_to): return 1
        if vm.need_saving_vdi:
            if vboxdao_vboxmanage.empty_vdis(self.vboxiface, vm): return 1
        for vdi in self.vdis:
            if vdi.attached_to_in_db != vm: continue
            vdi.attached_to_in_db = None
            break
        if vm.need_saving:
            if vboxdao_vboxmanage.save_vm(self.vboxiface, vm): return 1
            vm.need_saving_vdi = False
            vm.need_saving = False
        if hd != None and len(hd.uuid) < 2: hd.attached_to_in_db = vm
        return 0

    def update_hd(self, vm_number, hd):
        self.vms[vm_number].update_hd(hd)
        return 0
        
    def update_dvd(self, vm_number, dvd):
        vm = self.vms[vm_number]
        if vm.state == 'running': 
            if vboxdao_vboxmanage.swap_dvd(self.vboxiface, vm, dvd): return 1
            self.vboxrunner.swap_dvd(vm, dvd)
            vm.transient_dvd = dvd
        else: vm.update_dvd(dvd)
        return 0

    def add_shared(self, vm_number, name, path, writable):
        vm = self.vms[vm_number]
        shared = vmvdi.Shared(name, path, writable, vm.state == 'running')
        if vboxdao_vboxmanage.save_shared(self.vboxiface, vm, shared): return 1
        if vm.state == 'running': self.vboxrunner.add_shared(vm, shared)
        vm.add_shared(shared)
        return 0
        
    def del_shared(self, vm_number, shared_num):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.delete_shared(self.vboxiface, vm, shared_num): return 1
        if vm.state == 'running': self.vboxrunner.del_shared(vm, shared_num)
        vm.del_shared(shared_num)
        return 0

    def take_snapshot(self, vm_number, name):
        self.save_vm(vm_number)
        vm = self.vms[vm_number]
        snapshot = vmvdi.Snapshot(name)
        if vboxdao_vboxmanage.take_snapshot(self.vboxiface, vm, snapshot): return 1
        vm.take_snapshot(snapshot)
        if vm.hw.hd != None: vm.hw.hd.uuid.append('new-uuid') # The uuid doesn't matter, only the number of uuids
        return 0

    def discard_snapshot(self, vm_number, snapshot_num):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.discard_snapshot(self.vboxiface, vm, snapshot_num): return 1
        vm.discard_snapshot(snapshot_num)
        if vm.hw.hd != None and len(vm.hw.hd.uuid) > 1: del vm.hw.hd.uuid[-1] # Again, the uuid doesn't matter, only the uuid count
        return 0

    def discard_current_state(self, vm_number):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.discard_current_state(self.vboxiface, vm): return 1
        vm.discard_current_state()
        return 0

    def edit_snapshot(self, vm_number, snapshot_num, name):
        vm = self.vms[vm_number]
        if vboxdao_vboxmanage.edit_snapshot(self.vboxiface, vm, snapshot_num, name): return 1
        vm.edit_snapshot(snapshot_num, name)
        return 0
