/* valasourcefile.vala
 *
 * Copyright (C) 2006-2008  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include <vala/valasourcefile.h>
#include <gee/arraylist.h>
#include <gee/hashset.h>
#include <gee/collection.h>
#include <gee/readonlylist.h>
#include <gee/readonlyset.h>
#include <glib/gstdio.h>
#include <vala/valasourcefilecycle.h>
#include <vala/valacodecontext.h>
#include <vala/valanamespacereference.h>
#include <vala/valasymbol.h>
#include <vala/valacodevisitor.h>
#include <vala/valatypesymbol.h>
#include <vala/valamethod.h>
#include <vala/valafield.h>
#include <vala/valaproperty.h>
#include <vala/valaconstant.h>
#include <vala/valaformalparameter.h>
#include <vala/valadatatype.h>
#include <vala/valasourcereference.h>
#include <vala/valareport.h>




struct _ValaSourceFilePrivate {
	char* _filename;
	char* _comment;
	gboolean _external_package;
	ValaSourceFileCycle* _cycle;
	gboolean _is_cycle_head;
	gint _mark;
	ValaCodeContext* _context;
	GeeList* using_directives;
	GeeList* nodes;
	char* cheader_filename;
	char* csource_filename;
	char* cinclude_filename;
	GeeList* header_external_includes;
	GeeList* header_internal_includes;
	GeeList* source_external_includes;
	GeeList* source_internal_includes;
	GeeList* header_internal_full_dependencies;
	GeeList* header_internal_dependencies;
	GeeSet* source_symbol_dependencies;
	GeeArrayList* source_array;
	GMappedFile* mapped_file;
};

#define VALA_SOURCE_FILE_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SOURCE_FILE, ValaSourceFilePrivate))
enum  {
	VALA_SOURCE_FILE_DUMMY_PROPERTY,
	VALA_SOURCE_FILE_FILENAME,
	VALA_SOURCE_FILE_COMMENT,
	VALA_SOURCE_FILE_EXTERNAL_PACKAGE,
	VALA_SOURCE_FILE_CYCLE,
	VALA_SOURCE_FILE_IS_CYCLE_HEAD,
	VALA_SOURCE_FILE_MARK,
	VALA_SOURCE_FILE_CONTEXT
};
static char* vala_source_file_get_subdir (ValaSourceFile* self);
static char* vala_source_file_get_destination_directory (ValaSourceFile* self);
static char* vala_source_file_get_basename (ValaSourceFile* self);
static void vala_source_file_read_source_file (ValaSourceFile* self);
static gpointer vala_source_file_parent_class = NULL;
static void vala_source_file_finalize (GObject * obj);
static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func);
static int _vala_strcmp0 (const char * str1, const char * str2);



/**
 * Creates a new source file.
 *
 * @param filename source file name
 * @param pkg      true if this is a VAPI package file
 * @return         newly created source file
 */
ValaSourceFile* vala_source_file_new (ValaCodeContext* context, const char* filename, gboolean pkg) {
	ValaSourceFile * self;
	g_return_val_if_fail (VALA_IS_CODE_CONTEXT (context), NULL);
	g_return_val_if_fail (filename != NULL, NULL);
	self = g_object_newv (VALA_TYPE_SOURCE_FILE, 0, NULL);
	vala_source_file_set_filename (self, filename);
	vala_source_file_set_external_package (self, pkg);
	vala_source_file_set_context (self, context);
	return self;
}


/**
 * Adds a new using directive with the specified namespace.
 *
 * @param ns reference to namespace
 */
void vala_source_file_add_using_directive (ValaSourceFile* self, ValaNamespaceReference* ns) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_NAMESPACE_REFERENCE (ns));
	{
		GeeList* using_directive_collection;
		int using_directive_it;
		using_directive_collection = self->priv->using_directives;
		for (using_directive_it = 0; using_directive_it < gee_collection_get_size (GEE_COLLECTION (using_directive_collection)); using_directive_it = using_directive_it + 1) {
			ValaNamespaceReference* using_directive;
			using_directive = ((ValaNamespaceReference*) (gee_list_get (GEE_LIST (using_directive_collection), using_directive_it)));
			{
				if (_vala_strcmp0 (vala_namespace_reference_get_name (using_directive), vala_namespace_reference_get_name (ns)) == 0) {
					/* ignore duplicates*/
					(using_directive == NULL ? NULL : (using_directive = (g_object_unref (using_directive), NULL)));
					return;
				}
				(using_directive == NULL ? NULL : (using_directive = (g_object_unref (using_directive), NULL)));
			}
		}
	}
	gee_collection_add (GEE_COLLECTION (self->priv->using_directives), ns);
}


/**
 * Returns a copy of the list of using directives.
 *
 * @return using directive list
 */
GeeList* vala_source_file_get_using_directives (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_NAMESPACE_REFERENCE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->using_directives));
}


/**
 * Adds the specified code node to this source file.
 *
 * @param node a code node
 */
void vala_source_file_add_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_NODE (node));
	gee_collection_add (GEE_COLLECTION (self->priv->nodes), node);
}


void vala_source_file_remove_node (ValaSourceFile* self, ValaCodeNode* node) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_NODE (node));
	gee_collection_remove (GEE_COLLECTION (self->priv->nodes), node);
}


/**
 * Returns a copy of the list of code nodes.
 *
 * @return code node list
 */
GeeList* vala_source_file_get_nodes (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_CODE_NODE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->nodes));
}


void vala_source_file_accept (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_source_file (visitor, self);
}


void vala_source_file_accept_children (ValaSourceFile* self, ValaCodeVisitor* visitor) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* ns_ref_collection;
		int ns_ref_it;
		ns_ref_collection = self->priv->using_directives;
		for (ns_ref_it = 0; ns_ref_it < gee_collection_get_size (GEE_COLLECTION (ns_ref_collection)); ns_ref_it = ns_ref_it + 1) {
			ValaNamespaceReference* ns_ref;
			ns_ref = ((ValaNamespaceReference*) (gee_list_get (GEE_LIST (ns_ref_collection), ns_ref_it)));
			{
				vala_code_node_accept (VALA_CODE_NODE (ns_ref), visitor);
				(ns_ref == NULL ? NULL : (ns_ref = (g_object_unref (ns_ref), NULL)));
			}
		}
	}
	{
		GeeList* node_collection;
		int node_it;
		node_collection = self->priv->nodes;
		for (node_it = 0; node_it < gee_collection_get_size (GEE_COLLECTION (node_collection)); node_it = node_it + 1) {
			ValaCodeNode* node;
			node = ((ValaCodeNode*) (gee_list_get (GEE_LIST (node_collection), node_it)));
			{
				vala_code_node_accept (node, visitor);
				(node == NULL ? NULL : (node = (g_object_unref (node), NULL)));
			}
		}
	}
}


static char* vala_source_file_get_subdir (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (vala_code_context_get_basedir (self->priv->_context) == NULL) {
		return g_strdup ("");
	}
	/* filename and basedir are already canonicalized*/
	if (g_str_has_prefix (self->priv->_filename, vala_code_context_get_basedir (self->priv->_context))) {
		char* basename;
		char* _tmp1;
		char* subdir;
		char* _tmp4;
		basename = g_path_get_basename (self->priv->_filename);
		_tmp1 = NULL;
		subdir = (_tmp1 = g_utf8_offset_to_pointer (self->priv->_filename, g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1)), g_strndup (_tmp1, g_utf8_offset_to_pointer (_tmp1, g_utf8_strlen (self->priv->_filename, -1) - g_utf8_strlen (vala_code_context_get_basedir (self->priv->_context), -1) - g_utf8_strlen (basename, -1)) - _tmp1));
		while (g_utf8_get_char (g_utf8_offset_to_pointer (subdir, 0)) == '/') {
			char* _tmp3;
			const char* _tmp2;
			_tmp3 = NULL;
			_tmp2 = NULL;
			subdir = (_tmp3 = (_tmp2 = g_utf8_offset_to_pointer (subdir, ((glong) (1))), (_tmp2 == NULL ? NULL : g_strdup (_tmp2))), (subdir = (g_free (subdir), NULL)), _tmp3);
		}
		_tmp4 = NULL;
		return (_tmp4 = subdir, (basename = (g_free (basename), NULL)), _tmp4);
	}
	return g_strdup ("");
}


static char* vala_source_file_get_destination_directory (ValaSourceFile* self) {
	char* _tmp1;
	char* _tmp2;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (vala_code_context_get_directory (self->priv->_context) == NULL) {
		return vala_source_file_get_subdir (self);
	}
	_tmp1 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strdup_printf ("%s/%s", vala_code_context_get_directory (self->priv->_context), (_tmp1 = vala_source_file_get_subdir (self))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2);
}


static char* vala_source_file_get_basename (ValaSourceFile* self) {
	glong dot;
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	dot = g_utf8_pointer_to_offset (self->priv->_filename, g_utf8_strrchr (self->priv->_filename, ((glong) (-1)), ((gunichar) ('.'))));
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_path_get_basename ((_tmp1 = (_tmp0 = g_utf8_offset_to_pointer (self->priv->_filename, ((glong) (0))), g_strndup (_tmp0, g_utf8_offset_to_pointer (_tmp0, dot) - _tmp0)))), (_tmp1 = (g_free (_tmp1), NULL)), _tmp2);
}


char* vala_source_file_get_relative_filename (ValaSourceFile* self) {
	char* _tmp1;
	char* _tmp0;
	char* _tmp2;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	_tmp1 = NULL;
	_tmp0 = NULL;
	_tmp2 = NULL;
	return (_tmp2 = g_strconcat ((_tmp0 = vala_source_file_get_subdir (self)), (_tmp1 = g_path_get_basename (self->priv->_filename)), NULL), (_tmp1 = (g_free (_tmp1), NULL)), (_tmp0 = (g_free (_tmp0), NULL)), _tmp2);
}


/**
 * Returns the filename to use when generating C header files.
 *
 * @return generated C header filename
 */
char* vala_source_file_get_cheader_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cheader_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cheader_filename = (_tmp2 = g_strdup_printf ("%s%s.h", (_tmp0 = vala_source_file_get_destination_directory (self)), (_tmp1 = vala_source_file_get_basename (self))), (self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL)), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cheader_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Returns the filename to use when generating C source files.
 *
 * @return generated C source filename
 */
char* vala_source_file_get_csource_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->csource_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->csource_filename = (_tmp2 = g_strdup_printf ("%s%s.c", (_tmp0 = vala_source_file_get_destination_directory (self)), (_tmp1 = vala_source_file_get_basename (self))), (self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL)), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->csource_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Returns the filename to use when including the generated C header
 * file.
 *
 * @return C header filename to include
 */
char* vala_source_file_get_cinclude_filename (ValaSourceFile* self) {
	const char* _tmp3;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->cinclude_filename == NULL) {
		char* _tmp2;
		char* _tmp1;
		char* _tmp0;
		_tmp2 = NULL;
		_tmp1 = NULL;
		_tmp0 = NULL;
		self->priv->cinclude_filename = (_tmp2 = g_strdup_printf ("%s%s.h", (_tmp0 = vala_source_file_get_subdir (self)), (_tmp1 = vala_source_file_get_basename (self))), (self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL)), _tmp2);
		_tmp1 = (g_free (_tmp1), NULL);
		_tmp0 = (g_free (_tmp0), NULL);
	}
	_tmp3 = NULL;
	return (_tmp3 = self->priv->cinclude_filename, (_tmp3 == NULL ? NULL : g_strdup (_tmp3)));
}


/**
 * Adds the specified symbol to the list of symbols code in this source
 * file depends on.
 *
 * @param sym      a symbol
 * @param dep_type type of dependency
 */
void vala_source_file_add_symbol_dependency (ValaSourceFile* self, ValaSymbol* sym, ValaSourceFileDependencyType dep_type) {
	ValaSymbol* s;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (sym == NULL || VALA_IS_SYMBOL (sym));
	if (self->priv->_external_package) {
		return;
	}
	s = NULL;
	if (VALA_IS_TYPESYMBOL (sym) || VALA_IS_METHOD (sym) || VALA_IS_FIELD (sym) || VALA_IS_PROPERTY (sym) || VALA_IS_CONSTANT (sym)) {
		ValaSymbol* _tmp1;
		ValaSymbol* _tmp0;
		_tmp1 = NULL;
		_tmp0 = NULL;
		s = (_tmp1 = (_tmp0 = sym, (_tmp0 == NULL ? NULL : g_object_ref (_tmp0))), (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp1);
	} else {
		if (VALA_IS_FORMAL_PARAMETER (sym)) {
			ValaFormalParameter* _tmp2;
			ValaFormalParameter* fp;
			ValaSymbol* _tmp4;
			ValaSymbol* _tmp3;
			_tmp2 = NULL;
			fp = (_tmp2 = VALA_FORMAL_PARAMETER (sym), (_tmp2 == NULL ? NULL : g_object_ref (_tmp2)));
			_tmp4 = NULL;
			_tmp3 = NULL;
			s = (_tmp4 = (_tmp3 = VALA_SYMBOL (vala_data_type_get_data_type (vala_formal_parameter_get_parameter_type (fp))), (_tmp3 == NULL ? NULL : g_object_ref (_tmp3))), (s == NULL ? NULL : (s = (g_object_unref (s), NULL))), _tmp4);
			if (s == NULL) {
				/* generic type parameter */
				(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
				(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
				return;
			}
			(fp == NULL ? NULL : (fp = (g_object_unref (fp), NULL)));
		} else {
			(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
			return;
		}
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE) {
		gee_collection_add (GEE_COLLECTION (self->priv->source_symbol_dependencies), s);
		if (vala_symbol_get_external_package (s)) {
			{
				GeeList* fn_collection;
				int fn_it;
				fn_collection = vala_symbol_get_cheader_filenames (s);
				for (fn_it = 0; fn_it < gee_collection_get_size (GEE_COLLECTION (fn_collection)); fn_it = fn_it + 1) {
					char* fn;
					fn = ((char*) (gee_list_get (GEE_LIST (fn_collection), fn_it)));
					{
						gee_collection_add (GEE_COLLECTION (self->priv->source_external_includes), fn);
						fn = (g_free (fn), NULL);
					}
				}
				(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
			}
		} else {
			{
				GeeList* fn_collection;
				int fn_it;
				fn_collection = vala_symbol_get_cheader_filenames (s);
				for (fn_it = 0; fn_it < gee_collection_get_size (GEE_COLLECTION (fn_collection)); fn_it = fn_it + 1) {
					char* fn;
					fn = ((char*) (gee_list_get (GEE_LIST (fn_collection), fn_it)));
					{
						gee_collection_add (GEE_COLLECTION (self->priv->source_internal_includes), fn);
						fn = (g_free (fn), NULL);
					}
				}
				(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
			}
		}
		(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
		return;
	}
	if (vala_symbol_get_external_package (s)) {
		/* external package */
		{
			GeeList* fn_collection;
			int fn_it;
			fn_collection = vala_symbol_get_cheader_filenames (s);
			for (fn_it = 0; fn_it < gee_collection_get_size (GEE_COLLECTION (fn_collection)); fn_it = fn_it + 1) {
				char* fn;
				fn = ((char*) (gee_list_get (GEE_LIST (fn_collection), fn_it)));
				{
					gee_collection_add (GEE_COLLECTION (self->priv->header_external_includes), fn);
					fn = (g_free (fn), NULL);
				}
			}
			(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
		}
		(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
		return;
	}
	if (dep_type == VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL || (VALA_IS_TYPESYMBOL (s) && !vala_typesymbol_is_reference_type ((VALA_TYPESYMBOL (s))))) {
		{
			GeeList* fn_collection;
			int fn_it;
			fn_collection = vala_symbol_get_cheader_filenames (s);
			for (fn_it = 0; fn_it < gee_collection_get_size (GEE_COLLECTION (fn_collection)); fn_it = fn_it + 1) {
				char* fn;
				fn = ((char*) (gee_list_get (GEE_LIST (fn_collection), fn_it)));
				{
					gee_collection_add (GEE_COLLECTION (self->priv->header_internal_includes), fn);
					fn = (g_free (fn), NULL);
				}
			}
			(fn_collection == NULL ? NULL : (fn_collection = (g_object_unref (fn_collection), NULL)));
		}
		gee_collection_add (GEE_COLLECTION (self->priv->header_internal_full_dependencies), vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (s))));
	}
	gee_collection_add (GEE_COLLECTION (self->priv->header_internal_dependencies), vala_source_reference_get_file (vala_code_node_get_source_reference (VALA_CODE_NODE (s))));
	(s == NULL ? NULL : (s = (g_object_unref (s), NULL)));
}


/**
 * Adds the symbols that define the specified type to the list of
 * symbols code in this source file depends on.
 *
 * @param type     a data type
 * @param dep_type type of dependency
 */
void vala_source_file_add_type_dependency (ValaSourceFile* self, ValaDataType* type, ValaSourceFileDependencyType dep_type) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (VALA_IS_DATA_TYPE (type));
	{
		GeeList* type_symbol_collection;
		int type_symbol_it;
		type_symbol_collection = vala_data_type_get_symbols (type);
		for (type_symbol_it = 0; type_symbol_it < gee_collection_get_size (GEE_COLLECTION (type_symbol_collection)); type_symbol_it = type_symbol_it + 1) {
			ValaSymbol* type_symbol;
			type_symbol = ((ValaSymbol*) (gee_list_get (GEE_LIST (type_symbol_collection), type_symbol_it)));
			{
				vala_source_file_add_symbol_dependency (self, type_symbol, dep_type);
				(type_symbol == NULL ? NULL : (type_symbol = (g_object_unref (type_symbol), NULL)));
			}
		}
		(type_symbol_collection == NULL ? NULL : (type_symbol_collection = (g_object_unref (type_symbol_collection), NULL)));
	}
}


/**
 * Returns the list of external includes the generated C header file
 * requires.
 *
 * @return external include list for C header file
 */
GeeList* vala_source_file_get_header_external_includes (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, self->priv->header_external_includes));
}


/**
 * Adds the specified filename to the list of package-internal includes
 * the generated C header file requires.
 *
 * @param include internal include for C header file
 */
void vala_source_file_add_header_internal_include (ValaSourceFile* self, const char* include) {
	char* _tmp0;
	gboolean _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	g_return_if_fail (include != NULL);
	/* skip includes to self */
	_tmp0 = NULL;
	if ((_tmp1 = _vala_strcmp0 (include, (_tmp0 = vala_source_file_get_cinclude_filename (self))) != 0, (_tmp0 = (g_free (_tmp0), NULL)), _tmp1)) {
		gee_collection_add (GEE_COLLECTION (self->priv->header_internal_includes), include);
	}
}


/**
 * Returns the list of package-internal includes the generated C header
 * file requires.
 *
 * @return internal include list for C header file
 */
GeeList* vala_source_file_get_header_internal_includes (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, self->priv->header_internal_includes));
}


/**
 * Returns the list of external includes the generated C source file
 * requires.
 *
 * @return include list for C source file
 */
GeeList* vala_source_file_get_source_external_includes (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, self->priv->source_external_includes));
}


/**
 * Returns the list of package-internal includes the generated C source
 * file requires.
 *
 * @return include list for C source file
 */
GeeList* vala_source_file_get_source_internal_includes (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, self->priv->source_internal_includes));
}


/**
 * Returns the list of source files the generated C header file requires
 * definitely.
 *
 * @return definite source file dependencies
 */
GeeList* vala_source_file_get_header_internal_full_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, self->priv->header_internal_full_dependencies));
}


/**
 * Returns the list of source files the generated C header file loosely
 * depends on.
 *
 * @return loose source file dependencies
 */
GeeList* vala_source_file_get_header_internal_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_LIST (gee_read_only_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, self->priv->header_internal_dependencies));
}


GeeSet* vala_source_file_get_source_symbol_dependencies (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return GEE_SET (gee_read_only_set_new (VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, self->priv->source_symbol_dependencies));
}


/**
 * Returns the requested line from this file, loading it if needed.
 *
 * @param lineno 1-based line number
 * @return       the specified source line
 */
char* vala_source_file_get_source_line (ValaSourceFile* self, gint lineno) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	if (self->priv->source_array == NULL) {
		vala_source_file_read_source_file (self);
	}
	if (lineno < 1 || lineno > gee_collection_get_size (GEE_COLLECTION (self->priv->source_array))) {
		return NULL;
	}
	return ((char*) (gee_list_get (GEE_LIST (self->priv->source_array), lineno - 1)));
}


/**
 * Parses the input file into ::source_array.
 */
static void vala_source_file_read_source_file (ValaSourceFile* self) {
	GError * inner_error;
	char* cont;
	GeeArrayList* _tmp0;
	char** _tmp4;
	gint lines_length1;
	char** lines;
	guint idx;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	inner_error = NULL;
	cont = NULL;
	_tmp0 = NULL;
	self->priv->source_array = (_tmp0 = gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal), (self->priv->source_array == NULL ? NULL : (self->priv->source_array = (g_object_unref (self->priv->source_array), NULL))), _tmp0);
	{
		char* _tmp3;
		gboolean _tmp2;
		char* _tmp1;
		_tmp3 = NULL;
		_tmp1 = NULL;
		_tmp2 = g_file_get_contents (self->priv->_filename, &_tmp1, NULL, &inner_error);
		cont = (_tmp3 = _tmp1, (cont = (g_free (cont), NULL)), _tmp3);
		_tmp2;
		if (inner_error != NULL) {
			if (inner_error->domain == G_FILE_ERROR) {
				goto __catch6_g_file_error;
			}
			g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
			g_clear_error (&inner_error);
		}
	}
	goto __finally6;
	__catch6_g_file_error:
	{
		GError * fe;
		fe = inner_error;
		inner_error = NULL;
		{
			(fe == NULL ? NULL : (fe = (g_error_free (fe), NULL)));
			cont = (g_free (cont), NULL);
			return;
		}
	}
	__finally6:
	;
	_tmp4 = NULL;
	lines = (_tmp4 = g_strsplit (cont, "\n", 0), lines_length1 = -1, _tmp4);
	idx = 0U;
	for (idx = ((guint) (0)); lines[idx] != NULL; (idx = idx + 1)) {
		gee_collection_add (GEE_COLLECTION (self->priv->source_array), lines[idx]);
	}
	cont = (g_free (cont), NULL);
	lines = (_vala_array_free (lines, lines_length1, ((GDestroyNotify) (g_free))), NULL);
}


gchar* vala_source_file_get_mapped_contents (ValaSourceFile* self) {
	GError * inner_error;
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	inner_error = NULL;
	if (self->priv->mapped_file == NULL) {
		{
			GMappedFile* _tmp0;
			_tmp0 = NULL;
			self->priv->mapped_file = (_tmp0 = g_mapped_file_new (self->priv->_filename, FALSE, &inner_error), (self->priv->mapped_file == NULL ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL))), _tmp0);
			if (inner_error != NULL) {
				if (inner_error->domain == G_FILE_ERROR) {
					goto __catch7_g_file_error;
				}
				g_critical ("file %s: line %d: uncaught error: %s", __FILE__, __LINE__, inner_error->message);
				g_clear_error (&inner_error);
			}
		}
		goto __finally7;
		__catch7_g_file_error:
		{
			GError * e;
			e = inner_error;
			inner_error = NULL;
			{
				char* _tmp1;
				gchar* _tmp2;
				_tmp1 = NULL;
				vala_report_error (NULL, (_tmp1 = g_strdup_printf ("Unable to map file `%s': %s", self->priv->_filename, e->message)));
				_tmp1 = (g_free (_tmp1), NULL);
				return (_tmp2 = NULL, (e == NULL ? NULL : (e = (g_error_free (e), NULL))), _tmp2);
			}
		}
		__finally7:
		;
	}
	return g_mapped_file_get_contents (self->priv->mapped_file);
}


gsize vala_source_file_get_mapped_length (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), 0UL);
	return ((gsize) (g_mapped_file_get_length (self->priv->mapped_file)));
}


const char* vala_source_file_get_filename (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_filename;
}


void vala_source_file_set_filename (ValaSourceFile* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_filename = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_filename = (g_free (self->priv->_filename), NULL)), _tmp2);
	g_object_notify (((GObject *) (self)), "filename");
}


const char* vala_source_file_get_comment (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_comment;
}


void vala_source_file_set_comment (ValaSourceFile* self, const char* value) {
	char* _tmp2;
	const char* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_comment = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_strdup (_tmp1))), (self->priv->_comment = (g_free (self->priv->_comment), NULL)), _tmp2);
	g_object_notify (((GObject *) (self)), "comment");
}


gboolean vala_source_file_get_external_package (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_external_package;
}


void vala_source_file_set_external_package (ValaSourceFile* self, gboolean value) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_external_package = value;
	g_object_notify (((GObject *) (self)), "external-package");
}


ValaSourceFileCycle* vala_source_file_get_cycle (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_cycle;
}


void vala_source_file_set_cycle (ValaSourceFile* self, ValaSourceFileCycle* value) {
	ValaSourceFileCycle* _tmp2;
	ValaSourceFileCycle* _tmp1;
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	_tmp2 = NULL;
	_tmp1 = NULL;
	self->priv->_cycle = (_tmp2 = (_tmp1 = value, (_tmp1 == NULL ? NULL : g_object_ref (_tmp1))), (self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL))), _tmp2);
	g_object_notify (((GObject *) (self)), "cycle");
}


gboolean vala_source_file_get_is_cycle_head (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), FALSE);
	return self->priv->_is_cycle_head;
}


void vala_source_file_set_is_cycle_head (ValaSourceFile* self, gboolean value) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_is_cycle_head = value;
	g_object_notify (((GObject *) (self)), "is-cycle-head");
}


gint vala_source_file_get_mark (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), 0);
	return self->priv->_mark;
}


void vala_source_file_set_mark (ValaSourceFile* self, gint value) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_mark = value;
	g_object_notify (((GObject *) (self)), "mark");
}


ValaCodeContext* vala_source_file_get_context (ValaSourceFile* self) {
	g_return_val_if_fail (VALA_IS_SOURCE_FILE (self), NULL);
	return self->priv->_context;
}


void vala_source_file_set_context (ValaSourceFile* self, ValaCodeContext* value) {
	g_return_if_fail (VALA_IS_SOURCE_FILE (self));
	self->priv->_context = value;
	g_object_notify (((GObject *) (self)), "context");
}


static void vala_source_file_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec) {
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		g_value_set_string (value, vala_source_file_get_filename (self));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		g_value_set_string (value, vala_source_file_get_comment (self));
		break;
		case VALA_SOURCE_FILE_EXTERNAL_PACKAGE:
		g_value_set_boolean (value, vala_source_file_get_external_package (self));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		g_value_set_object (value, vala_source_file_get_cycle (self));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		g_value_set_boolean (value, vala_source_file_get_is_cycle_head (self));
		break;
		case VALA_SOURCE_FILE_MARK:
		g_value_set_int (value, vala_source_file_get_mark (self));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		g_value_set_object (value, vala_source_file_get_context (self));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_source_file_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec) {
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (object);
	switch (property_id) {
		case VALA_SOURCE_FILE_FILENAME:
		vala_source_file_set_filename (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_COMMENT:
		vala_source_file_set_comment (self, g_value_get_string (value));
		break;
		case VALA_SOURCE_FILE_EXTERNAL_PACKAGE:
		vala_source_file_set_external_package (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_CYCLE:
		vala_source_file_set_cycle (self, g_value_get_object (value));
		break;
		case VALA_SOURCE_FILE_IS_CYCLE_HEAD:
		vala_source_file_set_is_cycle_head (self, g_value_get_boolean (value));
		break;
		case VALA_SOURCE_FILE_MARK:
		vala_source_file_set_mark (self, g_value_get_int (value));
		break;
		case VALA_SOURCE_FILE_CONTEXT:
		vala_source_file_set_context (self, g_value_get_object (value));
		break;
		default:
		G_OBJECT_WARN_INVALID_PROPERTY_ID (object, property_id, pspec);
		break;
	}
}


static void vala_source_file_class_init (ValaSourceFileClass * klass) {
	vala_source_file_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSourceFilePrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_source_file_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_source_file_set_property;
	G_OBJECT_CLASS (klass)->finalize = vala_source_file_finalize;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_FILENAME, g_param_spec_string ("filename", "filename", "filename", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_COMMENT, g_param_spec_string ("comment", "comment", "comment", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_EXTERNAL_PACKAGE, g_param_spec_boolean ("external-package", "external-package", "external-package", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CYCLE, g_param_spec_object ("cycle", "cycle", "cycle", VALA_TYPE_SOURCE_FILE_CYCLE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_IS_CYCLE_HEAD, g_param_spec_boolean ("is-cycle-head", "is-cycle-head", "is-cycle-head", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_MARK, g_param_spec_int ("mark", "mark", "mark", G_MININT, G_MAXINT, 0, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SOURCE_FILE_CONTEXT, g_param_spec_object ("context", "context", "context", VALA_TYPE_CODE_CONTEXT, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_source_file_instance_init (ValaSourceFile * self) {
	self->priv = VALA_SOURCE_FILE_GET_PRIVATE (self);
	self->priv->using_directives = GEE_LIST (gee_array_list_new (VALA_TYPE_NAMESPACE_REFERENCE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
	self->priv->nodes = GEE_LIST (gee_array_list_new (VALA_TYPE_CODE_NODE, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_equal));
	self->priv->cheader_filename = NULL;
	self->priv->csource_filename = NULL;
	self->priv->cinclude_filename = NULL;
	self->priv->header_external_includes = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal));
	self->priv->header_internal_includes = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal));
	self->priv->source_external_includes = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal));
	self->priv->source_internal_includes = GEE_LIST (gee_array_list_new (G_TYPE_STRING, ((GBoxedCopyFunc) (g_strdup)), g_free, g_direct_equal));
	self->priv->header_internal_full_dependencies = GEE_LIST (gee_array_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, g_direct_equal));
	self->priv->header_internal_dependencies = GEE_LIST (gee_array_list_new (VALA_TYPE_SOURCE_FILE, NULL, NULL, g_direct_equal));
	self->priv->source_symbol_dependencies = GEE_SET (gee_hash_set_new (VALA_TYPE_SYMBOL, ((GBoxedCopyFunc) (g_object_ref)), g_object_unref, g_direct_hash, g_direct_equal));
	self->priv->source_array = NULL;
	self->priv->mapped_file = NULL;
}


static void vala_source_file_finalize (GObject * obj) {
	ValaSourceFile * self;
	self = VALA_SOURCE_FILE (obj);
	self->priv->_filename = (g_free (self->priv->_filename), NULL);
	self->priv->_comment = (g_free (self->priv->_comment), NULL);
	(self->priv->_cycle == NULL ? NULL : (self->priv->_cycle = (g_object_unref (self->priv->_cycle), NULL)));
	(self->priv->using_directives == NULL ? NULL : (self->priv->using_directives = (g_object_unref (self->priv->using_directives), NULL)));
	(self->priv->nodes == NULL ? NULL : (self->priv->nodes = (g_object_unref (self->priv->nodes), NULL)));
	self->priv->cheader_filename = (g_free (self->priv->cheader_filename), NULL);
	self->priv->csource_filename = (g_free (self->priv->csource_filename), NULL);
	self->priv->cinclude_filename = (g_free (self->priv->cinclude_filename), NULL);
	(self->priv->header_external_includes == NULL ? NULL : (self->priv->header_external_includes = (g_object_unref (self->priv->header_external_includes), NULL)));
	(self->priv->header_internal_includes == NULL ? NULL : (self->priv->header_internal_includes = (g_object_unref (self->priv->header_internal_includes), NULL)));
	(self->priv->source_external_includes == NULL ? NULL : (self->priv->source_external_includes = (g_object_unref (self->priv->source_external_includes), NULL)));
	(self->priv->source_internal_includes == NULL ? NULL : (self->priv->source_internal_includes = (g_object_unref (self->priv->source_internal_includes), NULL)));
	(self->priv->header_internal_full_dependencies == NULL ? NULL : (self->priv->header_internal_full_dependencies = (g_object_unref (self->priv->header_internal_full_dependencies), NULL)));
	(self->priv->header_internal_dependencies == NULL ? NULL : (self->priv->header_internal_dependencies = (g_object_unref (self->priv->header_internal_dependencies), NULL)));
	(self->priv->source_symbol_dependencies == NULL ? NULL : (self->priv->source_symbol_dependencies = (g_object_unref (self->priv->source_symbol_dependencies), NULL)));
	(self->priv->source_array == NULL ? NULL : (self->priv->source_array = (g_object_unref (self->priv->source_array), NULL)));
	(self->priv->mapped_file == NULL ? NULL : (self->priv->mapped_file = (g_mapped_file_free (self->priv->mapped_file), NULL)));
	G_OBJECT_CLASS (vala_source_file_parent_class)->finalize (obj);
}


GType vala_source_file_get_type (void) {
	static GType vala_source_file_type_id = 0;
	if (vala_source_file_type_id == 0) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSourceFileClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_source_file_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSourceFile), 0, (GInstanceInitFunc) vala_source_file_instance_init };
		vala_source_file_type_id = g_type_register_static (G_TYPE_OBJECT, "ValaSourceFile", &g_define_type_info, 0);
	}
	return vala_source_file_type_id;
}



GType vala_source_file_dependency_type_get_type (void) {
	static GType vala_source_file_dependency_type_type_id = 0;
	if (G_UNLIKELY (vala_source_file_dependency_type_type_id == 0)) {
		static const GEnumValue values[] = {{VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_FULL", "header-full"}, {VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_HEADER_SHALLOW", "header-shallow"}, {VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE, "VALA_SOURCE_FILE_DEPENDENCY_TYPE_SOURCE", "source"}, {0, NULL, NULL}};
		vala_source_file_dependency_type_type_id = g_enum_register_static ("ValaSourceFileDependencyType", values);
	}
	return vala_source_file_dependency_type_type_id;
}


static void _vala_array_free (gpointer array, gint array_length, GDestroyNotify destroy_func) {
	if (array != NULL && destroy_func != NULL) {
		int i;
		if (array_length >= 0)
		for (i = 0; i < array_length; i = i + 1) {
			if (((gpointer*) (array))[i] != NULL)
			destroy_func (((gpointer*) (array))[i]);
		}
		else
		for (i = 0; ((gpointer*) (array))[i] != NULL; i = i + 1) {
			destroy_func (((gpointer*) (array))[i]);
		}
	}
	g_free (array);
}


static int _vala_strcmp0 (const char * str1, const char * str2) {
	if (str1 == NULL) {
		return -(str1 != str2);
	}
	if (str2 == NULL) {
		return (str1 != str2);
	}
	return strcmp (str1, str2);
}




