/* valasymbol.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valasymbol.h"
#include <gee/readonlycollection.h>
#include <vala/valascope.h>
#include <vala/valasymbol.h>

struct _ValaSymbolPrivate {
	char* _name;
	gboolean _active;
	ValaScope* _owner;
	ValaScope* _scope;
};
#define VALA_SYMBOL_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_SYMBOL, ValaSymbolPrivate))
enum  {
	VALA_SYMBOL_DUMMY_PROPERTY,
	VALA_SYMBOL_PARENT_SYMBOL,
	VALA_SYMBOL_OWNER,
	VALA_SYMBOL_NAME,
	VALA_SYMBOL_ACTIVE,
	VALA_SYMBOL_SCOPE
};
static char* vala_symbol_real_get_cprefix (ValaSymbol* self);
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix);
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self);
static GeeCollection* vala_symbol_real_get_cheader_filenames (ValaSymbol* self);
static GObject * vala_symbol_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties);
static gpointer vala_symbol_parent_class = NULL;
static void vala_symbol_dispose (GObject * obj);


/**
 * Returns the fully expanded name of this symbol for use in
 * human-readable messages.
 *
 * @return full name
 */
char* vala_symbol_get_full_name (ValaSymbol* self)
{
	char* __temp3;
	gboolean __temp4;
	char* __temp7;
	char* __temp8;
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	if (vala_symbol_get_parent_symbol (self) == NULL) {
		const char* __temp0;
		__temp0 = NULL;
		return (__temp0 = vala_symbol_get_name (self), (__temp0 == NULL ? NULL : g_strdup (__temp0)));
	}
	if (vala_symbol_get_name (self) == NULL) {
		return vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self));
	}
	__temp3 = NULL;
	if ((__temp4 = (__temp3 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self))) == NULL, (__temp3 = (g_free (__temp3), NULL)), __temp4)) {
		const char* __temp5;
		__temp5 = NULL;
		return (__temp5 = vala_symbol_get_name (self), (__temp5 == NULL ? NULL : g_strdup (__temp5)));
	}
	__temp7 = NULL;
	__temp8 = NULL;
	return (__temp8 = g_strdup_printf ("%s.%s", (__temp7 = vala_symbol_get_full_name (vala_symbol_get_parent_symbol (self))), vala_symbol_get_name (self)), (__temp7 = (g_free (__temp7), NULL)), __temp8);
}


/**
 * Returns the camel case string to be prepended to the name of members
 * of this symbol when used in C code.
 *
 * @return the camel case prefix to be used in C code
 */
static char* vala_symbol_real_get_cprefix (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	if (vala_symbol_get_name (self) == NULL) {
		return g_strdup ("");
	} else {
		const char* __temp11;
		__temp11 = NULL;
		return (__temp11 = vala_symbol_get_name (self), (__temp11 == NULL ? NULL : g_strdup (__temp11)));
	}
}


char* vala_symbol_get_cprefix (ValaSymbol* self)
{
	return VALA_SYMBOL_GET_CLASS (self)->get_cprefix (self);
}


/**
 * Returns the C name of this symbol in lower case. Words are
 * separated by underscores. The lower case C name of the parent symbol
 * is prefix of the result, if there is one.
 *
 * @param infix a string to be placed between namespace and data type
 *              name or null
 * @return      the lower case name to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cname (ValaSymbol* self, const char* infix)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return NULL;
}


char* vala_symbol_get_lower_case_cname (ValaSymbol* self, const char* infix)
{
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cname (self, infix);
}


/**
 * Returns the string to be prefixed to members of this symbol in
 * lower case when used in C code.
 *
 * @return      the lower case prefix to be used in C code
 */
static char* vala_symbol_real_get_lower_case_cprefix (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return g_strdup ("");
}


char* vala_symbol_get_lower_case_cprefix (ValaSymbol* self)
{
	return VALA_SYMBOL_GET_CLASS (self)->get_lower_case_cprefix (self);
}


/**
 * Returns a list of C header filenames users of this symbol must
 * include.
 *
 * @return list of C header filenames for this symbol
 */
static GeeCollection* vala_symbol_real_get_cheader_filenames (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_strdup, g_free, NULL));
}


GeeCollection* vala_symbol_get_cheader_filenames (ValaSymbol* self)
{
	return VALA_SYMBOL_GET_CLASS (self)->get_cheader_filenames (self);
}


/**
 * Converts a string from CamelCase to lower_case.
 *
 * @param camel_case a string in camel case
 * @return           the specified string converted to lower case
 */
char* vala_symbol_camel_case_to_lower_case (const char* camel_case)
{
	GString* result;
	const char* i;
	gboolean first;
	const char* __temp16;
	char* __temp17;
	g_return_val_if_fail (camel_case != NULL, NULL);
	result = g_string_new ("");
	i = camel_case;
	first = TRUE;
	while (g_utf8_strlen (i, -1) > 0) {
		gunichar c;
		c = g_utf8_get_char (i);
		if (g_unichar_isupper (c) && !first) {
			const char* t;
			gboolean prev_upper;
			gboolean next_upper;
			/* current character is upper case and
			 * we're not at the beginning */
			t = g_utf8_prev_char (i);
			prev_upper = g_unichar_isupper (g_utf8_get_char (t));
			t = g_utf8_next_char (i);
			next_upper = g_unichar_isupper (g_utf8_get_char (t));
			if (!prev_upper || (g_utf8_strlen (i, -1) >= 2 && !next_upper)) {
				gint len;
				/* previous character wasn't upper case or
				 * next character isn't upper case*/
				len = g_utf8_strlen (result->str, -1);
				if (len != 1 && g_utf8_get_char (g_utf8_offset_to_pointer (result->str, len - 2)) != '_') {
					/* we're not creating 1 character words */
					g_string_append_c (result, '_');
				}
			}
		}
		g_string_append_unichar (result, g_unichar_tolower (c));
		first = FALSE;
		i = g_utf8_next_char (i);
	}
	__temp16 = NULL;
	__temp17 = NULL;
	return (__temp17 = (__temp16 = result->str, (__temp16 == NULL ? NULL : g_strdup (__temp16))), (result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL))), __temp17);
	(result == NULL ? NULL : (result = (g_string_free (result, TRUE), NULL)));
}


ValaSymbol* vala_symbol_get_parent_symbol (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	if (vala_symbol_get_owner (self) == NULL) {
		return VALA_SYMBOL (NULL);
	} else {
		return vala_scope_get_owner (vala_symbol_get_owner (self));
	}
}


ValaScope* vala_symbol_get_owner (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return self->priv->_owner;
}


void vala_symbol_set_owner (ValaSymbol* self, ValaScope* value)
{
	g_return_if_fail (VALA_IS_SYMBOL (self));
	self->priv->_owner = value;
	vala_scope_set_parent_scope (self->priv->_scope, value);
}


char* vala_symbol_get_name (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return self->priv->_name;
}


void vala_symbol_set_name (ValaSymbol* self, const char* value)
{
	char* __temp23;
	const char* __temp22;
	g_return_if_fail (VALA_IS_SYMBOL (self));
	__temp23 = NULL;
	__temp22 = NULL;
	self->priv->_name = (__temp23 = (__temp22 = value, (__temp22 == NULL ? NULL : g_strdup (__temp22))), (self->priv->_name = (g_free (self->priv->_name), NULL)), __temp23);
}


gboolean vala_symbol_get_active (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), FALSE);
	return self->priv->_active;
}


void vala_symbol_set_active (ValaSymbol* self, gboolean value)
{
	g_return_if_fail (VALA_IS_SYMBOL (self));
	self->priv->_active = value;
}


ValaScope* vala_symbol_get_scope (ValaSymbol* self)
{
	g_return_val_if_fail (VALA_IS_SYMBOL (self), NULL);
	return self->priv->_scope;
}


static GObject * vala_symbol_constructor (GType type, guint n_construct_properties, GObjectConstructParam * construct_properties)
{
	GObject * obj;
	ValaSymbolClass * klass;
	GObjectClass * parent_class;
	ValaSymbol * self;
	klass = VALA_SYMBOL_CLASS (g_type_class_peek (VALA_TYPE_SYMBOL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	obj = parent_class->constructor (type, n_construct_properties, construct_properties);
	self = VALA_SYMBOL (obj);
	{
		ValaScope* __temp26;
		__temp26 = NULL;
		self->priv->_scope = (__temp26 = vala_scope_new (self), (self->priv->_scope == NULL ? NULL : (self->priv->_scope = (g_object_unref (self->priv->_scope), NULL))), __temp26);
		vala_symbol_set_active (self, TRUE);
	}
	return obj;
}


static void vala_symbol_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaSymbol * self;
	self = VALA_SYMBOL (object);
	switch (property_id) {
		case VALA_SYMBOL_PARENT_SYMBOL:
		g_value_set_object (value, vala_symbol_get_parent_symbol (self));
		break;
		case VALA_SYMBOL_OWNER:
		g_value_set_object (value, vala_symbol_get_owner (self));
		break;
		case VALA_SYMBOL_NAME:
		g_value_set_string (value, vala_symbol_get_name (self));
		break;
		case VALA_SYMBOL_ACTIVE:
		g_value_set_boolean (value, vala_symbol_get_active (self));
		break;
		case VALA_SYMBOL_SCOPE:
		g_value_set_object (value, vala_symbol_get_scope (self));
		break;
	}
}


static void vala_symbol_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaSymbol * self;
	self = VALA_SYMBOL (object);
	switch (property_id) {
		case VALA_SYMBOL_OWNER:
		vala_symbol_set_owner (self, g_value_get_object (value));
		break;
		case VALA_SYMBOL_NAME:
		vala_symbol_set_name (self, g_value_get_string (value));
		break;
		case VALA_SYMBOL_ACTIVE:
		vala_symbol_set_active (self, g_value_get_boolean (value));
		break;
	}
}


static void vala_symbol_class_init (ValaSymbolClass * klass)
{
	vala_symbol_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaSymbolPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_symbol_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_symbol_set_property;
	G_OBJECT_CLASS (klass)->constructor = vala_symbol_constructor;
	G_OBJECT_CLASS (klass)->dispose = vala_symbol_dispose;
	VALA_SYMBOL_CLASS (klass)->get_cprefix = vala_symbol_real_get_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_symbol_real_get_lower_case_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_symbol_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_cheader_filenames = vala_symbol_real_get_cheader_filenames;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SYMBOL_PARENT_SYMBOL, g_param_spec_object ("parent-symbol", "foo", "bar", VALA_TYPE_SYMBOL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SYMBOL_OWNER, g_param_spec_object ("owner", "foo", "bar", VALA_TYPE_SCOPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SYMBOL_NAME, g_param_spec_string ("name", "foo", "bar", NULL, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SYMBOL_ACTIVE, g_param_spec_boolean ("active", "foo", "bar", FALSE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_SYMBOL_SCOPE, g_param_spec_object ("scope", "foo", "bar", VALA_TYPE_SCOPE, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE));
}


static void vala_symbol_init (ValaSymbol * self)
{
	self->priv = VALA_SYMBOL_GET_PRIVATE (self);
}


static void vala_symbol_dispose (GObject * obj)
{
	ValaSymbol * self;
	ValaSymbolClass * klass;
	GObjectClass * parent_class;
	self = VALA_SYMBOL (obj);
	(self->priv->_name = (g_free (self->priv->_name), NULL));
	(self->priv->_scope == NULL ? NULL : (self->priv->_scope = (g_object_unref (self->priv->_scope), NULL)));
	klass = VALA_SYMBOL_CLASS (g_type_class_peek (VALA_TYPE_SYMBOL));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_symbol_get_type (void)
{
	static GType vala_symbol_type_id = 0;
	if (G_UNLIKELY (vala_symbol_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaSymbolClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_symbol_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaSymbol), 0, (GInstanceInitFunc) vala_symbol_init };
		vala_symbol_type_id = g_type_register_static (VALA_TYPE_CODE_NODE, "ValaSymbol", &g_define_type_info, G_TYPE_FLAG_ABSTRACT);
	}
	return vala_symbol_type_id;
}




