/* valastruct.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 */

#include "valastruct.h"
#include <gee/arraylist.h>
#include <gee/list.h>
#include <gee/readonlycollection.h>
#include <vala/valamethod.h>
#include <vala/valasourcereference.h>
#include <vala/valasymbol.h>
#include <vala/valatypeparameter.h>
#include <vala/valastruct.h>
#include <vala/valascope.h>
#include <vala/valaconstant.h>
#include <vala/valafield.h>
#include <vala/valaformalparameter.h>
#include <vala/valatypereference.h>
#include <vala/valacreationmethod.h>
#include <vala/valacodevisitor.h>
#include <vala/valaattribute.h>
#include <vala/valareport.h>

struct _ValaStructPrivate {
	GeeList* type_parameters;
	GeeList* constants;
	GeeList* fields;
	GeeList* methods;
	GeeList* base_types;
	char* cname;
	char* const_cname;
	char* dup_function;
	char* free_function;
	char* type_id;
	char* lower_case_cprefix;
	char* lower_case_csuffix;
	gboolean reference_type;
	gboolean integer_type;
	gboolean floating_type;
	gint rank;
	char* marshaller_type_name;
	char* get_value_function;
	char* set_value_function;
	char* default_value;
	ValaMethod* _default_construction_method;
};
#define VALA_STRUCT_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_STRUCT, ValaStructPrivate))
enum  {
	VALA_STRUCT_DUMMY_PROPERTY,
	VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD
};
static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor);
static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor);
static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type);
static void vala_struct_set_cname (ValaStruct* self, const char* cname);
static void vala_struct_set_const_cname (ValaStruct* self, const char* cname);
static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base);
static char* vala_struct_get_lower_case_csuffix (ValaStruct* self);
static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix);
static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix);
static gboolean vala_struct_real_is_reference_type (ValaDataType* base);
static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_ref_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a);
static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a);
static gboolean vala_struct_real_is_reference_counting (ValaDataType* base);
static char* vala_struct_real_get_dup_function (ValaDataType* base);
static char* vala_struct_real_get_free_function (ValaDataType* base);
static void vala_struct_set_free_function (ValaStruct* self, const char* name);
static char* vala_struct_real_get_type_id (ValaDataType* base);
static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base);
static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name);
static char* vala_struct_real_get_get_value_function (ValaDataType* base);
static char* vala_struct_real_get_set_value_function (ValaDataType* base);
static void vala_struct_set_get_value_function (ValaStruct* self, const char* function);
static void vala_struct_set_set_value_function (ValaStruct* self, const char* function);
static char* vala_struct_real_get_default_value (ValaDataType* base);
static void vala_struct_set_default_value (ValaStruct* self, const char* value);
static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name);
static gpointer vala_struct_parent_class = NULL;
static void vala_struct_dispose (GObject * obj);


/**
 * Creates a new struct.
 *
 * @param name             type name
 * @param source_reference reference to source code
 * @return                 newly created struct
 */
ValaStruct* vala_struct_new (const char* name, ValaSourceReference* source_reference)
{
	ValaStruct * self;
	g_return_val_if_fail (name != NULL, NULL);
	g_return_val_if_fail (source_reference == NULL || VALA_IS_SOURCE_REFERENCE (source_reference), NULL);
	self = g_object_newv (VALA_TYPE_STRUCT, 0, NULL);
	vala_symbol_set_name (VALA_SYMBOL (self), name);
	vala_code_node_set_source_reference (VALA_CODE_NODE (self), source_reference);
	return self;
}


/**
 * Appends the specified parameter to the list of type parameters.
 *
 * @param p a type parameter
 */
void vala_struct_add_type_parameter (ValaStruct* self, ValaTypeParameter* p)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_PARAMETER (p));
	gee_collection_add (GEE_COLLECTION (self->priv->type_parameters), p);
	p->type = VALA_DATA_TYPE (self);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) p)), VALA_SYMBOL (p));
}


/**
 * Adds the specified constant as a member to this struct.
 *
 * @param c a constant
 */
void vala_struct_add_constant (ValaStruct* self, ValaConstant* c)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_CONSTANT (c));
	gee_collection_add (GEE_COLLECTION (self->priv->constants), c);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) c)), VALA_SYMBOL (c));
}


/**
 * Adds the specified field as a member to this struct.
 *
 * @param f a field
 */
void vala_struct_add_field (ValaStruct* self, ValaField* f)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_FIELD (f));
	gee_collection_add (GEE_COLLECTION (self->priv->fields), f);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) f)), VALA_SYMBOL (f));
}


/**
 * Returns a copy of the list of fields.
 *
 * @return list of fields
 */
GeeCollection* vala_struct_get_fields (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->fields)));
}


/**
 * Adds the specified method as a member to this struct.
 *
 * @param m a method
 */
void vala_struct_add_method (ValaStruct* self, ValaMethod* m)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_METHOD (m));
	g_return_if_fail (m != NULL);
	if (vala_method_get_instance (m)) {
		ValaFormalParameter* __temp2;
		ValaTypeReference* __temp1;
		__temp2 = NULL;
		__temp1 = NULL;
		vala_method_set_this_parameter (m, (__temp2 = vala_formal_parameter_new ("this", (__temp1 = vala_type_reference_new ()), NULL)));
		(__temp2 == NULL ? NULL : (__temp2 = (g_object_unref (__temp2), NULL)));
		(__temp1 == NULL ? NULL : (__temp1 = (g_object_unref (__temp1), NULL)));
		vala_type_reference_set_data_type (vala_formal_parameter_get_type_reference (vala_method_get_this_parameter (m)), VALA_DATA_TYPE (self));
		vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) m)), vala_symbol_get_name (((ValaSymbol*) vala_method_get_this_parameter (m))), VALA_SYMBOL (vala_method_get_this_parameter (m)));
	}
	if (VALA_IS_CREATION_METHOD (m) && vala_symbol_get_name (((ValaSymbol*) m)) == NULL) {
		vala_struct_set_default_construction_method (self, m);
	}
	gee_collection_add (GEE_COLLECTION (self->priv->methods), m);
	vala_scope_add (vala_symbol_get_scope (((ValaSymbol*) self)), vala_symbol_get_name (((ValaSymbol*) m)), VALA_SYMBOL (m));
}


/**
 * Returns a copy of the list of methods.
 *
 * @return list of methods
 */
GeeCollection* vala_struct_get_methods (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->methods)));
}


static void vala_struct_real_accept (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	vala_code_visitor_visit_struct (visitor, self);
}


static void vala_struct_real_accept_children (ValaCodeNode* base, ValaCodeVisitor* visitor)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	g_return_if_fail (VALA_IS_CODE_VISITOR (visitor));
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (p), visitor);
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	{
		GeeList* f_collection;
		GeeIterator* f_it;
		f_collection = self->priv->fields;
		f_it = gee_iterable_iterator (GEE_ITERABLE (f_collection));
		while (gee_iterator_next (f_it)) {
			ValaField* f;
			f = gee_iterator_get (f_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (f), visitor);
				(f == NULL ? NULL : (f = (g_object_unref (f), NULL)));
			}
		}
		(f_it == NULL ? NULL : (f_it = (g_object_unref (f_it), NULL)));
	}
	{
		GeeList* c_collection;
		GeeIterator* c_it;
		c_collection = self->priv->constants;
		c_it = gee_iterable_iterator (GEE_ITERABLE (c_collection));
		while (gee_iterator_next (c_it)) {
			ValaConstant* c;
			c = gee_iterator_get (c_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (c), visitor);
				(c == NULL ? NULL : (c = (g_object_unref (c), NULL)));
			}
		}
		(c_it == NULL ? NULL : (c_it = (g_object_unref (c_it), NULL)));
	}
	{
		GeeList* m_collection;
		GeeIterator* m_it;
		m_collection = self->priv->methods;
		m_it = gee_iterable_iterator (GEE_ITERABLE (m_collection));
		while (gee_iterator_next (m_it)) {
			ValaMethod* m;
			m = gee_iterator_get (m_it);
			{
				vala_code_node_accept (VALA_CODE_NODE (m), visitor);
				(m == NULL ? NULL : (m = (g_object_unref (m), NULL)));
			}
		}
		(m_it == NULL ? NULL : (m_it = (g_object_unref (m_it), NULL)));
	}
}


static char* vala_struct_real_get_cname (ValaDataType* base, gboolean const_type)
{
	ValaStruct * self;
	const char* __temp8;
	self = ((ValaStruct*) base);
	if (const_type && self->priv->const_cname != NULL) {
		const char* __temp4;
		__temp4 = NULL;
		return (__temp4 = self->priv->const_cname, (__temp4 == NULL ? NULL : g_strdup (__temp4)));
	}
	if (self->priv->cname == NULL) {
		char* __temp7;
		char* __temp6;
		__temp7 = NULL;
		__temp6 = NULL;
		self->priv->cname = (__temp7 = g_strdup_printf ("%s%s", (__temp6 = vala_symbol_get_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp7);
		(__temp6 = (g_free (__temp6), NULL));
	}
	__temp8 = NULL;
	return (__temp8 = self->priv->cname, (__temp8 == NULL ? NULL : g_strdup (__temp8)));
}


static void vala_struct_set_cname (ValaStruct* self, const char* cname)
{
	char* __temp10;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	__temp10 = NULL;
	self->priv->cname = (__temp10 = g_strdup (cname), (self->priv->cname = (g_free (self->priv->cname), NULL)), __temp10);
}


static void vala_struct_set_const_cname (ValaStruct* self, const char* cname)
{
	char* __temp11;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (cname != NULL);
	__temp11 = NULL;
	self->priv->const_cname = (__temp11 = g_strdup (cname), (self->priv->const_cname = (g_free (self->priv->const_cname), NULL)), __temp11);
}


static char* vala_struct_real_get_lower_case_cprefix (ValaSymbol* base)
{
	ValaStruct * self;
	const char* __temp14;
	self = ((ValaStruct*) base);
	if (self->priv->lower_case_cprefix == NULL) {
		char* __temp13;
		char* __temp12;
		__temp13 = NULL;
		__temp12 = NULL;
		self->priv->lower_case_cprefix = (__temp13 = g_strdup_printf ("%s_", (__temp12 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), NULL))), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp13);
		(__temp12 = (g_free (__temp12), NULL));
	}
	__temp14 = NULL;
	return (__temp14 = self->priv->lower_case_cprefix, (__temp14 == NULL ? NULL : g_strdup (__temp14)));
}


static char* vala_struct_get_lower_case_csuffix (ValaStruct* self)
{
	const char* __temp17;
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	if (self->priv->lower_case_csuffix == NULL) {
		char* __temp16;
		__temp16 = NULL;
		self->priv->lower_case_csuffix = (__temp16 = vala_symbol_camel_case_to_lower_case (vala_symbol_get_name (((ValaSymbol*) self))), (self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL)), __temp16);
	}
	__temp17 = NULL;
	return (__temp17 = self->priv->lower_case_csuffix, (__temp17 == NULL ? NULL : g_strdup (__temp17)));
}


static char* vala_struct_real_get_lower_case_cname (ValaSymbol* base, const char* infix)
{
	ValaStruct * self;
	char* __temp20;
	char* __temp19;
	char* __temp21;
	self = ((ValaStruct*) base);
	if (infix == NULL) {
		infix = "";
	}
	__temp20 = NULL;
	__temp19 = NULL;
	__temp21 = NULL;
	return (__temp21 = g_strdup_printf ("%s%s%s", (__temp19 = vala_symbol_get_lower_case_cprefix (vala_symbol_get_parent_symbol (((ValaSymbol*) self)))), infix, (__temp20 = vala_struct_get_lower_case_csuffix (self))), (__temp20 = (g_free (__temp20), NULL)), (__temp19 = (g_free (__temp19), NULL)), __temp21);
}


static char* vala_struct_real_get_upper_case_cname (ValaDataType* base, const char* infix)
{
	ValaStruct * self;
	char* __temp23;
	char* __temp24;
	self = ((ValaStruct*) base);
	__temp23 = NULL;
	__temp24 = NULL;
	return (__temp24 = g_utf8_strup ((__temp23 = vala_symbol_get_lower_case_cname (VALA_SYMBOL (self), infix)), -1), (__temp23 = (g_free (__temp23), NULL)), __temp24);
}


static gboolean vala_struct_real_is_reference_type (ValaDataType* base)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	return self->priv->reference_type;
}


/**
 * Returns whether this is an integer type.
 *
 * @return true if this is an integer type, false otherwise
 */
gboolean vala_struct_is_integer_type (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->integer_type;
}


/**
 * Returns whether this is a floating point type.
 *
 * @return true if this is a floating point type, false otherwise
 */
gboolean vala_struct_is_floating_type (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), FALSE);
	return self->priv->floating_type;
}


/**
 * Returns the rank of this integer or floating point type.
 *
 * @return the rank if this is an integer or floating point type
 */
gint vala_struct_get_rank (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), 0);
	return self->priv->rank;
}


/**
 * Sets whether this data type has value or reference type semantics.
 *
 * @param ref_type true if this data type has reference type semantics
 */
void vala_struct_set_is_reference_type (ValaStruct* self, gboolean ref_type)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	self->priv->reference_type = ref_type;
}


static void vala_struct_process_ccode_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	if (vala_attribute_has_argument (a, "cname")) {
		char* __temp30;
		__temp30 = NULL;
		vala_struct_set_cname (self, (__temp30 = vala_attribute_get_string (a, "cname")));
		(__temp30 = (g_free (__temp30), NULL));
	}
	if (vala_attribute_has_argument (a, "const_cname")) {
		char* __temp31;
		__temp31 = NULL;
		vala_struct_set_const_cname (self, (__temp31 = vala_attribute_get_string (a, "const_cname")));
		(__temp31 = (g_free (__temp31), NULL));
	}
	if (vala_attribute_has_argument (a, "cprefix")) {
		char* __temp32;
		__temp32 = NULL;
		self->priv->lower_case_cprefix = (__temp32 = vala_attribute_get_string (a, "cprefix"), (self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL)), __temp32);
	}
	if (vala_attribute_has_argument (a, "cheader_filename")) {
		char* val;
		val = vala_attribute_get_string (a, "cheader_filename");
		{
			char** filename_collection;
			char** filename_it;
			filename_collection = g_strsplit (val, ",", 0);
			for (filename_it = filename_collection; *filename_it != NULL; filename_it = filename_it + 1) {
				const char* __temp33;
				char* filename;
				__temp33 = NULL;
				filename = (__temp33 = *filename_it, (__temp33 == NULL ? NULL : g_strdup (__temp33)));
				{
					vala_data_type_add_cheader_filename (VALA_DATA_TYPE (self), filename);
					(filename = (g_free (filename), NULL));
				}
			}
			(filename_collection = (g_strfreev (filename_collection), NULL));
		}
		(val = (g_free (val), NULL));
	}
	if (vala_attribute_has_argument (a, "type_id")) {
		char* __temp34;
		__temp34 = NULL;
		vala_struct_set_type_id (self, (__temp34 = vala_attribute_get_string (a, "type_id")));
		(__temp34 = (g_free (__temp34), NULL));
	}
	if (vala_attribute_has_argument (a, "marshaller_type_name")) {
		char* __temp35;
		__temp35 = NULL;
		vala_struct_set_marshaller_type_name (self, (__temp35 = vala_attribute_get_string (a, "marshaller_type_name")));
		(__temp35 = (g_free (__temp35), NULL));
	}
	if (vala_attribute_has_argument (a, "get_value_function")) {
		char* __temp36;
		__temp36 = NULL;
		vala_struct_set_get_value_function (self, (__temp36 = vala_attribute_get_string (a, "get_value_function")));
		(__temp36 = (g_free (__temp36), NULL));
	}
	if (vala_attribute_has_argument (a, "set_value_function")) {
		char* __temp37;
		__temp37 = NULL;
		vala_struct_set_set_value_function (self, (__temp37 = vala_attribute_get_string (a, "set_value_function")));
		(__temp37 = (g_free (__temp37), NULL));
	}
	if (vala_attribute_has_argument (a, "default_value")) {
		char* __temp38;
		__temp38 = NULL;
		vala_struct_set_default_value (self, (__temp38 = vala_attribute_get_string (a, "default_value")));
		(__temp38 = (g_free (__temp38), NULL));
	}
}


static void vala_struct_process_ref_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->reference_type = TRUE;
	if (vala_attribute_has_argument (a, "dup_function")) {
		char* __temp39;
		__temp39 = NULL;
		vala_struct_set_dup_function (self, (__temp39 = vala_attribute_get_string (a, "dup_function")));
		(__temp39 = (g_free (__temp39), NULL));
	}
	if (vala_attribute_has_argument (a, "free_function")) {
		char* __temp40;
		__temp40 = NULL;
		vala_struct_set_free_function (self, (__temp40 = vala_attribute_get_string (a, "free_function")));
		(__temp40 = (g_free (__temp40), NULL));
	}
}


static void vala_struct_process_integer_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->integer_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


static void vala_struct_process_floating_type_attribute (ValaStruct* self, ValaAttribute* a)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_ATTRIBUTE (a));
	self->priv->floating_type = TRUE;
	if (vala_attribute_has_argument (a, "rank")) {
		self->priv->rank = vala_attribute_get_integer (a, "rank");
	}
}


/**
 * Process all associated attributes.
 */
void vala_struct_process_attributes (ValaStruct* self)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	{
		GList* a_collection;
		GList* a_it;
		a_collection = ((ValaCodeNode*) self)->attributes;
		for (a_it = a_collection; a_it != NULL; a_it = a_it->next) {
			ValaAttribute* __temp41;
			ValaAttribute* a;
			__temp41 = NULL;
			a = (__temp41 = a_it->data, (__temp41 == NULL ? NULL : g_object_ref (__temp41)));
			{
				if (g_utf8_collate (vala_attribute_get_name (a), "CCode") == 0) {
					vala_struct_process_ccode_attribute (self, a);
				} else {
					if (g_utf8_collate (vala_attribute_get_name (a), "ReferenceType") == 0) {
						vala_struct_process_ref_type_attribute (self, a);
					} else {
						if (g_utf8_collate (vala_attribute_get_name (a), "IntegerType") == 0) {
							vala_struct_process_integer_type_attribute (self, a);
						} else {
							if (g_utf8_collate (vala_attribute_get_name (a), "FloatingType") == 0) {
								vala_struct_process_floating_type_attribute (self, a);
							}
						}
					}
				}
				(a == NULL ? NULL : (a = (g_object_unref (a), NULL)));
			}
		}
	}
}


static gboolean vala_struct_real_is_reference_counting (ValaDataType* base)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	return FALSE;
}


static char* vala_struct_real_get_dup_function (ValaDataType* base)
{
	ValaStruct * self;
	const char* __temp45;
	self = ((ValaStruct*) base);
	if (self->priv->dup_function == NULL) {
		char* __temp44;
		char* __temp43;
		__temp44 = NULL;
		__temp43 = NULL;
		vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp44 = g_strdup_printf ("The type `%s` doesn't contain a copy function", (__temp43 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
		(__temp44 = (g_free (__temp44), NULL));
		(__temp43 = (g_free (__temp43), NULL));
	}
	__temp45 = NULL;
	return (__temp45 = self->priv->dup_function, (__temp45 == NULL ? NULL : g_strdup (__temp45)));
}


void vala_struct_set_dup_function (ValaStruct* self, const char* name)
{
	char* __temp47;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	__temp47 = NULL;
	self->priv->dup_function = (__temp47 = g_strdup (name), (self->priv->dup_function = (g_free (self->priv->dup_function), NULL)), __temp47);
}


static char* vala_struct_real_get_free_function (ValaDataType* base)
{
	ValaStruct * self;
	const char* __temp52;
	self = ((ValaStruct*) base);
	if (self->priv->free_function == NULL) {
		if (vala_struct_get_default_construction_method (self) != NULL) {
			char* __temp49;
			char* __temp48;
			__temp49 = NULL;
			__temp48 = NULL;
			self->priv->free_function = (__temp49 = g_strconcat ((__temp48 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (self))), "free", NULL), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), __temp49);
			(__temp48 = (g_free (__temp48), NULL));
		} else {
			char* __temp51;
			char* __temp50;
			__temp51 = NULL;
			__temp50 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp51 = g_strdup_printf ("The type `%s` doesn't contain a free function", (__temp50 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
			(__temp51 = (g_free (__temp51), NULL));
			(__temp50 = (g_free (__temp50), NULL));
		}
	}
	__temp52 = NULL;
	return (__temp52 = self->priv->free_function, (__temp52 == NULL ? NULL : g_strdup (__temp52)));
}


static void vala_struct_set_free_function (ValaStruct* self, const char* name)
{
	char* __temp54;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	__temp54 = NULL;
	self->priv->free_function = (__temp54 = g_strdup (name), (self->priv->free_function = (g_free (self->priv->free_function), NULL)), __temp54);
}


static char* vala_struct_real_get_type_id (ValaDataType* base)
{
	ValaStruct * self;
	const char* __temp58;
	self = ((ValaStruct*) base);
	if (self->priv->type_id == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			char* __temp55;
			__temp55 = NULL;
			self->priv->type_id = (__temp55 = g_strdup ("G_TYPE_POINTER"), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp55);
		} else {
			char* __temp57;
			char* __temp56;
			__temp57 = NULL;
			__temp56 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp57 = g_strdup_printf ("The type `%s` doesn't declare a type id", (__temp56 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
			(__temp57 = (g_free (__temp57), NULL));
			(__temp56 = (g_free (__temp56), NULL));
		}
	}
	__temp58 = NULL;
	return (__temp58 = self->priv->type_id, (__temp58 == NULL ? NULL : g_strdup (__temp58)));
}


void vala_struct_set_type_id (ValaStruct* self, const char* name)
{
	char* __temp60;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	__temp60 = NULL;
	self->priv->type_id = (__temp60 = g_strdup (name), (self->priv->type_id = (g_free (self->priv->type_id), NULL)), __temp60);
}


static char* vala_struct_real_get_marshaller_type_name (ValaDataType* base)
{
	ValaStruct * self;
	const char* __temp64;
	self = ((ValaStruct*) base);
	if (self->priv->marshaller_type_name == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			char* __temp61;
			__temp61 = NULL;
			self->priv->marshaller_type_name = (__temp61 = g_strdup ("POINTER"), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), __temp61);
		} else {
			char* __temp63;
			char* __temp62;
			__temp63 = NULL;
			__temp62 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp63 = g_strdup_printf ("The type `%s` doesn't declare a marshaller type name", (__temp62 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
			(__temp63 = (g_free (__temp63), NULL));
			(__temp62 = (g_free (__temp62), NULL));
		}
	}
	__temp64 = NULL;
	return (__temp64 = self->priv->marshaller_type_name, (__temp64 == NULL ? NULL : g_strdup (__temp64)));
}


static void vala_struct_set_marshaller_type_name (ValaStruct* self, const char* name)
{
	char* __temp66;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (name != NULL);
	__temp66 = NULL;
	self->priv->marshaller_type_name = (__temp66 = g_strdup (name), (self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL)), __temp66);
}


static char* vala_struct_real_get_get_value_function (ValaDataType* base)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	if (self->priv->get_value_function == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			return g_strdup ("g_value_get_pointer");
		} else {
			char* __temp69;
			char* __temp68;
			__temp69 = NULL;
			__temp68 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp69 = g_strdup_printf ("The value type `%s` doesn't declare a GValue get function", (__temp68 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
			(__temp69 = (g_free (__temp69), NULL));
			(__temp68 = (g_free (__temp68), NULL));
			return NULL;
		}
	} else {
		const char* __temp71;
		__temp71 = NULL;
		return (__temp71 = self->priv->get_value_function, (__temp71 == NULL ? NULL : g_strdup (__temp71)));
	}
}


static char* vala_struct_real_get_set_value_function (ValaDataType* base)
{
	ValaStruct * self;
	self = ((ValaStruct*) base);
	if (self->priv->set_value_function == NULL) {
		if (vala_data_type_is_reference_type (VALA_DATA_TYPE (self))) {
			return g_strdup ("g_value_set_pointer");
		} else {
			char* __temp75;
			char* __temp74;
			__temp75 = NULL;
			__temp74 = NULL;
			vala_report_error (vala_code_node_get_source_reference (((ValaCodeNode*) self)), (__temp75 = g_strdup_printf ("The value type `%s` doesn't declare a GValue set function", (__temp74 = vala_symbol_get_full_name (VALA_SYMBOL (self))))));
			(__temp75 = (g_free (__temp75), NULL));
			(__temp74 = (g_free (__temp74), NULL));
			return NULL;
		}
	} else {
		const char* __temp77;
		__temp77 = NULL;
		return (__temp77 = self->priv->set_value_function, (__temp77 == NULL ? NULL : g_strdup (__temp77)));
	}
}


static void vala_struct_set_get_value_function (ValaStruct* self, const char* function)
{
	char* __temp79;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	__temp79 = NULL;
	self->priv->get_value_function = (__temp79 = g_strdup (function), (self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL)), __temp79);
}


static void vala_struct_set_set_value_function (ValaStruct* self, const char* function)
{
	char* __temp80;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (function != NULL);
	__temp80 = NULL;
	self->priv->set_value_function = (__temp80 = g_strdup (function), (self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL)), __temp80);
}


static char* vala_struct_real_get_default_value (ValaDataType* base)
{
	ValaStruct * self;
	const char* __temp81;
	self = ((ValaStruct*) base);
	__temp81 = NULL;
	return (__temp81 = self->priv->default_value, (__temp81 == NULL ? NULL : g_strdup (__temp81)));
}


static void vala_struct_set_default_value (ValaStruct* self, const char* value)
{
	char* __temp83;
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (value != NULL);
	__temp83 = NULL;
	self->priv->default_value = (__temp83 = g_strdup (value), (self->priv->default_value = (g_free (self->priv->default_value), NULL)), __temp83);
}


/**
 * Adds the specified struct to the list of base types of this struct.
 *
 * @param type a class or interface reference
 */
void vala_struct_add_base_type (ValaStruct* self, ValaTypeReference* type)
{
	g_return_if_fail (VALA_IS_STRUCT (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	gee_collection_add (GEE_COLLECTION (self->priv->base_types), type);
}


/**
 * Returns a copy of the base type list.
 *
 * @return list of base types
 */
GeeCollection* vala_struct_get_base_types (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return GEE_COLLECTION (gee_read_only_collection_new (g_object_ref, g_object_unref, GEE_COLLECTION (self->priv->base_types)));
}


static gint vala_struct_real_get_type_parameter_index (ValaDataType* base, const char* name)
{
	ValaStruct * self;
	gint i;
	self = ((ValaStruct*) base);
	g_return_val_if_fail (name != NULL, 0);
	i = 0;
	{
		GeeList* p_collection;
		GeeIterator* p_it;
		p_collection = self->priv->type_parameters;
		p_it = gee_iterable_iterator (GEE_ITERABLE (p_collection));
		while (gee_iterator_next (p_it)) {
			ValaTypeParameter* p;
			p = gee_iterator_get (p_it);
			{
				if (g_utf8_collate (vala_symbol_get_name (((ValaSymbol*) p)), name) == 0) {
					gint __temp85;
					return (__temp85 = (i), (p == NULL ? NULL : (p = (g_object_unref (p), NULL))), (p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL))), __temp85);
				}
				i++;
				(p == NULL ? NULL : (p = (g_object_unref (p), NULL)));
			}
		}
		(p_it == NULL ? NULL : (p_it = (g_object_unref (p_it), NULL)));
	}
	return -1;
}


ValaMethod* vala_struct_get_default_construction_method (ValaStruct* self)
{
	g_return_val_if_fail (VALA_IS_STRUCT (self), NULL);
	return self->priv->_default_construction_method;
}


void vala_struct_set_default_construction_method (ValaStruct* self, ValaMethod* value)
{
	ValaMethod* __temp89;
	ValaMethod* __temp88;
	g_return_if_fail (VALA_IS_STRUCT (self));
	__temp89 = NULL;
	__temp88 = NULL;
	self->priv->_default_construction_method = (__temp89 = (__temp88 = value, (__temp88 == NULL ? NULL : g_object_ref (__temp88))), (self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL))), __temp89);
}


static void vala_struct_get_property (GObject * object, guint property_id, GValue * value, GParamSpec * pspec)
{
	ValaStruct * self;
	self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		g_value_set_object (value, vala_struct_get_default_construction_method (self));
		break;
	}
}


static void vala_struct_set_property (GObject * object, guint property_id, const GValue * value, GParamSpec * pspec)
{
	ValaStruct * self;
	self = VALA_STRUCT (object);
	switch (property_id) {
		case VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD:
		vala_struct_set_default_construction_method (self, g_value_get_object (value));
		break;
	}
}


static void vala_struct_class_init (ValaStructClass * klass)
{
	vala_struct_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaStructPrivate));
	G_OBJECT_CLASS (klass)->get_property = vala_struct_get_property;
	G_OBJECT_CLASS (klass)->set_property = vala_struct_set_property;
	G_OBJECT_CLASS (klass)->dispose = vala_struct_dispose;
	VALA_CODE_NODE_CLASS (klass)->accept = vala_struct_real_accept;
	VALA_CODE_NODE_CLASS (klass)->accept_children = vala_struct_real_accept_children;
	VALA_DATA_TYPE_CLASS (klass)->get_cname = vala_struct_real_get_cname;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cprefix = vala_struct_real_get_lower_case_cprefix;
	VALA_SYMBOL_CLASS (klass)->get_lower_case_cname = vala_struct_real_get_lower_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->get_upper_case_cname = vala_struct_real_get_upper_case_cname;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_type = vala_struct_real_is_reference_type;
	VALA_DATA_TYPE_CLASS (klass)->is_reference_counting = vala_struct_real_is_reference_counting;
	VALA_DATA_TYPE_CLASS (klass)->get_dup_function = vala_struct_real_get_dup_function;
	VALA_DATA_TYPE_CLASS (klass)->get_free_function = vala_struct_real_get_free_function;
	VALA_DATA_TYPE_CLASS (klass)->get_type_id = vala_struct_real_get_type_id;
	VALA_DATA_TYPE_CLASS (klass)->get_marshaller_type_name = vala_struct_real_get_marshaller_type_name;
	VALA_DATA_TYPE_CLASS (klass)->get_get_value_function = vala_struct_real_get_get_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_set_value_function = vala_struct_real_get_set_value_function;
	VALA_DATA_TYPE_CLASS (klass)->get_default_value = vala_struct_real_get_default_value;
	VALA_DATA_TYPE_CLASS (klass)->get_type_parameter_index = vala_struct_real_get_type_parameter_index;
	g_object_class_install_property (G_OBJECT_CLASS (klass), VALA_STRUCT_DEFAULT_CONSTRUCTION_METHOD, g_param_spec_object ("default-construction-method", "foo", "bar", VALA_TYPE_METHOD, G_PARAM_STATIC_NAME | G_PARAM_STATIC_NICK | G_PARAM_STATIC_BLURB | G_PARAM_READABLE | G_PARAM_WRITABLE));
}


static void vala_struct_init (ValaStruct * self)
{
	self->priv = VALA_STRUCT_GET_PRIVATE (self);
	self->priv->type_parameters = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->constants = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->fields = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->methods = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->base_types = GEE_LIST (gee_array_list_new (g_object_ref, g_object_unref, g_direct_equal));
	self->priv->default_value = NULL;
}


static void vala_struct_dispose (GObject * obj)
{
	ValaStruct * self;
	ValaStructClass * klass;
	GObjectClass * parent_class;
	self = VALA_STRUCT (obj);
	(self->priv->type_parameters == NULL ? NULL : (self->priv->type_parameters = (g_object_unref (self->priv->type_parameters), NULL)));
	(self->priv->constants == NULL ? NULL : (self->priv->constants = (g_object_unref (self->priv->constants), NULL)));
	(self->priv->fields == NULL ? NULL : (self->priv->fields = (g_object_unref (self->priv->fields), NULL)));
	(self->priv->methods == NULL ? NULL : (self->priv->methods = (g_object_unref (self->priv->methods), NULL)));
	(self->priv->base_types == NULL ? NULL : (self->priv->base_types = (g_object_unref (self->priv->base_types), NULL)));
	(self->priv->cname = (g_free (self->priv->cname), NULL));
	(self->priv->const_cname = (g_free (self->priv->const_cname), NULL));
	(self->priv->dup_function = (g_free (self->priv->dup_function), NULL));
	(self->priv->free_function = (g_free (self->priv->free_function), NULL));
	(self->priv->type_id = (g_free (self->priv->type_id), NULL));
	(self->priv->lower_case_cprefix = (g_free (self->priv->lower_case_cprefix), NULL));
	(self->priv->lower_case_csuffix = (g_free (self->priv->lower_case_csuffix), NULL));
	(self->priv->marshaller_type_name = (g_free (self->priv->marshaller_type_name), NULL));
	(self->priv->get_value_function = (g_free (self->priv->get_value_function), NULL));
	(self->priv->set_value_function = (g_free (self->priv->set_value_function), NULL));
	(self->priv->default_value = (g_free (self->priv->default_value), NULL));
	(self->priv->_default_construction_method == NULL ? NULL : (self->priv->_default_construction_method = (g_object_unref (self->priv->_default_construction_method), NULL)));
	klass = VALA_STRUCT_CLASS (g_type_class_peek (VALA_TYPE_STRUCT));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_struct_get_type (void)
{
	static GType vala_struct_type_id = 0;
	if (G_UNLIKELY (vala_struct_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaStructClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_struct_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaStruct), 0, (GInstanceInitFunc) vala_struct_init };
		vala_struct_type_id = g_type_register_static (VALA_TYPE_DATA_TYPE, "ValaStruct", &g_define_type_info, 0);
	}
	return vala_struct_type_id;
}




