/* valainterfacewriter.vala
 *
 * Copyright (C) 2006-2007  Jürg Billeter, Raffaele Sandrini
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.

 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.

 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301  USA
 *
 * Author:
 * 	Jürg Billeter <j@bitron.ch>
 *	Raffaele Sandrini <rasa@gmx.ch>
 */

#include "valainterfacewriter.h"
#include <stdio.h>
#include <gee/list.h>
#include <vala/valainterfacewriter.h>
#include <vala/valasymbol.h>
#include <vala/valacodenode.h>
#include <vala/valasourcereference.h>
#include <vala/valasourcefile.h>
#include <vala/valadatatype.h>
#include <vala/valamemberaccessibility.h>
#include <vala/valatypeparameter.h>
#include <vala/valaformalparameter.h>
#include <vala/valaarray.h>
#include <vala/valapropertyaccessor.h>

struct _ValaInterfaceWriterPrivate {
	ValaCodeContext* context;
	FILE* stream;
	gint indent;
	gboolean bol;
	char* current_cheader_filename;
};
#define VALA_INTERFACE_WRITER_GET_PRIVATE(o) (G_TYPE_INSTANCE_GET_PRIVATE ((o), VALA_TYPE_INTERFACE_WRITER, ValaInterfaceWriterPrivate))
enum  {
	VALA_INTERFACE_WRITER_DUMMY_PROPERTY
};
static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns);
static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl);
static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st);
static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface);
static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en);
static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev);
static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c);
static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f);
static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GeeCollection* params);
static void vala_interface_writer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb);
static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m);
static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m);
static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop);
static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig);
static void vala_interface_writer_write_indent (ValaInterfaceWriter* self);
static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_type (ValaInterfaceWriter* self, ValaTypeReference* type);
static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s);
static void vala_interface_writer_write_newline (ValaInterfaceWriter* self);
static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self);
static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self);
static gpointer vala_interface_writer_parent_class = NULL;
static void vala_interface_writer_dispose (GObject * obj);


/**
 * Writes the public interface of the specified code context into the
 * specified file.
 *
 * @param context  a code context
 * @param filename a relative or absolute filename
 */
void vala_interface_writer_write_file (ValaInterfaceWriter* self, ValaCodeContext* context, const char* filename)
{
	ValaCodeContext* __temp0;
	FILE* __temp1;
	FILE* __temp2;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (VALA_IS_CODE_CONTEXT (context));
	g_return_if_fail (filename != NULL);
	__temp0 = NULL;
	self->priv->context = (__temp0 = g_object_ref (context), (self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL))), __temp0);
	__temp1 = NULL;
	self->priv->stream = (__temp1 = fopen (filename, "w"), (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp1);
	vala_code_context_accept (context, VALA_CODE_VISITOR (self));
	__temp2 = NULL;
	self->priv->stream = (__temp2 = NULL, (self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL))), __temp2);
}


static void vala_interface_writer_real_visit_namespace (ValaCodeVisitor* base, ValaNamespace* ns)
{
	ValaInterfaceWriter * self;
	char* __temp5;
	char* __temp4;
	char* __temp3;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_NAMESPACE (ns));
	if (vala_namespace_get_pkg (ns)) {
		return;
	}
	if (vala_symbol_get_name (((ValaSymbol*) ns)) == NULL) {
		vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
		return;
	}
	vala_interface_writer_write_indent (self);
	__temp5 = NULL;
	__temp4 = NULL;
	__temp3 = NULL;
	vala_interface_writer_write_string (self, (__temp5 = g_strdup_printf ("[CCode (cprefix = \"%s\", lower_case_cprefix = \"%s\")]", (__temp3 = vala_symbol_get_cprefix (VALA_SYMBOL (ns))), (__temp4 = vala_symbol_get_lower_case_cprefix (VALA_SYMBOL (ns))))));
	(__temp5 = (g_free (__temp5), NULL));
	(__temp4 = (g_free (__temp4), NULL));
	(__temp3 = (g_free (__temp3), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "namespace ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) ns)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (ns), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_class (ValaCodeVisitor* base, ValaClass* cl)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp9;
	GeeCollection* type_params;
	GeeCollection* base_types;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_CLASS (cl));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) cl)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) cl))))) {
		return;
	}
	if (((ValaDataType*) cl)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (cl));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* __temp7;
					const char* __temp6;
					__temp7 = NULL;
					__temp6 = NULL;
					cheaders = (__temp7 = (__temp6 = cheader, (__temp6 == NULL ? NULL : g_strdup (__temp6))), (cheaders = (g_free (cheaders), NULL)), __temp7);
					first = FALSE;
				} else {
					char* __temp8;
					__temp8 = NULL;
					cheaders = (__temp8 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp8);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	__temp9 = NULL;
	vala_interface_writer_write_string (self, (__temp9 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp9 = (g_free (__temp9), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_class_get_is_abstract (cl)) {
		vala_interface_writer_write_string (self, "abstract ");
	}
	vala_interface_writer_write_string (self, "class ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) cl)));
	type_params = vala_class_get_type_parameters (cl);
	if (gee_collection_get_size (type_params) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeCollection* type_param_collection;
			GeeIterator* type_param_it;
			type_param_collection = type_params;
			type_param_it = gee_iterable_iterator (GEE_ITERABLE (type_param_collection));
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = gee_iterator_get (type_param_it);
				{
					if (first) {
						first = FALSE;
					} else {
						vala_interface_writer_write_string (self, ",");
					}
					vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) type_param)));
					(type_param == NULL ? NULL : (type_param = (g_object_unref (type_param), NULL)));
				}
			}
			(type_param_it == NULL ? NULL : (type_param_it = (g_object_unref (type_param_it), NULL)));
		}
		vala_interface_writer_write_string (self, ">");
	}
	base_types = vala_class_get_base_types (cl);
	if (gee_collection_get_size (base_types) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeCollection* base_type_collection;
			GeeIterator* base_type_it;
			base_type_collection = base_types;
			base_type_it = gee_iterable_iterator (GEE_ITERABLE (base_type_collection));
			while (gee_iterator_next (base_type_it)) {
				ValaTypeReference* base_type;
				base_type = gee_iterator_get (base_type_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					vala_interface_writer_write_type (self, base_type);
					(base_type == NULL ? NULL : (base_type = (g_object_unref (base_type), NULL)));
				}
			}
			(base_type_it == NULL ? NULL : (base_type_it = (g_object_unref (base_type_it), NULL)));
		}
	}
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (cl), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
	(type_params == NULL ? NULL : (type_params = (g_object_unref (type_params), NULL)));
	(base_types == NULL ? NULL : (base_types = (g_object_unref (base_types), NULL)));
}


static void vala_interface_writer_real_visit_struct (ValaCodeVisitor* base, ValaStruct* st)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp13;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_STRUCT (st));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) st)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) st))))) {
		return;
	}
	if (((ValaDataType*) st)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	if (vala_data_type_is_reference_type (VALA_DATA_TYPE (st))) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[ReferenceType]");
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (st));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* __temp11;
					const char* __temp10;
					__temp11 = NULL;
					__temp10 = NULL;
					cheaders = (__temp11 = (__temp10 = cheader, (__temp10 == NULL ? NULL : g_strdup (__temp10))), (cheaders = (g_free (cheaders), NULL)), __temp11);
					first = FALSE;
				} else {
					char* __temp12;
					__temp12 = NULL;
					cheaders = (__temp12 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp12);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	__temp13 = NULL;
	vala_interface_writer_write_string (self, (__temp13 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp13 = (g_free (__temp13), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public struct ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) st)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (st), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
}


static void vala_interface_writer_real_visit_interface (ValaCodeVisitor* base, ValaInterface* iface)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp17;
	GeeCollection* type_params;
	GeeCollection* prerequisites;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_INTERFACE (iface));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) iface)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) iface))))) {
		return;
	}
	if (((ValaDataType*) iface)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (iface));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* __temp15;
					const char* __temp14;
					__temp15 = NULL;
					__temp14 = NULL;
					cheaders = (__temp15 = (__temp14 = cheader, (__temp14 == NULL ? NULL : g_strdup (__temp14))), (cheaders = (g_free (cheaders), NULL)), __temp15);
					first = FALSE;
				} else {
					char* __temp16;
					__temp16 = NULL;
					cheaders = (__temp16 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp16);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	__temp17 = NULL;
	vala_interface_writer_write_string (self, (__temp17 = g_strdup_printf ("[CCode (cheader_filename = \"%s\")]", cheaders)));
	(__temp17 = (g_free (__temp17), NULL));
	vala_interface_writer_write_newline (self);
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	vala_interface_writer_write_string (self, "interface ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) iface)));
	type_params = vala_interface_get_type_parameters (iface);
	if (gee_collection_get_size (type_params) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeCollection* type_param_collection;
			GeeIterator* type_param_it;
			type_param_collection = type_params;
			type_param_it = gee_iterable_iterator (GEE_ITERABLE (type_param_collection));
			while (gee_iterator_next (type_param_it)) {
				ValaTypeParameter* type_param;
				type_param = gee_iterator_get (type_param_it);
				{
					if (first) {
						first = FALSE;
					} else {
						vala_interface_writer_write_string (self, ",");
					}
					vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) type_param)));
					(type_param == NULL ? NULL : (type_param = (g_object_unref (type_param), NULL)));
				}
			}
			(type_param_it == NULL ? NULL : (type_param_it = (g_object_unref (type_param_it), NULL)));
		}
		vala_interface_writer_write_string (self, ">");
	}
	prerequisites = vala_interface_get_prerequisites (iface);
	if (gee_collection_get_size (prerequisites) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, " : ");
		first = TRUE;
		{
			GeeCollection* prerequisite_collection;
			GeeIterator* prerequisite_it;
			prerequisite_collection = prerequisites;
			prerequisite_it = gee_iterable_iterator (GEE_ITERABLE (prerequisite_collection));
			while (gee_iterator_next (prerequisite_it)) {
				ValaTypeReference* prerequisite;
				prerequisite = gee_iterator_get (prerequisite_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ", ");
					} else {
						first = FALSE;
					}
					vala_interface_writer_write_type (self, prerequisite);
					(prerequisite == NULL ? NULL : (prerequisite = (g_object_unref (prerequisite), NULL)));
				}
			}
			(prerequisite_it == NULL ? NULL : (prerequisite_it = (g_object_unref (prerequisite_it), NULL)));
		}
	}
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (iface), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
	(type_params == NULL ? NULL : (type_params = (g_object_unref (type_params), NULL)));
	(prerequisites == NULL ? NULL : (prerequisites = (g_object_unref (prerequisites), NULL)));
}


static void vala_interface_writer_real_visit_enum (ValaCodeVisitor* base, ValaEnum* en)
{
	ValaInterfaceWriter * self;
	gboolean first;
	char* cheaders;
	char* __temp22;
	char* __temp21;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_ENUM (en));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) en)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) en))))) {
		return;
	}
	if (((ValaDataType*) en)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	first = TRUE;
	cheaders = NULL;
	{
		GeeCollection* cheader_collection;
		GeeIterator* cheader_it;
		cheader_collection = vala_symbol_get_cheader_filenames (VALA_SYMBOL (en));
		cheader_it = gee_iterable_iterator (GEE_ITERABLE (cheader_collection));
		while (gee_iterator_next (cheader_it)) {
			char* cheader;
			cheader = gee_iterator_get (cheader_it);
			{
				if (first) {
					char* __temp19;
					const char* __temp18;
					__temp19 = NULL;
					__temp18 = NULL;
					cheaders = (__temp19 = (__temp18 = cheader, (__temp18 == NULL ? NULL : g_strdup (__temp18))), (cheaders = (g_free (cheaders), NULL)), __temp19);
					first = FALSE;
				} else {
					char* __temp20;
					__temp20 = NULL;
					cheaders = (__temp20 = g_strdup_printf ("%s,%s", cheaders, cheader), (cheaders = (g_free (cheaders), NULL)), __temp20);
				}
				(cheader = (g_free (cheader), NULL));
			}
		}
		(cheader_collection == NULL ? NULL : (cheader_collection = (g_object_unref (cheader_collection), NULL)));
		(cheader_it == NULL ? NULL : (cheader_it = (g_object_unref (cheader_it), NULL)));
	}
	__temp22 = NULL;
	__temp21 = NULL;
	vala_interface_writer_write_string (self, (__temp22 = g_strdup_printf ("[CCode (cprefix = \"%s\", cheader_filename = \"%s\")]", (__temp21 = vala_enum_get_cprefix (en)), cheaders)));
	(__temp22 = (g_free (__temp22), NULL));
	(__temp21 = (g_free (__temp21), NULL));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public enum ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) en)));
	vala_interface_writer_write_begin_block (self);
	vala_code_node_accept_children (VALA_CODE_NODE (en), VALA_CODE_VISITOR (self));
	vala_interface_writer_write_end_block (self);
	vala_interface_writer_write_newline (self);
	(cheaders = (g_free (cheaders), NULL));
}


static void vala_interface_writer_real_visit_enum_value (ValaCodeVisitor* base, ValaEnumValue* ev)
{
	ValaInterfaceWriter * self;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_ENUM_VALUE (ev));
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) ev)));
	vala_interface_writer_write_string (self, ",");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_constant (ValaCodeVisitor* base, ValaConstant* c)
{
	ValaInterfaceWriter * self;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_CONSTANT (c));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) c)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) c))))) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public const ");
	vala_interface_writer_write_type (self, vala_constant_get_type_reference (c));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) c)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_field (ValaCodeVisitor* base, ValaField* f)
{
	ValaInterfaceWriter * self;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_FIELD (f));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) f)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) f))))) {
		return;
	}
	if (f->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (vala_type_reference_get_data_type (vala_field_get_type_reference (f)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_field_get_type_reference (f))) && !vala_type_reference_get_takes_ownership (vala_field_get_type_reference (f))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	vala_interface_writer_write_type (self, vala_field_get_type_reference (f));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) f)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_write_params (ValaInterfaceWriter* self, GeeCollection* params)
{
	gboolean first;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (params == NULL || GEE_IS_COLLECTION (params));
	vala_interface_writer_write_string (self, "(");
	first = TRUE;
	{
		GeeCollection* param_collection;
		GeeIterator* param_it;
		param_collection = params;
		param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
		while (gee_iterator_next (param_it)) {
			ValaFormalParameter* param;
			param = gee_iterator_get (param_it);
			{
				if (!first) {
					vala_interface_writer_write_string (self, ", ");
				} else {
					first = FALSE;
				}
				if (vala_formal_parameter_get_ellipsis (param)) {
					vala_interface_writer_write_string (self, "...");
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
					continue;
				}
				if (vala_type_reference_get_is_ref (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "ref ");
				} else {
					if (vala_type_reference_get_is_out (vala_formal_parameter_get_type_reference (param))) {
						vala_interface_writer_write_string (self, "out ");
					}
				}
				vala_interface_writer_write_type (self, vala_formal_parameter_get_type_reference (param));
				if (vala_type_reference_get_takes_ownership (vala_formal_parameter_get_type_reference (param))) {
					vala_interface_writer_write_string (self, "#");
				}
				vala_interface_writer_write_string (self, " ");
				vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) param)));
				if (vala_formal_parameter_get_default_expression (param) != NULL) {
					char* __temp23;
					vala_interface_writer_write_string (self, " = ");
					__temp23 = NULL;
					vala_interface_writer_write_string (self, (__temp23 = vala_code_node_to_string (VALA_CODE_NODE (vala_formal_parameter_get_default_expression (param)))));
					(__temp23 = (g_free (__temp23), NULL));
				}
				(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
			}
		}
		(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
	}
	vala_interface_writer_write_string (self, ")");
}


static void vala_interface_writer_real_visit_callback (ValaCodeVisitor* base, ValaCallback* cb)
{
	ValaInterfaceWriter * self;
	ValaDataType* __temp24;
	ValaDataType* type;
	GeeCollection* __temp25;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_CALLBACK (cb));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) cb)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) cb))))) {
		return;
	}
	if (((ValaDataType*) cb)->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public static delegate ");
	__temp24 = NULL;
	type = (__temp24 = vala_type_reference_get_data_type (vala_callback_get_return_type (cb)), (__temp24 == NULL ? NULL : g_object_ref (__temp24)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		if (vala_type_reference_get_transfers_ownership (vala_callback_get_return_type (cb))) {
		} else {
			if ((vala_type_reference_get_data_type (vala_callback_get_return_type (cb)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_callback_get_return_type (cb)))) || vala_type_reference_get_type_parameter (vala_callback_get_return_type (cb)) != NULL) {
				vala_interface_writer_write_string (self, "weak ");
			}
		}
		vala_interface_writer_write_type (self, vala_callback_get_return_type (cb));
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) cb)));
	vala_interface_writer_write_string (self, " ");
	__temp25 = NULL;
	vala_interface_writer_write_params (self, (__temp25 = vala_callback_get_parameters (cb)));
	(__temp25 == NULL ? NULL : (__temp25 = (g_object_unref (__temp25), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_real_visit_method (ValaCodeVisitor* base, ValaMethod* m)
{
	ValaInterfaceWriter * self;
	char* __temp27;
	char* __temp26;
	gboolean __temp28;
	GeeCollection* __temp32;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_METHOD (m));
	if (vala_code_node_get_source_reference (((ValaCodeNode*) m)) != NULL && vala_source_file_get_pkg (vala_source_reference_get_file (vala_code_node_get_source_reference (((ValaCodeNode*) m))))) {
		return;
	}
	if (m->access == VALA_MEMBER_ACCESSIBILITY_PRIVATE || vala_method_get_overrides (m)) {
		return;
	}
	if (vala_method_get_no_array_length (m)) {
		gboolean array_found;
		array_found = (vala_method_get_return_type (m) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_method_get_return_type (m))));
		{
			GeeCollection* param_collection;
			GeeIterator* param_it;
			param_collection = vala_invokable_get_parameters (VALA_INVOKABLE (m));
			param_it = gee_iterable_iterator (GEE_ITERABLE (param_collection));
			while (gee_iterator_next (param_it)) {
				ValaFormalParameter* param;
				param = gee_iterator_get (param_it);
				{
					if (vala_formal_parameter_get_type_reference (param) != NULL && VALA_IS_ARRAY (vala_type_reference_get_data_type (vala_formal_parameter_get_type_reference (param)))) {
						array_found = TRUE;
						(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
						break;
					}
					(param == NULL ? NULL : (param = (g_object_unref (param), NULL)));
				}
			}
			(param_collection == NULL ? NULL : (param_collection = (g_object_unref (param_collection), NULL)));
			(param_it == NULL ? NULL : (param_it = (g_object_unref (param_it), NULL)));
		}
		if (array_found) {
			vala_interface_writer_write_indent (self);
			vala_interface_writer_write_string (self, "[NoArrayLength]");
		}
	}
	if (vala_method_get_instance_last (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceLast]");
	}
	if (vala_method_get_instance_by_reference (m)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[InstanceByReference]");
	}
	__temp27 = NULL;
	__temp26 = NULL;
	if ((__temp28 = g_utf8_collate ((__temp26 = vala_method_get_cname (m)), (__temp27 = vala_method_get_default_cname (m))) != 0, (__temp27 = (g_free (__temp27), NULL)), (__temp26 = (g_free (__temp26), NULL)), __temp28)) {
		char* __temp30;
		char* __temp29;
		vala_interface_writer_write_indent (self);
		__temp30 = NULL;
		__temp29 = NULL;
		vala_interface_writer_write_string (self, (__temp30 = g_strdup_printf ("[CCode (cname = \"%s\")]", (__temp29 = vala_method_get_cname (m)))));
		(__temp30 = (g_free (__temp30), NULL));
		(__temp29 = (g_free (__temp29), NULL));
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public");
	if (VALA_IS_CREATION_METHOD (m)) {
		ValaDataType* __temp31;
		ValaDataType* datatype;
		vala_interface_writer_write_string (self, " ");
		__temp31 = NULL;
		datatype = (__temp31 = VALA_DATA_TYPE (vala_symbol_get_parent_symbol (((ValaSymbol*) m))), (__temp31 == NULL ? NULL : g_object_ref (__temp31)));
		vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) datatype)));
		if (vala_symbol_get_name (((ValaSymbol*) m)) != NULL) {
			vala_interface_writer_write_string (self, ".");
			vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) m)));
		}
		(datatype == NULL ? NULL : (datatype = (g_object_unref (datatype), NULL)));
	} else {
		if (!vala_method_get_instance (m)) {
			vala_interface_writer_write_string (self, " static");
		} else {
			if (vala_method_get_is_abstract (m)) {
				vala_interface_writer_write_string (self, " abstract");
			} else {
				if (vala_method_get_is_virtual (m)) {
					vala_interface_writer_write_string (self, " virtual");
				}
			}
		}
	}
	if (!(VALA_IS_CREATION_METHOD (m))) {
		vala_interface_writer_write_string (self, " ");
		if (vala_type_reference_get_data_type (vala_method_get_return_type (m)) == NULL && vala_type_reference_get_type_parameter (vala_method_get_return_type (m)) == NULL) {
			vala_interface_writer_write_string (self, "void");
		} else {
			if (vala_type_reference_get_transfers_ownership (vala_method_get_return_type (m))) {
			} else {
				if ((vala_type_reference_get_data_type (vala_method_get_return_type (m)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_method_get_return_type (m)))) || vala_type_reference_get_type_parameter (vala_method_get_return_type (m)) != NULL) {
					vala_interface_writer_write_string (self, "weak ");
				}
			}
			vala_interface_writer_write_type (self, vala_method_get_return_type (m));
		}
		vala_interface_writer_write_string (self, " ");
		vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) m)));
	}
	vala_interface_writer_write_string (self, " ");
	__temp32 = NULL;
	vala_interface_writer_write_params (self, (__temp32 = vala_invokable_get_parameters (VALA_INVOKABLE (m))));
	(__temp32 == NULL ? NULL : (__temp32 = (g_object_unref (__temp32), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_creation_method (ValaCodeVisitor* base, ValaCreationMethod* m)
{
	ValaInterfaceWriter * self;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_CREATION_METHOD (m));
	vala_code_visitor_visit_method (VALA_CODE_VISITOR (self), VALA_METHOD (m));
}


static void vala_interface_writer_real_visit_property (ValaCodeVisitor* base, ValaProperty* prop)
{
	ValaInterfaceWriter * self;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_PROPERTY (prop));
	if (vala_property_get_no_accessor_method (prop)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[NoAccessorMethod]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public ");
	if (!vala_type_reference_get_takes_ownership (vala_property_get_type_reference (prop))) {
		vala_interface_writer_write_string (self, "weak ");
	}
	vala_interface_writer_write_type (self, vala_property_get_type_reference (prop));
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) prop)));
	vala_interface_writer_write_string (self, " {");
	if (vala_property_get_get_accessor (prop) != NULL) {
		vala_interface_writer_write_string (self, " get;");
	}
	if (vala_property_get_set_accessor (prop) != NULL) {
		if (vala_property_accessor_get_writable (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " set");
		}
		if (vala_property_accessor_get_construction (vala_property_get_set_accessor (prop))) {
			vala_interface_writer_write_string (self, " construct");
		}
		vala_interface_writer_write_string (self, ";");
	}
	vala_interface_writer_write_string (self, " }");
	vala_interface_writer_write_newline (self);
}


static void vala_interface_writer_real_visit_signal (ValaCodeVisitor* base, ValaSignal* sig)
{
	ValaInterfaceWriter * self;
	ValaDataType* __temp33;
	ValaDataType* type;
	GeeCollection* __temp34;
	self = ((ValaInterfaceWriter*) base);
	g_return_if_fail (VALA_IS_SIGNAL (sig));
	if (vala_signal_get_access (sig) == VALA_MEMBER_ACCESSIBILITY_PRIVATE) {
		return;
	}
	if (vala_signal_get_has_emitter (sig)) {
		vala_interface_writer_write_indent (self);
		vala_interface_writer_write_string (self, "[HasEmitter]");
	}
	vala_interface_writer_write_indent (self);
	vala_interface_writer_write_string (self, "public signal ");
	__temp33 = NULL;
	type = (__temp33 = vala_type_reference_get_data_type (vala_signal_get_return_type (sig)), (__temp33 == NULL ? NULL : g_object_ref (__temp33)));
	if (type == NULL) {
		vala_interface_writer_write_string (self, "void");
	} else {
		if (vala_type_reference_get_transfers_ownership (vala_signal_get_return_type (sig))) {
		} else {
			if ((vala_type_reference_get_data_type (vala_signal_get_return_type (sig)) != NULL && vala_data_type_is_reference_type (vala_type_reference_get_data_type (vala_signal_get_return_type (sig)))) || vala_type_reference_get_type_parameter (vala_signal_get_return_type (sig)) != NULL) {
				vala_interface_writer_write_string (self, "weak ");
			}
		}
		vala_interface_writer_write_type (self, vala_signal_get_return_type (sig));
	}
	vala_interface_writer_write_string (self, " ");
	vala_interface_writer_write_identifier (self, vala_symbol_get_name (((ValaSymbol*) sig)));
	vala_interface_writer_write_string (self, " ");
	__temp34 = NULL;
	vala_interface_writer_write_params (self, (__temp34 = vala_invokable_get_parameters (VALA_INVOKABLE (sig))));
	(__temp34 == NULL ? NULL : (__temp34 = (g_object_unref (__temp34), NULL)));
	vala_interface_writer_write_string (self, ";");
	vala_interface_writer_write_newline (self);
	(type == NULL ? NULL : (type = (g_object_unref (type), NULL)));
}


static void vala_interface_writer_write_indent (ValaInterfaceWriter* self)
{
	gint i;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	i = 0;
	if (!self->priv->bol) {
		fputc ('\n', self->priv->stream);
	}
	for (i = 0; i < self->priv->indent; i++) {
		fputc ('\t', self->priv->stream);
	}
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_identifier (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	if (g_utf8_collate (s, "base") == 0 || g_utf8_collate (s, "class") == 0 || g_utf8_collate (s, "construct") == 0 || g_utf8_collate (s, "delegate") == 0 || g_utf8_collate (s, "foreach") == 0 || g_utf8_collate (s, "in") == 0 || g_utf8_collate (s, "interface") == 0 || g_utf8_collate (s, "lock") == 0 || g_utf8_collate (s, "namespace") == 0 || g_utf8_collate (s, "out") == 0 || g_utf8_collate (s, "ref") == 0) {
		fputc ('@', self->priv->stream);
	}
	vala_interface_writer_write_string (self, s);
}


static void vala_interface_writer_write_type (ValaInterfaceWriter* self, ValaTypeReference* type)
{
	GeeList* type_args;
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (VALA_IS_TYPE_REFERENCE (type));
	if (vala_type_reference_get_data_type (type) != NULL) {
		char* __temp35;
		__temp35 = NULL;
		vala_interface_writer_write_string (self, (__temp35 = vala_symbol_get_full_name (VALA_SYMBOL (vala_type_reference_get_data_type (type)))));
		(__temp35 = (g_free (__temp35), NULL));
	} else {
		vala_interface_writer_write_string (self, vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (type))));
	}
	type_args = vala_type_reference_get_type_arguments (type);
	if (!(VALA_IS_ARRAY (vala_type_reference_get_data_type (type))) && gee_collection_get_size (((GeeCollection*) type_args)) > 0) {
		gboolean first;
		vala_interface_writer_write_string (self, "<");
		first = TRUE;
		{
			GeeList* type_arg_collection;
			GeeIterator* type_arg_it;
			type_arg_collection = type_args;
			type_arg_it = gee_iterable_iterator (GEE_ITERABLE (type_arg_collection));
			while (gee_iterator_next (type_arg_it)) {
				ValaTypeReference* type_arg;
				type_arg = gee_iterator_get (type_arg_it);
				{
					if (!first) {
						vala_interface_writer_write_string (self, ",");
					} else {
						first = FALSE;
					}
					if (!vala_type_reference_get_takes_ownership (type_arg)) {
						vala_interface_writer_write_string (self, "weak ");
					}
					if (vala_type_reference_get_data_type (type_arg) != NULL) {
						char* __temp36;
						__temp36 = NULL;
						vala_interface_writer_write_string (self, (__temp36 = vala_symbol_get_full_name (VALA_SYMBOL (vala_type_reference_get_data_type (type_arg)))));
						(__temp36 = (g_free (__temp36), NULL));
					} else {
						vala_interface_writer_write_string (self, vala_symbol_get_name (((ValaSymbol*) vala_type_reference_get_type_parameter (type_arg))));
					}
					(type_arg == NULL ? NULL : (type_arg = (g_object_unref (type_arg), NULL)));
				}
			}
			(type_arg_it == NULL ? NULL : (type_arg_it = (g_object_unref (type_arg_it), NULL)));
		}
		vala_interface_writer_write_string (self, ">");
	}
	if (vala_type_reference_get_non_null (type)) {
		vala_interface_writer_write_string (self, "!");
	}
	(type_args == NULL ? NULL : (type_args = (g_object_unref (type_args), NULL)));
}


static void vala_interface_writer_write_string (ValaInterfaceWriter* self, const char* s)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	g_return_if_fail (s != NULL);
	fprintf (self->priv->stream, "%s", s);
	self->priv->bol = FALSE;
}


static void vala_interface_writer_write_newline (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	fputc ('\n', self->priv->stream);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_write_begin_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	if (!self->priv->bol) {
		fputc (' ', self->priv->stream);
	} else {
		vala_interface_writer_write_indent (self);
	}
	fputc ('{', self->priv->stream);
	vala_interface_writer_write_newline (self);
	self->priv->indent++;
}


static void vala_interface_writer_write_end_block (ValaInterfaceWriter* self)
{
	g_return_if_fail (VALA_IS_INTERFACE_WRITER (self));
	self->priv->indent--;
	vala_interface_writer_write_indent (self);
	fprintf (self->priv->stream, "}");
}


static void vala_interface_writer_class_init (ValaInterfaceWriterClass * klass)
{
	vala_interface_writer_parent_class = g_type_class_peek_parent (klass);
	g_type_class_add_private (klass, sizeof (ValaInterfaceWriterPrivate));
	G_OBJECT_CLASS (klass)->dispose = vala_interface_writer_dispose;
	VALA_CODE_VISITOR_CLASS (klass)->visit_namespace = vala_interface_writer_real_visit_namespace;
	VALA_CODE_VISITOR_CLASS (klass)->visit_class = vala_interface_writer_real_visit_class;
	VALA_CODE_VISITOR_CLASS (klass)->visit_struct = vala_interface_writer_real_visit_struct;
	VALA_CODE_VISITOR_CLASS (klass)->visit_interface = vala_interface_writer_real_visit_interface;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum = vala_interface_writer_real_visit_enum;
	VALA_CODE_VISITOR_CLASS (klass)->visit_enum_value = vala_interface_writer_real_visit_enum_value;
	VALA_CODE_VISITOR_CLASS (klass)->visit_constant = vala_interface_writer_real_visit_constant;
	VALA_CODE_VISITOR_CLASS (klass)->visit_field = vala_interface_writer_real_visit_field;
	VALA_CODE_VISITOR_CLASS (klass)->visit_callback = vala_interface_writer_real_visit_callback;
	VALA_CODE_VISITOR_CLASS (klass)->visit_method = vala_interface_writer_real_visit_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_creation_method = vala_interface_writer_real_visit_creation_method;
	VALA_CODE_VISITOR_CLASS (klass)->visit_property = vala_interface_writer_real_visit_property;
	VALA_CODE_VISITOR_CLASS (klass)->visit_signal = vala_interface_writer_real_visit_signal;
}


static void vala_interface_writer_init (ValaInterfaceWriter * self)
{
	self->priv = VALA_INTERFACE_WRITER_GET_PRIVATE (self);
	self->priv->bol = TRUE;
}


static void vala_interface_writer_dispose (GObject * obj)
{
	ValaInterfaceWriter * self;
	ValaInterfaceWriterClass * klass;
	GObjectClass * parent_class;
	self = VALA_INTERFACE_WRITER (obj);
	(self->priv->context == NULL ? NULL : (self->priv->context = (g_object_unref (self->priv->context), NULL)));
	(self->priv->stream == NULL ? NULL : (self->priv->stream = (fclose (self->priv->stream), NULL)));
	(self->priv->current_cheader_filename = (g_free (self->priv->current_cheader_filename), NULL));
	klass = VALA_INTERFACE_WRITER_CLASS (g_type_class_peek (VALA_TYPE_INTERFACE_WRITER));
	parent_class = G_OBJECT_CLASS (g_type_class_peek_parent (klass));
	parent_class->dispose (obj);
}


GType vala_interface_writer_get_type (void)
{
	static GType vala_interface_writer_type_id = 0;
	if (G_UNLIKELY (vala_interface_writer_type_id == 0)) {
		static const GTypeInfo g_define_type_info = { sizeof (ValaInterfaceWriterClass), (GBaseInitFunc) NULL, (GBaseFinalizeFunc) NULL, (GClassInitFunc) vala_interface_writer_class_init, (GClassFinalizeFunc) NULL, NULL, sizeof (ValaInterfaceWriter), 0, (GInstanceInitFunc) vala_interface_writer_init };
		vala_interface_writer_type_id = g_type_register_static (VALA_TYPE_CODE_VISITOR, "ValaInterfaceWriter", &g_define_type_info, 0);
	}
	return vala_interface_writer_type_id;
}




