/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#ifndef VISU_PAIRS
#define VISU_PAIRS

#include "visu_data.h"
#include "coreTools/toolColor.h"

G_BEGIN_DECLS

/**
 * VISU_PAIR_DISTANCE_MIN
 *
 * Flag used to define the minimum length to draw pair. This is useful with
 * the visu_pair_data_getDistance() and the visu_pair_data_setDistance() methods.
 */
#define VISU_PAIR_DISTANCE_MIN 0
/**
 * VISU_PAIR_DISTANCE_MAX
 *
 * Flag used to define the maximum length to draw pair. This is useful with
 * the visu_pair_data_getDistance() and the visu_pair_data_setDistance() methods.
 */
#define VISU_PAIR_DISTANCE_MAX 1

typedef struct _VisuPairData VisuPairData;

/**
 * VisuPairStartEndFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object.
 *
 * Prototype of functions called at the beginning and
 * the end of drawing of each pairs types. @ele1 and @ele2
 * arguments are the two elements between the pair defined by @data is drawn.
 * This is useful to set some OpenGL definition specific to each pair, such
 * as the color for example.
 */
typedef void (*VisuPairStartEndFunc)(VisuElement *ele1, VisuElement *ele2,
                                     VisuPairData *data);
/**
 * VisuPairDrawFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object ;
 * @view: a #VisuOpenGLView object, giving some constants describing
 *        the OpenGL scene ;
 * @x1: a floating point value ;
 * @y1: a floating point value ;
 * @z1: a floating point value ;
 * @x2: a floating point value ;
 * @y2: a floating point value ;
 * @z2: a floating point value ;
 * @d2: a floating point value ;
 * @alpha: a floating point value.
 *
 * Prototype of function to draw a pair. Such function are called each time a pair
 * is drawn between the two points (@x1, @y1, @z1) and (@x2, @y2, @z2). The @d2 argument
 * is the square distance between the two points. The @alpha argument
 * is a proposed alpha colour from the main program, its value is in [0;1].
 */
typedef void (*VisuPairDrawFunc)(VisuElement *ele1, VisuElement *ele2,
                                 VisuPairData *data, VisuOpenGLView *view,
                                 double x1, double y1, double z1,
                                 double x2, double y2, double z2,
                                 float d2, float alpha);

/**
 * VisuPairExtension:
 * @name: an internal name ;
 * @printName: the UTF-8 name, used for the GUI ;
 * @description: an UTF-8 short description of the pairs method.
 * @sensitiveToFacette: if TRUE, the OpenGL list of the pairs is recompute
 *                      each time the number of facette changes ;
 * @beginDrawingPairs: method called a startup of drawing pairs ;
 * @endDrawingPairs: method called a ending of drawing pairs ;
 * @drawPairs: the drawing method for each pair.
 *
 * Structure to store pairs extensions. All fields are private and should not
 * be accessed directly. This structure will not be public in near future, do not use it.
 */
struct _VisuPairExtension
{
  /* Some variable to describe this OpenGL extension.
     The attribute name is mandatory since it is
     used to identify the method. */
  char* name;
  char* printName;
  char* description;

  gboolean sensitiveToFacette;

  VisuPairStartEndFunc beginDrawingPairs;
  VisuPairStartEndFunc endDrawingPairs;
  VisuPairDrawFunc drawPairs;
};
typedef struct _VisuPairExtension VisuPairExtension;

/**
 * VisuPairData:
 * @minMax: storage for the length bounds for drawn pairs ;
 * @drawn: a boolean to say if the pair is drawn or not ;
 * @printLength: a boolean to say if length of pairs are drawn near
 * them ;
 * @method: the rendering method for this link.
 * @properties: some key:value association for this link.
 *
 * This structure is used to describe a link between two elements. A
 * link is drawn only its length is between the minimum and the
 * maximum value stored in the minMax array.
 */
struct _VisuPairData
{
  float minMax[2];
  gboolean drawn;
  gboolean printLength;

  VisuPairExtension *method;

  GHashTable *properties;
};

/**
 * VisuPair:
 * 
 * An opaque structure to define links (i.e. several #VisuPairData)
 * between elements.
 */
typedef struct _VisuPair VisuPair;

/**
 * VisuPairDistribution:
 * @ele1: one #VisuElement.
 * @ele2: one #VisuElement.
 * @histo: an array containing the distribution ;
 * @nValues: the size of the array ;
 * @initValue: the initial distance value (usualy 0) ;
 * @stepValue: the step increase in distance at each value ;
 * @nNodesEle1: the number of nodes used during the computation ;
 * @nNodesEle2: idem for #VisuElement 2.
 *
 * This structure stores for a given pair, the distance distribution
 * on a given range [@initValue;@nValues * @stepValue[.
 */
struct _VisuPairDistribution
{
  VisuElement *ele1, *ele2;
  guint *histo;
  guint nValues;
  float initValue, stepValue;
  guint nNodesEle1, nNodesEle2;
};
typedef struct _VisuPairDistribution VisuPairDistribution;


VisuPairExtension* visu_pair_extension_new(const char* name, const char* printName,
                                           const char* description, gboolean sensitive,
                                           VisuPairStartEndFunc start,
                                           VisuPairStartEndFunc end,
                                           VisuPairDrawFunc draw);
void visu_pair_extension_free(VisuPairExtension* extension);
void visu_pair_extension_add(VisuPairExtension *extension);
gboolean visu_pair_extension_setDefault(VisuPairExtension *extension);
GList* visu_pair_extension_getAllMethods();
VisuPairExtension* visu_pair_extension_getDefault();



gboolean visu_pair_build(VisuData *dataObj);
gboolean visu_pair_setStatus(gboolean onOff);
void visu_pair_setOutOfDate();
gboolean visu_pair_getStatus();

VisuPair* visu_pair_getPair(VisuElement *ele1, VisuElement *ele2);
void visu_pair_setProperty(VisuPair *pair, const gchar* key,
                           gpointer value, GDestroyNotify freeFunc);
gpointer visu_pair_getProperty(VisuPair *pair, const gchar* key);
VisuPairDistribution* visu_pair_getDistanceDistribution(VisuPair *pair,
                                                        VisuData *dataObj,
                                                        float step,
                                                        float min, float max);
gboolean visu_pair_distribution_getNextPick(VisuPairDistribution *dd,
                                            guint startStopId[2], guint *integral,
                                            guint *max, guint *posMax);

GList* visu_pair_getAllPairData(VisuElement *ele1, VisuElement *ele2);
VisuPairData* visu_pair_getPairData(VisuElement *ele1, VisuElement *ele2,
                                    float minMax[2]);
VisuPairData* visu_pair_getPairDataFromId(VisuElement *ele1, VisuElement *ele2,
                                          guint pos);
gboolean visu_pair_removePairData(VisuElement *ele1, VisuElement *ele2,
                                  VisuPairData *data);
gboolean           visu_pair_data_setDrawn      (VisuPairData *data, gboolean drawn);
gboolean           visu_pair_data_getDrawn      (VisuPairData *data);
gboolean           visu_pair_data_setDistance   (VisuPairData *data,
                                                 float val, int minOrMax);
gboolean           visu_pair_data_setColor      (VisuPairData *data,
                                                 ToolColor* destColor);
ToolColor*         visu_pair_data_getColor      (VisuPairData *data);
gboolean           visu_pair_data_setPrintLength(VisuPairData *data, gboolean status);
gboolean           visu_pair_data_setDrawMethod (VisuPairData *data,
                                                 VisuPairExtension *ext);
void               visu_pair_data_setProperty   (VisuPairData *data,
                                                 const gchar* key, gpointer value);
float              visu_pair_data_getDistance   (VisuPairData *data, int minOrMax);
gboolean           visu_pair_data_getPrintLength(VisuPairData *data);
gpointer           visu_pair_data_getProperty   (VisuPairData *data, const gchar* key);
VisuPairExtension* visu_pair_data_getDrawMethod (VisuPairData *data);


/**
 * VisuPairForeachFunc:
 * @ele1: a #VisuElement object ;
 * @ele2: a #VisuElement object ;
 * @data: a #VisuPairData object ;
 * @userData: some user defined data.
 *
 * Prototype of functions called with the foreach method apply to each
 * pairs.
 */
typedef void (*VisuPairForeachFunc)(VisuElement *ele1, VisuElement *ele2,
                                    VisuPairData *data, gpointer userData);
/**
 * visu_pair_foreach:
 * @whatToDo: a VisuPairForeachFunc() method ;
 * @userData: some user defined data.
 *
 * The way #VisuPairData are stored in V_Sim is private and could changed between version.
 * This method is used to apply some method each pairs.
 */
void visu_pair_foreach(VisuPairForeachFunc whatToDo, gpointer userData);
/**
 * visu_pair_readLinkFromTokens:
 * @tokens: array of tokens resulting from a call to g_strsplit() with " " as separator ;
 * @index: IN, the position of the beginning in @tokens ; OUT, one token
 *            after the last read ;
 * @data: a pointer to return an allocated link object ;
 * @position: the number of the line of the config
 *            file which the @line argument is taken from ;
 * @error: a location to store a possible reading error.
 *
 * This routine is used to read the resource file. Given @tokens, it
 * associate a link object by reading the two elements and the two
 * distances that characterised this link.
 *
 * Returns: TRUE if succeed.
 */
gboolean visu_pair_readLinkFromTokens(gchar **tokens, int *index, VisuPairData **data,
				     int position, GError **error);


/****************/
/* Private area */
/****************/

/**
 * initPairsModule:
 *
 * Initialise all the variable of this part. It calls all
 * the elements in listInitPairsFunc (that stores the
 * init function of the pairs extensions). If these elements return
 * valid VisuPairExtension, they are registered through a call
 * to visu_pair_extension_add().
 * This method is called by the main program at the initialisation stage and
 * should not be called in others circonstempses.
 *
 * Returns: 1 if everything goes allright during the initialisation.
 */
int initPairsModule();

G_END_DECLS

#endif
