/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse mèl :
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant à visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est régi par la licence CeCILL soumise au droit français et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffusée par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accéder à cet en-tête signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accepté les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "visu_object.h"

#include <glib.h>

#include "renderingMethods/renderingAtomic.h"

/**
 * SECTION:visu_object
 * @short_description: A general object to store the signals.
 * 
 * <para>At the present time, the signals are global to all V_Sim,
 * owned by a variable include in each parts called visu. This is
 * wherre these signals are defined.</para>
 */

/**
 * VisuObject:
 * @parent: an object to inherit from (NULL here).
 * @privateDt: a pointer to the private data.
 *
 * This structure describes a #VisuObject object.
 */
struct _VisuObject
{
  GObject parent;

  VisuRendering *render;
};

/* All signal emit by visu are handled by this ogbject. */
static VisuObject *visu = (VisuObject*)0;

G_DEFINE_TYPE(VisuObject, visuObject, G_TYPE_OBJECT)

static void visuObject_class_init(VisuObjectClass *klass)
{
  GType paramPointer[1] = {G_TYPE_POINTER};
  GType paramObject[1] = {G_TYPE_OBJECT};
  GType paramGuint[1] = {G_TYPE_UINT};

  DBG_fprintf(stderr, "Visu Object: installing signals.\n");
  /**
   * VisuObject::colorNewAvailable:
   * @visuObj: the object emitting the signal.
   * @color: the newly created #ToolColor.
   *
   * A new #ToolColor is available.
   *
   * Since: 3.2
   */
  klass->signals[COLORNEWAVAILABLE_SIGNAL] = 
    g_signal_newv ("colorNewAvailable",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE /* return_type */,
		   1     /* n_params */,
		   paramPointer /* pointer to the added color */);
  /**
   * VisuObject::dataNew:
   * @visuObj: the object emitting the signal.
   * @dataObj: the newly created #VisuData.
   *
   * A new #VisuData is available.
   *
   * Since: 3.2
   */
  klass->signals[DATANEW_SIGNAL] = 
    g_signal_newv("dataNew",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL,
		  NULL, 
		  NULL,              
		  g_cclosure_marshal_VOID__OBJECT,
		  G_TYPE_NONE, 1, paramObject);
  /**
   * VisuObject::dataLoaded:
   * @visuObj: the object emitting the signal.
   * @dataObj: the newly created #VisuData.
   *
   * The given @dataObj is fully populated and ready for usage.
   *
   * Since: 3.1
   */
  klass->signals[DATALOADED_SIGNAL] = 
    g_signal_newv("dataLoaded",
		  G_TYPE_FROM_CLASS(klass),
		  G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		  NULL,
		  NULL, 
		  NULL,              
		  g_cclosure_marshal_VOID__OBJECT,
		  G_TYPE_NONE, 1, paramObject);
  /**
   * VisuObject::dataReadyForRendering:
   * @visuObj: the object emitting the signal.
   * @dataObj: the newly created #VisuData.
   *
   * The given @dataObj is fully set up and ready for rendering (no
   * further internal modifications will occur).
   */
  klass->signals[DATAREADYFORRENDERING_SIGNAL] = 
    g_signal_newv ("dataReadyForRendering",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__OBJECT,
		   G_TYPE_NONE /* return_type */,
		   1     /* n_params */,
		   paramObject  /* param_types */);
  /**
   * VisuObject::renderingChanged:
   * @visuObj: the object emitting the signal.
   * @meth: the newly chosen #VisuRendering method.
   *
   * The rendering method has been changed.
   */
  klass->signals[RENDERINGCHANGED_SIGNAL] = 
    g_signal_newv ("renderingChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__POINTER,
		   G_TYPE_NONE /* return_type */,
		   1     /* n_params */,
		   paramPointer  /* param_types */);
  /**
   * VisuObject::resourcesLoaded:
   * @visuObj: the object emitting the signal.
   * @dataObj: the associated #VisuData.
   *
   * The resource file has been read.
   */
  klass->signals[RESOURCESLOADED_SIGNAL] = 
    g_signal_newv ("resourcesLoaded",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__OBJECT,
		   G_TYPE_NONE /* return_type */,
		   1     /* n_params */,
		   paramObject  /* param_types */);
  /**
   * VisuObject::OpenGLAskForReDraw:
   * @visuObj: the object emitting the signal.
   *
   * Internal signal, use VISU_ADD_REDRAW() instead.
   */
  klass->signals[OPENGLASKFORREDRAW_SIGNAL] = 
    g_signal_newv ("OpenGLAskForReDraw",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE /* return_type */,
		   0     /* n_params */,
		   NULL  /* param_types */);
  /**
   * VisuObject::OpenGLForceReDraw:
   * @visuObj: the object emitting the signal.
   *
   * Internal signal, use VISU_FORCE_REDRAW() instead.
   */
  klass->signals[OPENGLFORCEREDRAW_SIGNAL] = 
    g_signal_newv ("OpenGLForceReDraw",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL /* class closure */,
		   NULL /* accumulator */,
		   NULL /* accu_data */,
		   g_cclosure_marshal_VOID__VOID,
		   G_TYPE_NONE /* return_type */,
		   0     /* n_params */,
		   NULL  /* param_types */);

  /**
   * VisuObject::DirectoryChanged:
   * @visuObj: the object emitting the signal.
   * @kind: a flag.
   *
   * The current directory has been changed. The kind of directory is
   * defined by @kind (see #).
   *
   * Since: 3.6
   */
  klass->signals[DIR_SIGNAL] = 
    g_signal_newv ("DirectoryChanged",
		   G_TYPE_FROM_CLASS (klass),
		   G_SIGNAL_RUN_LAST | G_SIGNAL_NO_RECURSE | G_SIGNAL_NO_HOOKS,
		   NULL , NULL, NULL,
		   g_cclosure_marshal_VOID__UINT,
		   G_TYPE_NONE /* return_type */,
		   1     /* n_params */,
		   paramGuint  /* param_types */);
}

static void visuObject_init(VisuObject *obj)
{
  DBG_fprintf(stderr, "Visu Object: creating a new visu (%p).\n", (gpointer)obj);

  /* Putting the default. */
  obj->render = visu_rendering_getByName(VISU_RENDERING_ATOMIC_NAME);
}

/**
 * visu_object_redraw:
 * @data: (allow-none): a string.
 *
 * Call the signal OpenGLAskForReDraw. The API is adapted to the routine to be added
 * in the gloop. Use VISU_ADD_REDRAW instead of this routine.
 *
 * Returns: FALSE to stop the emission as soon as done.
 */
gboolean visu_object_redraw(gpointer data)
{
  if (data)
    DBG_fprintf(stderr, "Visu Object: '%s' call for redraw.\n", (gchar*)data);
  g_signal_emit (visu, VISU_GET_CLASS(visu)->signals[OPENGLASKFORREDRAW_SIGNAL],
		 0 /* details */, NULL);
  return FALSE;
}
/**
 * visu_object_redrawForce:
 * @data: (allow-none): a string.
 *
 * Call the signal OpenGLForceReDraw. The API is adapted to the routine to be added
 * in the gloop. Use VISU_FORCE_REDRAW instead of this routine.
 *
 * Returns: FALSE to stop the emission as soon as done.
 */
gboolean visu_object_redrawForce(gpointer data)
{
  if (data)
    DBG_fprintf(stderr, "Visu Object: '%s' call for force redraw.\n", (gchar*)data);
  g_signal_emit(visu, VISU_GET_CLASS(visu)->signals[OPENGLFORCEREDRAW_SIGNAL],
		0 , NULL);
  return FALSE;
}

/**
 * visuObjectGet_static:
 *
 * Internal routine to access the #VisuObject object instanciated by
 * default. Use #VISU_INSTANCE instead.
 */
VisuObject* visuObjectGet_static()
{
  if (!visu)
    /* Creating the visu object to handle the signals. */
    visu = VISU(g_object_new(VISU_TYPE, NULL));

  return visu;
}

/**
 * visu_object_setRendering:
 * @obj: a #VisuObject object.
 * @method: a #VisuRendering method.
 *
 * Choose the method used to render the data.
 *
 * Returns: TRUE if the rendering method of @obj is actually changed.
 */
gboolean visu_object_setRendering(VisuObject *obj, VisuRendering* method)
{
  g_return_val_if_fail(IS_VISU_TYPE(obj), FALSE);
  g_return_val_if_fail(method, FALSE);

  DBG_fprintf(stderr, "Visu Object: set the rendering method to '%s' (%p).\n",
              visu_rendering_getName(method, TRUE), (gpointer)method);
  if (obj->render == method)
    return FALSE;
  obj->render = method;

  DBG_fprintf(stderr, "Visu Object: emit method changed (%p).\n", (gpointer)obj);
  g_signal_emit(obj, VISU_GET_CLASS(obj)->signals[RENDERINGCHANGED_SIGNAL],
		0, (gpointer)method, NULL);
  return TRUE;
}
/**
 * visu_object_getRendering:
 * @obj: a #VisuObject object.
 *
 * Get the current method used to render the data.
 *
 * Returns: the rendering method attached to @obj.
 */
VisuRendering* visu_object_getRendering(VisuObject *obj)
{
  g_return_val_if_fail(IS_VISU_TYPE(obj), (VisuRendering*)0);

  return obj->render;
}
