/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/
#include "wire.h"

#include <stdlib.h>
#include <math.h>

#include <GL/gl.h>
#include <GL/glu.h> 

#include <visu_tools.h>
#include <visu_data.h>
#include <visu_configFile.h>
#include <opengl.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolConfigFile.h>

static PairsExtension* pointerToPairExtension_wire;
static void changeLightForWire();
static void restoreLightAfterWire();

/* This function details how to read what is needed by PairsData. */
static gboolean readWireGeneralWidth(gchar **lines, int nbLines, int position,
				     VisuData *dataObj, GError **error);
static gboolean readWireWidth(gchar **lines, int nbLines, int position,
			      VisuData *dataObj, GError **error);
static gboolean readLinkWidth(gchar **lines, int nbLines, int position,
			      VisuData *dataObj, GError **error);
static gboolean readLinkStipple(gchar **lines, int nbLines, int position,
				VisuData *dataObj, GError **error);
/* This function details how to read the parameter FLAG_PARAMETER_PAIRS. */
static gboolean readWireNonLinear(gchar **lines, int nbLines, int position,
				  VisuData *dataObj, GError **error);
/* This function writes the parameter. */
static void exportParametersWire(GString *data, VisuData *dataObj);
/* This function save the resources. */
static void exportResourcesWire(GString *data, VisuData *dataObj);

#define FLAG_PARAMETER_WIRE   "wire_nonLinear"
#define DESC_PARAMETER_WIRE   "If the color of the pairs are corrected by their length ; boolean 0 or 1"
#define PARAMETER_WIRE_DEFAULT 0
static int wireNonLinear;

#define FLAG_RESOURCES_PAIR_WIDTH "pairWire_pairWidth"
#define DESC_RESOURCES_PAIR_WIDTH "Widths detail for each pair drawn ; 0 < integer < 10"
#define FLAG_RESOURCES_LINK_WIDTH "pairWire_linkWidth"
#define DESC_RESOURCES_LINK_WIDTH "Widths detail for each drawn link ; 0 < integer < 10"
#define FLAG_RESOURCES_WIRE_WIDTH "pairWire_width"
#define DESC_RESOURCES_WIRE_WIDTH "This value is the width for all pairs drawn ; 0 < integer < 10"
#define RESOURCES_WIRE_WIDTH_DEFAULT 2
#define RESOURCES_WIRE_WIDTH_MAX 10
static int wireWidth;

#define FLAG_RESOURCES_LINK_STIPPLE "pairWire_linkStipple"
#define DESC_RESOURCES_LINK_STIPPLE "Dot scheme detail for each drawn link ; 0 < integer < 2^16"

/* The variation of the color intensity if stored in
   an array whose values are stored when wireNonLinear became true.
   The size of the array is controlled by a define. */
#define NB_SAMPLE_COLOR_LENGTH 101
static float **colorLengthSample;


PairsExtension* wireGet_static()
{
  return pointerToPairExtension_wire;
}

static void beginPairLink(VisuElement *ele1 _U_, VisuElement *ele2 _U_,
			  VisuPairData *data)
{
  Color *color;
  int *width;
  guint16 *stipple;

  width = (int*)visuPairGet_linkProperty(data, "width");
  if (width)
    glLineWidth(*width);
  else
    glLineWidth(wireWidth);
  color = (Color*)visuPairGet_linkProperty(data, "color");
  if (color)
    glColor3fv(color->rgba);
  stipple = (guint16*)visuPairGet_linkProperty(data, "stipple");
  if (stipple)
    {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, *stipple);
    }
}

static void endPairLink(VisuElement *ele1 _U_, VisuElement *ele2 _U_,
			VisuPairData *data _U_)
{
  glDisable(GL_LINE_STIPPLE);
}

static void drawPairLink(VisuElement *ele1 _U_, VisuElement *ele2 _U_,
			 VisuPairData *data, OpenGLView *view _U_,
			 double x1, double y1, double z1,
			 double x2, double y2, double z2, float d2, float alpha)
{
  float length;
  int prop;
  Color *color;

  if (wireNonLinear)
    {
      length = sqrt(d2);
      prop = CLAMP((int)( (length - data->minMax[PAIRS_MIN]) /
			  (data->minMax[PAIRS_MAX] - data->minMax[PAIRS_MIN]) *
			  (float)(NB_SAMPLE_COLOR_LENGTH - 1) ),
		   0, NB_SAMPLE_COLOR_LENGTH - 1);
      
/*       fprintf(stderr, "This length %f is at position %d %f (%f %f)\n", length, prop, colorLengthSample[0][prop], data->minMax[PAIRS_MAX], data->minMax[PAIRS_MIN]); */
      glColor4f(colorLengthSample[0][prop],
		colorLengthSample[1][prop],
		colorLengthSample[2][prop],
		alpha);
    }
  else
    {
      color = (Color*)visuPairGet_linkProperty(data, "color");
      if (color)
	glColor4f(color->rgba[0],
		  color->rgba[1],
		  color->rgba[2],
		  alpha);
    }

  glBegin(GL_LINES);
  glVertex3d(x1, y1, z1);
  glVertex3d(x2, y2, z2);
  glEnd();
}

PairsExtension* initPairsWire()
{
  char *name = _("Wire pairs");
  char *desc = _("Pairs are rendered by flat lines."
		 " The color and the width can by chosen.");
  PairsExtension *extension;
  VisuConfigFileEntry *resourceEntry, *oldEntry;
  int i, j;

  extension = visuPairExtensionNew("Wire pairs", name, desc, FALSE,
				   changeLightForWire, restoreLightAfterWire,
				   beginPairLink, endPairLink, drawPairLink);

  wireWidth = RESOURCES_WIRE_WIDTH_DEFAULT;
  wireNonLinear = PARAMETER_WIRE_DEFAULT;

  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_WIRE_WIDTH,
					  DESC_RESOURCES_WIRE_WIDTH,
					  1, readWireGeneralWidth);
  oldEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
				     FLAG_RESOURCES_PAIR_WIDTH,
				     DESC_RESOURCES_PAIR_WIDTH,
				     1, readWireWidth);
  visuConfigFileSet_version(resourceEntry, 3.1f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_LINK_WIDTH,
					  DESC_RESOURCES_LINK_WIDTH,
					  1, readLinkWidth);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileSet_replace(resourceEntry, oldEntry);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCES_LINK_STIPPLE,
					  DESC_RESOURCES_LINK_STIPPLE,
					  1, readLinkStipple);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesWire);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_PARAMETER,
					  FLAG_PARAMETER_WIRE,
					  DESC_PARAMETER_WIRE,
					  1, readWireNonLinear);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_PARAMETER,
				   exportParametersWire);

  colorLengthSample = g_malloc(sizeof(float*) * 3);
  for (i = 0; i < 3; i++)
    {
      colorLengthSample[i] = g_malloc(sizeof(float) * NB_SAMPLE_COLOR_LENGTH);
      for ( j = 0; j < NB_SAMPLE_COLOR_LENGTH; j++)
	colorLengthSample[i][j] = (float)j / (float)(NB_SAMPLE_COLOR_LENGTH - 1);
    }

  pointerToPairExtension_wire = extension;
  return extension;
}

static void changeLightForWire()
{
  glDisable(GL_LIGHTING);
  glDisable(GL_DITHER);
}
static void restoreLightAfterWire()
{
  glEnable(GL_DITHER); /* WARNING: it is the default! */
  glEnable(GL_LIGHTING);
}

gboolean wireSet_stipple(VisuPairData *data, guint16 stipple)
{
  guint16 *stipple_;

  g_return_val_if_fail(data, FALSE);

  DBG_fprintf(stderr, "Pairs Wire: set the wire pattern to %d.\n", stipple);
  stipple_ = (guint16*)visuPairGet_linkProperty(data, "stipple");
  if (!stipple_)
    {
      stipple_ = g_malloc(sizeof(guint16));
      visuPairSet_linkProperty(data, "stipple", (gpointer)stipple_);
      *stipple_ = stipple;
      visuPairSet_outOfDate();
      return TRUE;
    }
  else
    if (*stipple_ != stipple)
      {
	*stipple_ = stipple;
	visuPairSet_outOfDate();
	return TRUE;
      }

  return FALSE;
}
guint16 wireGet_stipple(VisuPairData *data)
{
  guint16 *stipple_;

  g_return_val_if_fail(data, (guint16)1);

  stipple_ = visuPairGet_linkProperty(data, "stipple");
  if (stipple_)
    return *stipple_;
  else
    return (guint16)65535;
}
int setWireGeneralWidth(int val)
{
  DBG_fprintf(stderr, "Pairs Wire: set the wire thickness to %d.\n", val);
  if (val <= 0 || val > RESOURCES_WIRE_WIDTH_MAX || val == wireWidth)
    return 0;

  visuPairSet_outOfDate();
  wireWidth = val;
  return 1;
}
int getWireGeneralWidth()
{
  return wireWidth;
}
int setWireWidth(VisuPairData *data, int val)
{
  int *width;

  if (!data)
    return 0;

  width = visuPairGet_linkProperty(data, "width");
  if (!width)
    {
      width = g_malloc(sizeof(int));
      visuPairSet_linkProperty(data, "width", (gpointer)width);
      *width = val;
      visuPairSet_outOfDate();
      return 1;
    }
  else
    if (*width != val)
      {
	*width = val;
	visuPairSet_outOfDate();
	return 1;
      }

  return 0;
}
int getWireWidth(VisuPairData *data)
{
  int *width;

  if (!data)
    return 0;

  width = visuPairGet_linkProperty(data, "width");
  if (width)
    return *width;
  else
    return wireWidth;
}
int setWireNonLinear(int val)
{
  DBG_fprintf(stderr, "Pairs Wire : set the wire color dependent of wire length to %d.\n", val);
  if (val == wireNonLinear)
    return 0;

  visuPairSet_outOfDate();
  wireNonLinear = val;
  return 1;
}
int getWireNonLinear()
{
  return wireNonLinear;
}
float* getLengthSampleForColor(int color, int *size)
{
  g_return_val_if_fail(color >= 0 && color < 3 && size, (float*)0);
  
  *size = NB_SAMPLE_COLOR_LENGTH;
  return colorLengthSample[color];
}

/*****************************************/
/* Dealing with parameters and resources */
/*****************************************/

static gboolean readLinkInteger(gchar **lines, int nbLines,
				int position, GError **error,
				VisuPairData **data, int *value)
{
  gchar **tokens;
  int id;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit_set(lines[0], " \n", MAX_LINE_LENGTH);
  id = 0;
  if (!visuPairRead_linkFromTokens(tokens, &id, data, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  /* Read the associated integer. */
  if (!configFileRead_integerFromTokens(tokens, &id, value, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  g_strfreev(tokens);

  return TRUE;
}
static gboolean readLinkStipple(gchar **lines, int nbLines, int position,
				VisuData *dataObj _U_, GError **error)
{
  int value;
  VisuPairData *data;

  if (!readLinkInteger(lines, nbLines, position, error, &data, &value))
    return FALSE;

  wireSet_stipple(data, (guint16)value);
  return TRUE;
}
static gboolean readLinkWidth(gchar **lines, int nbLines, int position,
			      VisuData *dataObj _U_, GError **error)
{
  int width;
  VisuPairData *data;
  
  if (!readLinkInteger(lines, nbLines, position, error, &data, &width))
    return FALSE;

  /* Check. */
  if (width <= 0 || width > RESOURCES_WIRE_WIDTH_MAX)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, RESOURCES_WIRE_WIDTH_MAX);
      return FALSE;
    }
  setWireWidth(data, width);
  return TRUE;
}
/* This function details how to read what is needed by PairsData. */
static gboolean readWireGeneralWidth(gchar **lines, int nbLines, int position,
				     VisuData *dataObj _U_, GError **error)
{
  int width;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_integer(lines[0], position, &width, 1, error))
    return FALSE;
  if (width <= 0 || width > RESOURCES_WIRE_WIDTH_MAX)
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: width must be in %d-%d.\n"),
			   position, 0, RESOURCES_WIRE_WIDTH_MAX);
      return FALSE;
    }
  setWireGeneralWidth(width);
  return TRUE;
}
/* This function details how to read the parameter FLAG_PARAMETER_PAIRS. */
static gboolean readWireNonLinear(gchar **lines, int nbLines, int position,
				  VisuData *dataObj _U_, GError **error)
{
  gboolean val;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  setWireNonLinear(val);
  return TRUE;
}
/* This function writes the parameter. */
static void exportParametersWire(GString *data, VisuData *dataObj _U_)
{
  g_string_append_printf(data, "# %s\n", DESC_PARAMETER_WIRE);
  g_string_append_printf(data, "%s: %i\n\n", FLAG_PARAMETER_WIRE, wireNonLinear);
}
void exportPairsWidth(VisuElement *ele1, VisuElement *ele2,
		      VisuPairData *data, gpointer userData)
{
  struct foreachFuncExport_struct *str;
  int *width;
  guint16 *stipple;
  gboolean output;

  width = (int*)visuPairGet_linkProperty(data, "width");
  if (width)
    {
      str = ( struct foreachFuncExport_struct*)userData;
      output = TRUE;
      /* We export the resource only if the elements are
	 part of the given VisuData. */
      if (str->dataObj)
	{
	  /* We test the first element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele1))
	    output = FALSE;
	  /* We test the second element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele2))
	    output = FALSE;
	}
      if (output)
	g_string_append_printf(str->data,
			       "%s:\n"
			       "    %s %s  %4.3f %4.3f  %d\n",
			       FLAG_RESOURCES_LINK_WIDTH,
			       ele1->name, ele2->name, data->minMax[PAIRS_MIN],
			       data->minMax[PAIRS_MAX], *width);
    }
  stipple = (guint16*)visuPairGet_linkProperty(data, "stipple");
  if (stipple)
    {
      str = ( struct foreachFuncExport_struct*)userData;
      output = TRUE;
      /* We export the resource only if the elements are
	 part of the given VisuData. */
      if (str->dataObj)
	{
	  /* We test the first element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele1))
	    output = FALSE;
	  /* We test the second element. */
	  if (!g_hash_table_lookup(str->dataObj->fromVisuElementToInt, ele2))
	    output = FALSE;
	}
      if (output)
	g_string_append_printf(str->data,
			       "%s:\n"
			       "    %s %s  %4.3f %4.3f  %d\n",
			       FLAG_RESOURCES_LINK_STIPPLE,
			       ele1->name, ele2->name, data->minMax[PAIRS_MIN],
			       data->minMax[PAIRS_MAX], *stipple);
    }
}
static void exportResourcesWire(GString *data, VisuData *dataObj)
{
  struct foreachFuncExport_struct str;

  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_WIRE_WIDTH);
  g_string_append_printf(data, "%s:\n    %d\n", FLAG_RESOURCES_WIRE_WIDTH, wireWidth);
  str.data           = data;
  str.dataObj        = dataObj;
  g_string_append_printf(data, "# %s\n", DESC_RESOURCES_LINK_WIDTH);
  visuPairForeach(exportPairsWidth, &str);
  g_string_append_printf(data, "\n");
}



/* OBSOLETE function, kept for backward compatibility. */
static gboolean readWireWidth(gchar **lines, int nbLines, int position _U_,
			      VisuData *dataObj _U_, GError **error)
{
  int token;
  VisuElement* ele[2];
  float radius;
  int val;
  VisuPairData *data;
  gchar **tokens;

  g_return_val_if_fail(nbLines == 1, FALSE);

  /* Tokenize the line of values. */
  tokens = g_strsplit_set(g_strchug(lines[0]), " \n", MAX_LINE_LENGTH);
  token = 0;

  /* Get the two elements. */
  if (!configFileRead_elementFromTokens(tokens, &token, ele, 2, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }

  data = visuPairGet_linkFromId(ele[0], ele[1], 0);
  g_return_val_if_fail(data, FALSE);

  /* Read 1 int. */
  if (!configFileRead_floatFromTokens(tokens, &token, &radius, 1, nbLines, error))
    {
      g_strfreev(tokens);
      return FALSE;
    }
  val = CLAMP((int)radius, 0, RESOURCES_WIRE_WIDTH_MAX);
  setWireWidth(data, val);
  
  g_strfreev(tokens);
  return TRUE;
}
