/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "visu_basic.h"
#include "visu_object.h"
#include "visu_rendering.h"
#include "visu_extension.h"
#include "visu_data.h"
#include "visu_elements.h"
#include "visu_commandLine.h"
#include "opengl.h"
#include "visu_dump.h"
#include "visu_pairs.h"
#include "visu_tools.h"
#include "visu_plugins.h"
#include "visu_configFile.h"
#include "visu_pickMesure.h"
#include "extraFunctions/dataFile.h"
#include "extraFunctions/plane.h"
#include "extraFunctions/surfaces.h"
#include "extraFunctions/pot2surf.h"
#include "extraFunctions/scalarFields.h"
#include "openGLFunctions/objectList.h"
#include "OSOpenGL/visu_openGL.h"
#include "extensions/fogAndBGColor.h"

#include "renderingMethods/renderingAtomic.h"
#include "renderingMethods/renderingSpin.h"

#include <stdio.h>
#include <locale.h>
#include <unistd.h> /* For the access markers R_OK, W_OK ... */

/* For the GdkPixbuf, to be removed later. */
#include <gtk/gtk.h>

/**
 * SECTION:visu_basic
 * @short_description: Main functions of V_Sim (except graphical
 * ones).
 *
 * <para>There are here the main functions of V_Sim (except for
 * graphical methods) such as open file.</para>
 */

/* Default value for the opengl window size. */
#define x_gl 650
#define y_gl 50
#define size_gl 600

/* Valeurs par dfaut pour les chemins utiles dans V_Sim. */
#if SYSTEM_X11 == 1
#define V_SIM_DATA_DIR_DEFAULT    "/usr/local/share/v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "/usr/local/share/v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "/usr/local/share/v_sim/pixmaps"
#define V_SIM_PLUGINS_DIR_DEFAULT "/usr/local/lib/v_sim/plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "/usr/local/share/locale"
#endif
#if SYSTEM_WIN32 == 1
#define V_SIM_DATA_DIR_DEFAULT    "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim"
#define V_SIM_LEGAL_DIR_DEFAULT   "C:\\PROGRAM FILES\\V_Sim\\share\\doc\\v_sim"
#define V_SIM_PIXMAPS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\share\\v_sim\\pixmaps"
#define V_SIM_PLUGINS_DIR_DEFAULT "C:\\PROGRAM FILES\\V_Sim\\lib\\v_sim\\plug-ins"
#define V_SIM_LOCALE_DIR_DEFAULT  "C:\\PROGRAM FILES\\V_Sim\\share\\locale"
#endif
static gchar *v_sim_data_dir;
static gchar *v_sim_legal_dir;
static gchar *v_sim_pixmaps_dir;
static gchar *v_sim_local_conf_dir;
static gchar *v_sim_old_local_conf_dir;
static gchar *v_sim_plugins_dir;
static gchar *v_sim_locale_dir;


/* Main loop if gtk is not used. */
static GMainLoop *gLoop = (GMainLoop*)0;

/* Local methods. */
static gboolean commandLineLoadWithoutGtk(gpointer data);

gboolean visuBasicLoad_dataFromFile(VisuData *data, FileFormat *format,
				    int nSet, GError **error)
{
  RenderingMethod *method;
  gboolean res;

  g_return_val_if_fail(error && *error == (GError*)0, FALSE);

  method = getRenderingMethodInUse();
  if (!method)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			   _("No rendering method is selected.\n"));
      return FALSE;
    }

  if (!method->loadMethod)
    {
      *error = g_error_new(VISU_ERROR_RENDERING, RENDERING_ERROR_METHOD,
			   _("No load method exists for the"
			     " specified rendering method.\n"));
      return FALSE;
    }

  res = method->loadMethod(data, format, nSet, error);
  /* Set flags... */
  if (res)
    {
      visuDataSet_setId(data, nSet);
      visuDataSet_changeElementFlag(data, FALSE);
    }

  DBG_fprintf(stderr, "Visu Basic : emitting 'dataLoaded' signal with %p data argument (%d).\n",
	      (gpointer)data, res);
  if (res)
    g_signal_emit(VISU_INSTANCE, VISU_GET_CLASS(VISU_INSTANCE)->dataLoaded_signal_id,
		  0 /* details */, (gpointer)data, NULL);
  else
    g_signal_emit(VISU_INSTANCE, VISU_GET_CLASS(VISU_INSTANCE)->dataLoaded_signal_id,
		  0 /* details */, (gpointer)0, NULL);
  return res;
}


void visuBasicInit()
{
  int res;

  DBG_fprintf(stderr,"--- Initialising variables ---\n");
  /* We want to read . as floating point separator : in french it is , */
  setlocale(LC_NUMERIC, "C");

  /* Initialise le stockage des couleurs */
  initColorStorage();

  /* initialise la gestion des resources et des param�tres. */
  DBG_fprintf(stderr,"--- Initialising the VisuConfigFile part ---\n");
  res = visuConfigFile_init();
  if (!res)
    exit(1);

  /* initialise les fonctions d'export. */
  DBG_fprintf(stderr,"--- Initialising dump modules ---\n");
  res = visuDumpInit();
  if (!res)
    exit(1);

  /* initialise OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL ---\n");
  res = initOpengl();
  if (!res)
    exit(1);

  /* initialise les ajouts OpenGL. */
  DBG_fprintf(stderr,"--- Initialising OpenGL extensions ---\n");
  res = initOpenGLExtensions();
  if (!res)
    exit(1);

  /* initialise les mthodes de rendu. */
  DBG_fprintf(stderr,"--- Initialising rendering methods ---\n");
  fileFormatInit();
  res = initRenderingMethods();
  if (!res)
    exit(1);

  /* initialise the VisuElement part. */
  DBG_fprintf(stderr,"--- Initialising the elements and data part of visu ---\n");
  res = initVisuElements();
  if (!res)
    exit(1);

  /* initialise la gestion des liaisons. */
  DBG_fprintf(stderr,"--- Initialising pairs modules ---\n");
  res = initPairsModule();
  if (!res)
    exit(1);

  /* initialise la gestion des sessions pick. */
  DBG_fprintf(stderr,"--- Initialising pick module ---\n");
  initPick_module();

  /* initialise la gestion de la couleur des �l�ments par un fichier ext�rieur. */
  DBG_fprintf(stderr,"--- Initialising dataFile modules ---\n");
  res = initDataFileModule();
  if (!res)
    exit(1);

  /* initialise the scalar field handler. */
  DBG_fprintf(stderr,"--- Initialising scalar field module ---\n");
  scalarFieldInit();

  /* Charge toutes les extensions OpenGL. */
  DBG_fprintf(stderr,"--- Initialising external OpenGL Extensions ---\n");
  loadExtensions();

  /* Charge les plugins. */
  DBG_fprintf(stderr,"--- Initialising plugins ---\n");
  visuPluginsInit();

  /* Initialise extra functions. */
  isosurfacesInit();
}


int visuBasicExport_main(void)
{
  char* path;
  gboolean res;
  char* filename, *spin_filename, *resFile;
  VisuData *newData;
  GError *error;

  gLoop = g_main_loop_new(NULL, FALSE);
  g_type_init();

  /* Force the creation of the main object class. */
  g_type_class_ref(visu_get_type());

  visuBasicInit();

/*   DBG_fprintf(stderr,"--- Attach the pixmap context ---\n"); */
/*   commandLineGet_XWindowGeometry(&width, &height); */

  DBG_fprintf(stderr,"--- Initialising the parameters ---\n");
  /* Look for parameters file */
  path = visuConfigFileGet_validPath(VISU_CONFIGFILE_PARAMETER, R_OK, 0);
  if (path)
    {
      error = (GError*)0;
      res = visuConfigFileLoad(VISU_CONFIGFILE_PARAMETER, path, (VisuData*)0, &error);
      if (error)
	{
	  g_warning("%s", error->message);
	  g_error_free(error);
	}
      g_free(path);
    }
  else
    g_warning("Unable to find a valid parameters file."
	      " Starting with defaults.\n");

  DBG_fprintf(stderr,"--- Initialising resources ---\n");
  /* Look for resources file */
  resFile = commandLineGet_resourcesFile();
  if (!resFile)
    path = visuConfigFileGet_validPath(VISU_CONFIGFILE_RESOURCE, R_OK, 0);
  else
    path = g_strdup(resFile);
  if (path)
    {
      error = (GError*)0;
      res = visuConfigFileLoad(VISU_CONFIGFILE_RESOURCE, path, (VisuData*)0, &error);
      if (error)
	{
	  g_warning("%s", error->message);
	  g_error_free(error);
	}
      g_free(path);
    }

  newData = visuDataNew();
  if (!newData)
    return 1;

  filename = commandLineGet_ArgFilename();
  spin_filename = commandLineGet_ArgSpinFileName();
  if (filename && !spin_filename)
    {
      setRenderingMethodInUse(renderingAtomicGet_static());

      visuDataAdd_file(newData, filename, FILE_KIND_POSITION, (FileFormat*)0);
      g_free(filename);
      DBG_fprintf(stderr, "Visu Main : adding the load call back in the queue.\n");
      g_idle_add(commandLineLoadWithoutGtk, (gpointer)newData);
    }
  else if(filename && spin_filename)
    {
      setRenderingMethodInUse(renderingSpinGet_static());

      visuDataAdd_file(newData, filename, FILE_KIND_POSITION, (FileFormat*)0);
      visuDataAdd_file(newData, spin_filename, FILE_KIND_SPIN, (FileFormat*)0);
      g_free(filename);
      g_free(spin_filename);
      
      DBG_fprintf(stderr, "Visu Main : adding the load call back in the queue.\n");
      g_idle_add(commandLineLoadWithoutGtk, (gpointer)newData);
    }
  else
    g_error(_("a file to render is mandatory with the '--export' option."));
  
  g_main_loop_run(gLoop);
  return 0;
}

gboolean visuBasicQuit(gpointer data _U_)
{
  if (gLoop)
    g_main_loop_quit(gLoop);

  return FALSE;
}

static Plane **planesList;
static int planesId, surfId, mapId;
static gboolean logScale;
static guint nIsolines;
static Surfaces **surf;
static Shade *shade;
static GList *fields;
static int mapPlaneId;

static void setBgImage(const gchar *filename)
{
  GError *error;
  GdkPixbuf *pixbuf;
  gchar *title;

  DBG_fprintf(stderr, "Visu Basic: set the background image to '%s'.\n",
	      filename);
  error = (GError*)0;
  pixbuf = gdk_pixbuf_new_from_file(filename, &error);
  title = g_path_get_basename(filename);

  if (!pixbuf)
    {
      g_warning(error->message);
      g_error_free(error);
      g_free(title);
      return;
    }
  bgSet_image(gdk_pixbuf_get_pixels(pixbuf),
	      gdk_pixbuf_get_width(pixbuf),
	      gdk_pixbuf_get_height(pixbuf),
	      gdk_pixbuf_get_has_alpha(pixbuf),
	      title);
  g_object_unref(pixbuf);
  g_free(title);
}

static void rebuildPlanes(VisuData *dataObj)
{
  OpenGLView *view;

  view = visuDataGet_openGLView(dataObj);
  /* Create the plane list. */
  glNewList(planesId, GL_COMPILE);
  glEnable(GL_BLEND);
  glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);
  glDisable(GL_LIGHTING);
  glDisable(GL_DITHER);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);
  planesDraw_list(planesList);
  glPopMatrix();  
  glEnable(GL_LIGHTING);
  glEnable(GL_DITHER);
  glEndList();
}
static void rebuildSurf(VisuData *dataObj)
{
  SurfacesOrder *surfaceOrder;

  /* Create the surface list. */
  surfaceOrder = isosurfacesOrder_new();
  isosurfacesDraw_surfaces(surfId + 1, visuDataGet_openGLView(dataObj),
			   surf, surfaceOrder);
  isosurfacesDuplicate(surfId, surfId + 1, dataObj, TRUE);
  isosurfacesOrder_free(surfaceOrder);
}
static void rebuildMap(VisuData *dataObj)
{
  float minMax[2];

  /* Create the map list. */
  glNewList(mapId, GL_COMPILE);
  scalarFieldDraw_map((ScalarField*)fields->data, planesList[mapPlaneId],
		      shade, visuDataGet_openGLView(dataObj),
                      100., minMax, logScale, nIsolines);
  glEndList();
}

static gboolean commandLineLoadWithoutGtk(gpointer data)
{
  gchar *colorFile, *planesFile, *surfFile, *fieldFile;
  GString *message;
  int errors, i, presetShade, nb, surfNb;
  gboolean somethingIsLoaded, res, new;
  int *colUsed;
  float *translations, *extension;
  float *values;
  gchar **names;
  VisuData *visuData;
  guint width, height;
  GError *error;
  OpenGLExtension *plane, *surfExt, *map;
  GList *tmplst, *list;
  OptionTable *table;

  visuData = VISU_DATA(data);
  g_return_val_if_fail(visuData, FALSE);

  visuBasicLoad_fileWithoutGtk(visuData);

  surfNb = 0;

  commandLineGet_XWindowGeometry((int*)&width, (int*)&height);
  visuDataSet_sizeOfView(data, width, height);

  /* Post loading sequence, corresponding to all other arguments given to command line. */
  /* translate argument */
  translations = commandLineGet_translation();
  if (translations)
    {
      visuDataSet_XYZtranslation(visuData, translations);
      visuData_constrainedInTheBox(visuData);
    }

  DBG_fprintf(stderr, "Visu basic : emitting the 'dataReadyForRendering' signal.\n");
  g_signal_emit(VISU_INSTANCE,
		VISU_GET_CLASS(VISU_INSTANCE)->dataReadyForRendering_signal_id,
		0 /* details */, (gpointer)visuData, NULL);
  DBG_fprintf(stderr, "Visu data : emitting done.\n");

  /* Post loading sequence, corresponding to all other arguments given to command line. */
  /* The shade option. */
  shade = (Shade*)0;
  presetShade = commandLineGet_presetColor();
  list        = toolShadeBuild_presetList();
  if (presetShade >= 0)
    {
      if (presetShade >= (int)g_list_length(list))
	{
	  g_warning(_("unknown shade id (max is %d)."), g_list_length(list) - 1);
	  shade = (Shade*)g_list_nth_data(list, 0);
	}
      else
	shade = (Shade*)g_list_nth_data(list, presetShade);
      DBG_fprintf(stderr, "Visu Basic: load a shade from command line %p.\n",
		  (gpointer)shade);
    }
  else
    shade = (Shade*)g_list_nth_data(list, 0);
  /* colorize argument */
  colorFile = commandLineGet_colorizeFileName();
  if (colorFile)
    {
      DBG_fprintf(stderr, "Visu basic : loading a new data file '%s'.\n", colorFile);
      message = g_string_new(_("Loading data file...\n"));
      somethingIsLoaded = dataFileSet_file(visuData, colorFile, &new,
					   message, &errors);

      /* Print errors if necessary. */
      if (errors)
	fprintf(stderr, "%s\n", message->str);
      g_string_free(message, TRUE);

      if (somethingIsLoaded)
	{
	  dataFileSet_shade(visuData, shade);
	  colUsed = commandLineGet_colorizeColUsed();
	  if (colUsed)
	    for (i = 0; i < 3; i++)
	      dataFileSet_colUsed(visuData, colUsed[i] - 1, i);
	}
    }
  /* expand argument */
  extension = commandLineGet_extension();
  if (extension)
    visuData_replicate(visuData, extension, (gboolean*)0);
  /* Planes argument. */
  planesList = (Plane**)0;
  planesFile = commandLineGet_planesFileName();
  if (planesFile)
    {
      error = (GError*)0;
      somethingIsLoaded = planesParse_XMLFile(planesFile, &planesList, &error);
      if (!somethingIsLoaded)
	{
	  if (error)
	    {
	      g_warning(error->message);
	      g_clear_error(&error);
	    }
	}
      else
	{
	  planesComputeInter_list(visuDataGet_openGLView(visuData), planesList);
	  planesId = openGLObjectList_new(1);
	  plane = OpenGLExtension_new("Planes", _("Planes"), (char*)0,
				      planesId, rebuildPlanes);
	  plane->used = 1;
	  OpenGLExtensionSet_priority(plane, OPENGL_EXTENSION_PRIORITY_LOW + 1);
	  OpenGLExtensionRegister(plane);
	  /* Hide nodes if required. */
	  res = planeShowHide_all(visuData, planesList);
	  if (res)
	    visuDataEmit_nodeRenderedChange(visuData);
	}
    }
  /* Iso-surfaces arguments. */
  surfFile = commandLineGet_isoSurfacesFileName();
  if (surfFile)
    {
      error = (GError*)0;
      surfNb = 1;
      surf = g_malloc(sizeof(Surfaces*) * (nb + 1));
      somethingIsLoaded = isosurfacesLoad_file(surfFile, surf, &error);
      if (!somethingIsLoaded)
	{
	  if (error)
	    {
	      g_warning(error->message);
	      g_clear_error(&error);
	    }
	  surfNb = 0;
	}
      else if (commandLineGet_fitToBox())
	isoSurfacesSet_fitToBox(visuData, surf[0]);
      isosurfacesSet_showAll(surf[0], TRUE);
      surf[surfNb] = (Surfaces*)0;
    }
  /* Scalar-field arguments. */
  fields = (GList*)0;
  fieldFile = commandLineGet_scalarFieldFileName();
  if (fieldFile)
    {
      error = (GError*)0;
      /* Create an option table for possible spin or complex
	 modifiers. */
      table = commandLineGet_options();
      /* Call the load method. */
      somethingIsLoaded = scalarFieldLoad_fromFile(fieldFile, &fields, &error, table);
      if (!somethingIsLoaded)
	{
	  if (error)
	    {
	      g_warning(error->message);
	      g_clear_error(&error);
	    }
	  surfNb = 0;
	}
      else
	{
	  if (commandLineGet_fitToBox())
	    {
	      tmplst = fields;
	      while(tmplst)
		{
		  scalarFieldSet_fitToBox(visuData, (ScalarField*)tmplst->data);
		  tmplst = g_list_next(tmplst);
		}
	    }

	  nb     = g_list_length(fields);
	  surf   = g_malloc(sizeof(Surfaces*) * (nb + 1));
	  surfNb = 0;
	  values = commandLineGet_isoValues(&nb);
	  names  = commandLineGet_isoNames(&nb);
	  if (values)
	    {
	      tmplst = fields;
	      while(tmplst)
		{
		  surf[0] = (Surfaces*)0;
		  for (i = 0; i < nb; i++)
		    res = pot2surfCreate(surf, (ScalarField*)tmplst->data, values[i], i, names[i]);
		  isosurfacesSet_showAll(surf[surfNb++], TRUE);
		  tmplst = g_list_next(tmplst);
		}
	    }
	  surf[surfNb] = (Surfaces*)0;
	}
    }
  if (surfNb > 0)
    {
      /* We apply the masking effect of planes, if necessary. */
      if (planesList)
	for (i = 0; surf[i]; i++)
	  isosurfacesHide(surf[i], planesList);
      /* We create the surface extension. */
      surfId = openGLObjectList_new(2);
      surfExt = OpenGLExtension_new("Isosurfaces", _("Isosurfaces"), (char*)0,
				    surfId, rebuildSurf);
      surfExt->used = 1;
      OpenGLExtensionSet_sensitiveToRenderingMode(surfExt, TRUE);
      OpenGLExtensionSet_saveOpenGLState(surfExt, TRUE);
      OpenGLExtensionSet_priority(surfExt, OPENGL_EXTENSION_PRIORITY_LOW + 2); 
      OpenGLExtensionRegister(surfExt);
    }
  /* The coloured map argument. */
  if (commandLineGet_coloredMap(&mapPlaneId))
    {
      if (!planesList)
	g_warning(_("option '--build-map' has been given but"
		    " no plane is available (use '--planes')."));
      if (!fields)
	g_warning(_("option '--build-map' has been given but"
		    " no scalar field is available (use '--scalar-field')."));
      if (!shade)
	g_warning(_("option '--build-map' has been given but"
		    " no shade is available (use '--color-preset')."));
      logScale = commandLineGet_logScale();
      nIsolines = commandLineGet_nIsoLines();
      if (planesList && fields && shade)
	{
	  /* We change the visibility of the selected plane. */
	  for (i = 0; planesList[i]; i++)
	    if (i == mapPlaneId)
	      {
		planeSet_rendered(planesList[i], FALSE);
		break;
	      }
	  if (!planesList[i])
	    g_warning(_("argument 'id' of option '--build-map' is out of"
			" bounds (max value is %d)."), i - 1);
	  else
	    {
	      /* We create the map extension. */
	      mapId = openGLObjectList_new(1);
	      map = OpenGLExtension_new("MapLegend", _("Coloured map"), (char*)0,
					mapId, rebuildMap);
	      map->used = 1;
	      OpenGLExtensionSet_sensitiveToRenderingMode(map, TRUE);
	      OpenGLExtensionRegister(map);
	    }
	}
    }
  /* The bg image. */
  if (commandLineGet_bgImage())
    setBgImage(commandLineGet_bgImage());

  visuBasicDump(visuData);

  /* Free allocated memory. */
  if (fields)
    {
      tmplst = fields;
      while(tmplst)
	{
	  scalarFieldFree((ScalarField*)tmplst->data);
	  tmplst = g_list_next(tmplst);
	}
	g_list_free(fields);
    }
  if (planesList)
    {
      for (i = 0; planesList[i]; i++)
	g_object_unref(G_OBJECT(planesList[i]));
      g_free(planesList);
    }

  /* Finaly quit. */
  g_idle_add(visuBasicQuit, (gpointer)gLoop);

  return FALSE;
}

gboolean visuBasicDump(VisuData *data)
{
  DumpType *format;
  GList *pnt;
  char *exportFileName;
  DumpImage *dumpData;
  int width, height;
  guchar* image;
  GError *error;
  gboolean status;

  exportFileName = (char*)commandLineGet_ExportFileName();
  if (!exportFileName)
    {
      g_error("This method should be called with"
	      " an argument that is the file name to export to.\n");
    }
  
  pnt = visuDumpGet_allModules();
  while (pnt && !fileFormatUse_match( ((DumpType*)pnt->data)->fileType,
				     exportFileName))
    pnt = g_list_next(pnt);
  if (!pnt)
    {
      g_warning("The format can't be found from the"
		" filename '%s' entered.\n", exportFileName);
      return FALSE;
    }
  format = (DumpType*)pnt->data;

  DBG_fprintf(stderr, "Visu Basic : Preparing to dump the image to '%s' as"
	      " format '%s'.\n", exportFileName, format->fileType->description);

  commandLineGet_XWindowGeometry((int*)&width, (int*)&height);

  dumpData = visuOpenGLNew_pixmapContext((guint)width, (guint)height);
  if (!dumpData)
    {
      g_error("can't create off-screen rendering. Command line"
	      " exportation is not available.");
    }
  /* We set the glViewport of this new context. */
  glViewport(0, 0, width, height);
  /* We call the given draw method. */
  openGLInit_context();
  visuDataSet_sizeOfView(data, (guint)width, (guint)height);
  openGLViewCompute_matrixAndView(visuDataGet_openGLView(data)); 
  rebuildAllExtensionsLists(data);
  if (format->bitmap)
    {
      openGL_reDraw((const char**)0, (gpointer)0);
      /* We copy the pixmap into generic data. */
      image = visuOpenGLGet_pixmapData((guint)width, (guint)height, format->hasAlpha);
    }
  else
    image = (guchar*)0;

  error = (GError*)0;
  if (!format->writeFunc(format->fileType, exportFileName, width, height,
			 data, image, &error, (voidDataFunc)0, (gpointer)0) && error)
    {
      g_warning(error->message);
      g_error_free(error);
      status = FALSE;
    }
  else
    status = TRUE;
  
  if (image)
    g_free(image);
  /* We free the pixmap context. */
  visuOpenGLFree_pixmapContext(dumpData);

  return status;
}

void visuBasicLoad_fileWithoutGtk(VisuData *data)
{
  GError *error;
  int res;

  if (!data)
    return;

  DBG_fprintf(stderr, "Visu Basic: Loading data.\n");

  error = (GError*)0;
  res = visuBasicLoad_dataFromFile(data, (FileFormat*)0, 0, &error);

  if (!res)
    {
      g_object_unref(data);
      g_warning(error->message);
      g_error_free(error);
      exit(1);
    }

  if (error)
    g_error_free(error);
}

void visuBasicSet_paths(const char *exeLocation _U_)
{
#if SYSTEM_WIN32 == 1
  #define V_SIM_INI_FILE PACKAGE".ini"
  GIOChannel *iniFile;
  gchar *iniPath;
  gchar *buffer, *line;
  GIOStatus res;
  GError *err;
  gchar **tokens;
  gsize length;

  v_sim_data_dir    = g_strdup(V_SIM_DATA_DIR_DEFAULT);
  v_sim_legal_dir   = g_strdup(V_SIM_LEGAL_DIR_DEFAULT);
  v_sim_pixmaps_dir = g_strdup(V_SIM_PIXMAPS_DIR_DEFAULT);
  v_sim_plugins_dir = g_strdup(V_SIM_PLUGINS_DIR_DEFAULT);
  v_sim_locale_dir  = g_strdup(V_SIM_LOCALE_DIR_DEFAULT);
  iniPath = g_strdup(V_SIM_INI_FILE);
  iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
  if (!iniFile)
    {
      iniPath = g_build_filename("C:\\WINDOWS", V_SIM_INI_FILE, NULL);
      iniFile = g_io_channel_new_file(iniPath, "r", (GError**)0);
    }
  if (iniFile)
    {
      buffer = (gchar*)0;
      err = (GError*)0;
      do
	{
	  res = g_io_channel_read_line(iniFile, &line, &length, NULL, &err);
	  if (line && res == G_IO_STATUS_NORMAL)
	    {
	      tokens = g_strsplit(line, "=", 2);
	      if (!strcmp(g_strstrip(tokens[0]), "data_dir") && tokens[1])
		{
		  g_free(v_sim_data_dir);
		  v_sim_data_dir = g_strdup(g_strstrip(tokens[1]));
		}
	      if (!strcmp(g_strstrip(tokens[0]), "legal_dir") && tokens[1])
		{
		  g_free(v_sim_legal_dir);
		  v_sim_legal_dir = g_strdup(g_strstrip(tokens[1]));
		}
	      if (!strcmp(g_strstrip(tokens[0]), "pixmaps_dir") && tokens[1])
		{
		  g_free(v_sim_pixmaps_dir);
		  v_sim_pixmaps_dir = g_strdup(g_strstrip(tokens[1]));
		}
	      if (!strcmp(g_strstrip(tokens[0]), "plugins_dir") && tokens[1])
		{
		  g_free(v_sim_plugins_dir);
		  v_sim_plugins_dir = g_strdup(g_strstrip(tokens[1]));
		}
	      if (!strcmp(g_strstrip(tokens[0]), "locale_dir") && tokens[1])
		{
		  g_free(v_sim_locale_dir);
		  v_sim_locale_dir = g_strdup(g_strstrip(tokens[1]));
		}
	      g_strfreev(tokens);
	      g_free(line);
	    }
	}
      while (res != G_IO_STATUS_EOF);
      g_io_channel_shutdown (iniFile, FALSE, (GError**)0);
      g_io_channel_unref (iniFile);
    }
  g_free(iniPath);
#endif
#if SYSTEM_X11 == 1
  v_sim_data_dir    = g_strdup(DATA_DIR);
  v_sim_legal_dir   = g_strdup(LEGAL_DIR);
  v_sim_pixmaps_dir = g_strdup(PIXMAPS_DIR);
  v_sim_plugins_dir = g_strdup(PLUGINS_DIR);
  v_sim_locale_dir  = g_strdup(LOCALE_DIR);
#endif

  /* Create the local dirs. */
#if GLIB_MINOR_VERSION > 5
  v_sim_local_conf_dir = g_build_filename(g_get_user_config_dir(), "v_sim", NULL);
#else
  v_sim_local_conf_dir = g_build_filename(g_get_home_dir(), ".config/v_sim", NULL);
#endif
  if (!v_sim_local_conf_dir)
    g_warning("WARNING! Impossible to get the default"
	      " path $XDG_CONFIG_HOME/v_sim.\n");
  v_sim_old_local_conf_dir = g_build_filename(g_get_home_dir(), ".v_sim", NULL);

  DBG_fprintf(stderr, "Visu Basic: data directory      : '%s'.\n", v_sim_data_dir);
  DBG_fprintf(stderr, "Visu Basic: local conf directory: '%s'.\n", v_sim_local_conf_dir);
  DBG_fprintf(stderr, "Visu Basic: legal directory     : '%s'.\n", v_sim_legal_dir);
  DBG_fprintf(stderr, "Visu Basic: pixmaps directory   : '%s'.\n", v_sim_pixmaps_dir);
  DBG_fprintf(stderr, "Visu Basic: plug-ins directory  : '%s'.\n", v_sim_plugins_dir);
  DBG_fprintf(stderr, "Visu Basic: locale directory    : '%s'.\n", v_sim_locale_dir);
}
const gchar* visuBasicGet_dataDir()
{
  return v_sim_data_dir;
}
const gchar* visuBasicGet_legalDir()
{
  return v_sim_legal_dir;
}
const gchar* visuBasicGet_pixmapsDir()
{
  return v_sim_pixmaps_dir;
}
const gchar* visuBasicGet_localDir()
{
  return v_sim_local_conf_dir;
}
const gchar* visuBasicGet_oldLocalDir()
{
  return v_sim_old_local_conf_dir;
}
const gchar* visuBasicGet_pluginsDir()
{
  return v_sim_plugins_dir;
}
const gchar* visuBasicGet_localeDir()
{
  return v_sim_locale_dir;
}
