/*   EXTRAITS DE LA LICENCE
	Copyright CEA, contributeurs : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)
  
	Adresse ml :
	BILLARD, non joignable par ml ;
	CALISTE, damien P caliste AT cea P fr.

	Ce logiciel est un programme informatique servant  visualiser des
	structures atomiques dans un rendu pseudo-3D. 

	Ce logiciel est rgi par la licence CeCILL soumise au droit franais et
	respectant les principes de diffusion des logiciels libres. Vous pouvez
	utiliser, modifier et/ou redistribuer ce programme sous les conditions
	de la licence CeCILL telle que diffuse par le CEA, le CNRS et l'INRIA 
	sur le site "http://www.cecill.info".

	Le fait que vous puissiez accder  cet en-tte signifie que vous avez 
	pris connaissance de la licence CeCILL, et que vous en avez accept les
	termes (cf. le fichier Documentation/licence.fr.txt fourni avec ce logiciel).
*/

/*   LICENCE SUM UP
	Copyright CEA, contributors : Luc BILLARD et Damien
	CALISTE, laboratoire L_Sim, (2001-2005)

	E-mail address:
	BILLARD, not reachable any more ;
	CALISTE, damien P caliste AT cea P fr.

	This software is a computer program whose purpose is to visualize atomic
	configurations in 3D.

	This software is governed by the CeCILL  license under French law and
	abiding by the rules of distribution of free software.  You can  use, 
	modify and/ or redistribute the software under the terms of the CeCILL
	license as circulated by CEA, CNRS and INRIA at the following URL
	"http://www.cecill.info". 

	The fact that you are presently reading this means that you have had
	knowledge of the CeCILL license and that you accept its terms. You can
	find a copy of this licence shipped with this software at Documentation/licence.en.txt.
*/

#include "scale.h"

#include <GL/gl.h>
#include <GL/glu.h> 

#include <opengl.h>
#include <openGLFunctions/text.h>
#include <visu_object.h>
#include <visu_tools.h>
#include <visu_configFile.h>
#include <visu_extension.h>
#include <openGLFunctions/objectList.h>
#include <coreTools/toolColor.h>
#include <coreTools/toolMatrix.h>
#include <coreTools/toolConfigFile.h>
#include <math.h>

/**
 * SECTION:scale
 * @short_description: Draw an arrow with a label.
 *
 * This little extension is used to draw an arrow at a given position
 * displaying a given length.
 *
 * Since: 3.3
 */

/* Parameters & resources*/
/* This is a boolean to control is the axes is render or not. */
#define FLAG_RESOURCE_SCALE_USED   "scales_are_on"
#define DESC_RESOURCE_SCALE_USED   "Control if scales are drawn ; boolean (0 or 1)"
#define SCALE_USED_DEFAULT 0
static gboolean readScaleIsOn(gchar **lines, int nbLines,
			      int position, VisuData *dataObj, GError **error);

/* A resource to control the color used to render the lines of the Scale. */
#define FLAG_RESOURCE_SCALE_COLOR   "scales_color"
#define DESC_RESOURCE_SCALE_COLOR   "Define the color RGBA of all scales ; four floating point values (0. <= v <= 1.)"
static float scaleRGBDefault[4] = {0.f, 0.f, 0.f, 1.f};
static gboolean readScaleColor(gchar **lines, int nbLines,
			       int position, VisuData *dataObj, GError **error);

/* A resource to control the width to render the lines of the Scale. */
#define FLAG_RESOURCE_SCALE_LINE   "scales_line_width"
#define DESC_RESOURCE_SCALE_LINE   "Define the width of the lines of all scales ; one floating point value (1. <= v <= 10.)"
#define SCALE_WIDTH_DEFAULT 1.
static gboolean readScaleLineWidth(gchar **lines, int nbLines,
				   int position, VisuData *dataObj, GError **error);

/* A resource to control the width to render the lines of the Scale. */
#define FLAG_RESOURCE_SCALE_STIPPLE   "scales_line_stipple"
#define DESC_RESOURCE_SCALE_STIPPLE   "Define the stipple pattern of the lines of all scales ; one integer value (0 <= v <= 65535)"
#define SCALE_STIPPLE_DEFAULT 65535
static gboolean readScaleLineStipple(gchar **lines, int nbLines,
				     int position, VisuData *dataObj, GError **error);

/* A resource to control the elements of a scale (origin, direction, length... */
#define FLAG_RESOURCE_SCALE_DEFINITION "scale_definition"
#define DESC_RESOURCE_SCALE_DEFINITION "Define the position, the direction, the length and the legend of a scale ; position[3] direction[3] length legend"
#define SCALE_LEGEND_DEFAULT _("Length: %6.2f")
static gboolean readScaleDefinition(gchar **lines, int nbLines,
				    int position, VisuData *dataObj, GError **error);

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesScale(GString *data, int *nbLinesWritten,
				     VisuData *dataObj);

#define SCALE_AUTO_LEGEND "[auto]"

struct _Scale
{
  /* Internal object gestion. */
  GObject parent;
  gboolean dispose_has_run;

  /* Definition of the scale. */
  float origin[3];
  float direction[3];
  float length;

  /* Characteristics. */
  gboolean drawn;
  gchar *legendPattern;
  GString *legend;
};

struct _ScaleClass
{
  GObjectClass parent;

  /* The list of available scales. */
  GList *scales;

  /* Some global modifiers. */
  float width;
  float rgba[4];
  guint16 stipple;

  /* The drawing hook. */
  OpenGLExtension* ext;
};

static ScaleClass *my_class = NULL;

/* This flag can be set internally to specify that Scale should be recreated. */
static gboolean scaleHasBeenBuilt;

/* Object gestion methods. */
static void scale_dispose (GObject* obj);
static void scale_finalize(GObject* obj);

/* Local methods. */
static void rebuildScale(VisuData *dataObj);
static void lineDraw(VisuData *dataObj, int nlat, Scale *scale);

/* Local callbacks */
static void onScaleParametersChange(VisuData *dataObj, OpenGLView *view,
				    gpointer data);
static void rebuildScaleOnResourcesLoaded(GObject *visu, VisuData *dataObj,
					  gpointer data);
static void onDataReadySignal(GObject *visu, VisuData *dataObj, gpointer data);
static void onDataNewSignal(GObject *visu, VisuData *dataObj, gpointer data);

G_DEFINE_TYPE(Scale, scale, G_TYPE_OBJECT)

static void scale_class_init(ScaleClass *klass)
{
  char *name = "Scale";
  char *description = _("Draw scales in the rendering area.");
  VisuConfigFileEntry *resourceEntry;

  DBG_fprintf(stderr, "Scale: creating the class of the object.\n");

  /* Connect freeing methods. */
  G_OBJECT_CLASS(klass)->dispose  = scale_dispose;
  G_OBJECT_CLASS(klass)->finalize = scale_finalize;

  /* Create the drawing extension. */
  DBG_fprintf(stderr," - initialising the Scale OpenGL extension.\n");
  klass->ext = OpenGLExtension_new(name, _(name), description,
				   openGLObjectList_new(1), rebuildScale);
  OpenGLExtensionSet_saveOpenGLState(klass->ext, TRUE);
  OpenGLExtensionRegister(klass->ext);

  /* Put the default values. */
  DBG_fprintf(stderr," - set the default values.\n");
  klass->width     = SCALE_WIDTH_DEFAULT;
  klass->stipple   = SCALE_STIPPLE_DEFAULT;
  klass->rgba[0]   = scaleRGBDefault[0];
  klass->rgba[1]   = scaleRGBDefault[1];
  klass->rgba[2]   = scaleRGBDefault[2];
  klass->rgba[3]   = scaleRGBDefault[3];
  klass->ext->used = SCALE_USED_DEFAULT;
  klass->scales    = (GList*)0;

  /* Create the entries in config files. */
  DBG_fprintf(stderr," - create entries for config file.\n");
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_USED,
					  DESC_RESOURCE_SCALE_USED,
					  1, readScaleIsOn);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_COLOR,
					  DESC_RESOURCE_SCALE_COLOR,
					  1, readScaleColor);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_LINE,
					  DESC_RESOURCE_SCALE_LINE,
					  1, readScaleLineWidth);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_DEFINITION,
					  DESC_RESOURCE_SCALE_DEFINITION,
					  1, readScaleDefinition);
  visuConfigFileSet_version(resourceEntry, 3.3f);
  resourceEntry = visuConfigFileAdd_entry(VISU_CONFIGFILE_RESOURCE,
					  FLAG_RESOURCE_SCALE_STIPPLE,
					  DESC_RESOURCE_SCALE_STIPPLE,
					  1, readScaleLineStipple);
  visuConfigFileSet_version(resourceEntry, 3.4f);
  visuConfigFileAdd_exportFunction(VISU_CONFIGFILE_RESOURCE,
				   exportResourcesScale);

  /* Connect external signals. */
  g_signal_connect(VISU_INSTANCE, "dataReadyForRendering",
		   G_CALLBACK(onDataReadySignal), (gpointer)klass);
  g_signal_connect(VISU_INSTANCE, "dataNew",
		   G_CALLBACK(onDataNewSignal), (gpointer)klass);
  g_signal_connect(VISU_INSTANCE, "resourcesLoaded",
 		   G_CALLBACK(rebuildScaleOnResourcesLoaded), (gpointer)klass);

  scaleHasBeenBuilt = FALSE;

  my_class = klass;
}

static void scale_init(Scale *obj)
{
  ScaleClass *klass;

  obj->dispose_has_run = FALSE;

  DBG_fprintf(stderr, "Scale: creating a new scale (%p).\n", (gpointer)obj);
  obj->drawn         = TRUE;
  obj->origin[0]     = 0.;
  obj->origin[1]     = 0.;
  obj->origin[2]     = 0.;
  obj->direction[0]  = 1.;
  obj->direction[1]  = 0.;
  obj->direction[2]  = 0.;
  obj->length        = 5.;
  obj->legendPattern = (gchar*)0;
  obj->legend        = g_string_new("");

  klass = SCALE_GET_CLASS(obj);
  klass->scales = g_list_append(klass->scales, obj);
  DBG_fprintf(stderr, "Scales: list of scales is now %d long.\n",
	      g_list_length(klass->scales));
}

Scale* scaleNew(float origin[3], float orientation[3],
		float length, const gchar *legend)
{
  Scale *scale;

  g_return_val_if_fail(length > 0.f, (Scale*)0);

  scale = SCALE(g_object_new(SCALE_TYPE, NULL));
  g_return_val_if_fail(scale, (Scale*)0);

  scale->origin[0] = origin[0];
  scale->origin[1] = origin[1];
  scale->origin[2] = origin[2];
  scale->direction[0] = orientation[0];
  scale->direction[1] = orientation[1];
  scale->direction[2] = orientation[2];
  scale->length = length;
  if (legend)
    scale->legendPattern = g_strdup(legend);
  if (legend)
    g_string_assign(scale->legend, legend);
  else
    g_string_printf(scale->legend, SCALE_LEGEND_DEFAULT, scale->length);

  return scale;
}

/* This method can be called several times.
   It should unref all of its reference to
   GObjects. */
static void scale_dispose(GObject* obj)
{
  DBG_fprintf(stderr, "Scale: dispose object %p.\n", (gpointer)obj);

  if (SCALE(obj)->dispose_has_run)
    return;

  SCALE(obj)->dispose_has_run = TRUE;
  /* Chain up to the parent class */
  G_OBJECT_CLASS(scale_parent_class)->dispose(obj);
}
/* This method is called once only. */
static void scale_finalize(GObject* obj)
{
  g_return_if_fail(obj);

  DBG_fprintf(stderr, "Scale: finalize object %p.\n", (gpointer)obj);

  /* Remove me from the global scale list. */
  SCALE_GET_CLASS(obj)->scales = g_list_remove(SCALE_GET_CLASS(obj)->scales, obj);
  DBG_fprintf(stderr, "Scales: list of scales is now %d long.\n",
	      g_list_length(SCALE_GET_CLASS(obj)->scales));

  /* Free my memory. */
  if (SCALE(obj)->legendPattern)
    g_free(SCALE(obj)->legendPattern);
  g_string_free(SCALE(obj)->legend, TRUE);

  /* Chain up to the parent class */
  G_OBJECT_CLASS(scale_parent_class)->finalize(obj);
}


static ScaleClass* scalesGetClass()
{
  GObject *scale;

  if (!my_class)
    {
      scale = g_object_new(SCALE_TYPE, NULL);
      g_object_unref(scale);
    }

  return my_class;
}

/* Method used to change the value of the parameter scale_color. */
gboolean scalesSet_defaultRGBValues(float rgba[4], int mask)
{
  gboolean diff = 0;
  ScaleClass *klass;
  
  klass = scalesGetClass();
  g_return_val_if_fail(klass, FALSE);

  if (mask & MASK_RGB_R && klass->rgba[0] != rgba[0])
    {
      klass->rgba[0] = rgba[0];
      diff = TRUE;
    }
  if (mask & MASK_RGB_G && klass->rgba[1] != rgba[1])
    {
      klass->rgba[1] = rgba[1];
      diff = TRUE;
    }
  if (mask & MASK_RGB_B && klass->rgba[2] != rgba[2])
    {
      klass->rgba[2] = rgba[2];
      diff = TRUE;
    }
  if (mask & MASK_RGB_A && klass->rgba[3] != rgba[3])
    {
      klass->rgba[3] = rgba[3];
      diff = TRUE;
    }
  if (!diff)
    return FALSE;

  /* Set the building tag to false. */
  scaleHasBeenBuilt = FALSE;
  return (gboolean)klass->ext->used;
}

/* Method used to change the value of the parameter scale_line_width. */
gboolean scalesSet_defaultLineWidth(float width)
{
  ScaleClass *klass;
  
  g_return_val_if_fail(width > 0.f && width <= 10.f, FALSE);
  
  klass = scalesGetClass();
  g_return_val_if_fail(klass, FALSE);

  if (width == klass->width)
    return FALSE;

  klass->width = width;
  /* Set the building tag to false. */
  scaleHasBeenBuilt = FALSE;
  return (gboolean)klass->ext->used;
}

/* Method used to change the value of the parameter scale_is_on. */
gboolean scalesSet_areOn(gboolean value)
{
  ScaleClass *klass;
  
  klass = scalesGetClass();
  g_return_val_if_fail(klass, FALSE);

  if (value == klass->ext->used)
    return FALSE;

  DBG_fprintf(stderr, "Scale: set scale status, %d.\n", value);
  klass->ext->used = value;
  return (value && !scaleHasBeenBuilt);
}

/* Get methods. */
float* scalesGet_defaultRGBColor()
{
  ScaleClass *klass;

  klass = scalesGetClass();
  return klass->rgba;
}
gboolean scalesGet_areOn()
{
  ScaleClass *klass;

  klass = scalesGetClass();
  g_return_val_if_fail(klass, FALSE);
  return klass->ext->used;
}
float scalesGet_defaultLineWidth()
{
  ScaleClass *klass;

  klass = scalesGetClass();
  g_return_val_if_fail(klass, -1.f);
  return klass->width;
}
float scaleGet_length(Scale *scale)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), -1.f);

  return scale->length;
}
float* scaleGet_origin(Scale *scale)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), (float*)0);

  return scale->origin;
}
float* scaleGet_orientation(Scale *scale)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), (float*)0);

  return scale->direction;
}

/* Set the coordinates of scale origin. */
gboolean scaleSet_origin(Scale *scale, float xyz[3], int mask)
{
  gboolean difference;

  g_return_val_if_fail(IS_SCALE_TYPE(scale), FALSE);

  difference = FALSE;

  if (mask & MASK_XYZ_X && scale->origin[0] != xyz[0])
    {
      scale->origin[0] = xyz[0];
      difference = 1;
    }
  if (mask & MASK_XYZ_Y && scale->origin[1] != xyz[1])
    {
      scale->origin[1] = xyz[1];
      difference = 1;
    }
  if (mask & MASK_XYZ_Z && scale->origin[2] != xyz[2])
    {
      scale->origin[2] = xyz[2];
      difference = 1;
    }
  if (!difference)
    return FALSE;

  scaleHasBeenBuilt = FALSE;
  return SCALE_GET_CLASS(scale)->ext->used;
}

/* Set the direction of the scale. */
gboolean scaleSet_orientation(Scale *scale, float xyz[3], int mask)
{
  gboolean difference;

  g_return_val_if_fail(IS_SCALE_TYPE(scale), FALSE);

  difference = FALSE;

  if (mask & MASK_XYZ_X && scale->direction[0] != xyz[0])
    {
      scale->direction[0] = xyz[0];
      difference = 1;
    }
  if (mask & MASK_XYZ_Y && scale->direction[1] != xyz[1])
    {
      scale->direction[1] = xyz[1];
      difference = 1;
    }
  if (mask & MASK_XYZ_Z && scale->direction[2] != xyz[2])
    {
      scale->direction[2] = xyz[2];
      difference = 1;
    }
  if (!difference)
    return FALSE;

  scaleHasBeenBuilt = FALSE;
  return SCALE_GET_CLASS(scale)->ext->used;
}

gboolean scaleSet_length(Scale *scale, float lg)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), FALSE);

  if (lg == scale->length)
    return FALSE;

  scale->length = lg;
  scaleHasBeenBuilt = FALSE;
  return SCALE_GET_CLASS(scale)->ext->used;
}

gboolean scaleSet_legend(Scale *scale, const gchar *value)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), FALSE);

  if (scale->legendPattern)
    g_free(scale->legendPattern);

  if (value && *g_strstrip((gchar*)value))
    scale->legendPattern = g_strdup(value);
  else
    scale->legendPattern = (gchar*)0;
  if (scale->legendPattern)
    g_string_assign(scale->legend, scale->legendPattern);
  else
    g_string_printf(scale->legend, SCALE_LEGEND_DEFAULT, scale->length);

  scaleHasBeenBuilt = FALSE;
  return SCALE_GET_CLASS(scale)->ext->used;
}
const gchar* scaleGet_legend(Scale *scale)
{
  g_return_val_if_fail(IS_SCALE_TYPE(scale), (const gchar*)0);
  return scale->legendPattern;
}

GList* scalesGet_scales()
{
  ScaleClass *klass;

  klass = scalesGetClass();
  DBG_fprintf(stderr, "Scales: get the scale list (%d).\n",
	      g_list_length(klass->scales));
  return klass->scales;
}

gboolean scalesSet_defaultStipple(guint16 stipple)
{
  ScaleClass *klass;
  
  klass = scalesGetClass();
  g_return_val_if_fail(klass, FALSE);

  if (stipple == klass->stipple)
    return FALSE;

  klass->stipple = stipple;
  /* Set the building tag to false. */
  scaleHasBeenBuilt = FALSE;
  return (gboolean)klass->ext->used;
}
guint16 scalesGet_defaultStipple()
{
  ScaleClass *klass;

  klass = scalesGetClass();
  g_return_val_if_fail(klass, 0);
  return klass->stipple;
}


/****************/
/* Private part */
/****************/

static void rebuildScale(VisuData *dataObj)
{
  openGLText_rebuildFontList();
  /* Set flag to false to force recreation of Scale. */
  scaleHasBeenBuilt = FALSE;
  scalesDraw(dataObj);
}
static void rebuildScaleOnResourcesLoaded(GObject *visu _U_, VisuData *dataObj,
					  gpointer data _U_)
{
  scalesDraw(dataObj);
}
static void onDataReadySignal(GObject *visu _U_, VisuData *dataObj, gpointer data _U_)
{
  if (!dataObj)
    return;

  scaleHasBeenBuilt = FALSE;
  scalesDraw(dataObj);
}
static void onDataNewSignal(GObject *visu _U_, VisuData *dataObj, gpointer data _U_)
{
  g_signal_connect(G_OBJECT(dataObj), "OpenGLNearFar",
		   G_CALLBACK(onScaleParametersChange), (gpointer)0);
  g_signal_connect(G_OBJECT(dataObj), "OpenGLWidthHeight",
		   G_CALLBACK(onScaleParametersChange), (gpointer)0);
}

static void onScaleParametersChange(VisuData *dataObj, OpenGLView *view _U_,
				    gpointer data _U_)
{
  scaleHasBeenBuilt = FALSE;
  scalesDraw(dataObj);
}

void scalesDraw(VisuData *dataObj)
{
  ScaleClass *klass;
  GList *tmpLst;
  int nlat;
  OpenGLView *view;
  float radius = 0.3;

  /* Nothing to draw; */
  if(!scalesGet_areOn() || scaleHasBeenBuilt) return;

  /* Nothing to draw if no data is associated to
     the current rendering window. */
  if (!dataObj)
    return;

  DBG_fprintf(stderr, "Extension Scale: creating scales.\n");
  klass = scalesGetClass();

  scaleHasBeenBuilt = TRUE;

  openGLText_initFontList();
  view = visuDataGet_openGLView(dataObj);
  nlat = OpenGLViewGet_numberOfFacettes(view, radius);
  
  glDeleteLists(klass->ext->objectListId, 1);

  glNewList(klass->ext->objectListId, GL_COMPILE);

  /* Deactivate light and fog. */
  glDisable(GL_LIGHTING);
  glDisable(GL_FOG);

  glMatrixMode(GL_MODELVIEW);
  glPushMatrix();
  glTranslated(-view->box->dxxs2, -view->box->dyys2, -view->box->dzzs2);

  glLineWidth(klass->width);
  glColor4fv(klass->rgba);
  if (klass->stipple != 65535)
    {
      glEnable(GL_LINE_STIPPLE);
      glLineStipple(1, klass->stipple);
    }

  tmpLst = klass->scales;
  while (tmpLst)
    {
      lineDraw(dataObj, nlat, SCALE(tmpLst->data));
      tmpLst = g_list_next(tmpLst);
    }

  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  glEndList();
}

/* Parameters & resources*/

/* This is a boolean to control is the Scale is render or not. */
static gboolean readScaleIsOn(gchar **lines, int nbLines,
			      int position, VisuData *dataObj _U_, GError **error)
{
  gboolean val;

  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_boolean(lines[0], position, &val, 1, error))
    return FALSE;
  scalesSet_areOn(val);
  return TRUE;
}
/* A resource to control the color used to render the lines of the Scale. */
static gboolean readScaleColor(gchar **lines, int nbLines,
			       int position, VisuData *dataObj _U_, GError **error)
{
  float rgb[4];
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, rgb, 4, error))
    {
      if (*error)
	g_error_free(*error);
      *error = (GError*)0;
      if (!configFileRead_float(lines[0], position, rgb, 3, error))
	return FALSE;
      rgb[3] = 1.f;
    }

  if (configFileClamp_float(&rgb[0], rgb[0], 0., 1.) ||
      configFileClamp_float(&rgb[1], rgb[1], 0., 1.) ||
      configFileClamp_float(&rgb[2], rgb[2], 0., 1.) ||
      configFileClamp_float(&rgb[3], rgb[3], 0., 1.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 4 floating points"
			     "(0 <= v <= 1) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_SCALE_COLOR);
      scalesSet_defaultRGBValues(scaleRGBDefault, MASK_RGB_ALL);
      return FALSE;
    }
  scalesSet_defaultRGBValues(rgb, MASK_RGB_ALL);

  return TRUE;
}
/* A resource to control the width to render the lines of the Scale. */
static gboolean readScaleLineWidth(gchar **lines, int nbLines,
				   int position, VisuData *dataObj _U_, GError **error)
{
  float width;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  if (!configFileRead_float(lines[0], position, &width, 1, error))
    return FALSE;
  if (configFileClamp_float(&width, width, 1., 10.))
    {
      *error = g_error_new(CONFIG_FILE_ERROR, CONFIG_FILE_ERROR_VALUE,
			   _("Parse error at line %d: 1 floating point"
			     "(1 <= v <= 10) must appear after the %s markup.\n"),
			   position, FLAG_RESOURCE_SCALE_LINE);
      scalesSet_defaultLineWidth(SCALE_WIDTH_DEFAULT);
      return FALSE;
    }
  scalesSet_defaultLineWidth(width);

  return TRUE;
}
static gboolean readScaleLineStipple(gchar **lines, int nbLines,
				     int position, VisuData *dataObj _U_, GError **error)
{
  int stipple;

  g_return_val_if_fail(nbLines == 1, FALSE);
  
  if (!configFileRead_integer(lines[0], position, &stipple, 1, error))
    return FALSE;
  scalesSet_defaultStipple((guint16)stipple);

  return TRUE;
}
static gboolean readScaleDefinition(gchar **lines, int nbLines,
				    int position, VisuData *dataObj _U_, GError **error)
{
  float xyz[3], orientation[3], len;
  gchar **tokens, *remains, *legend;
  int pos;
  
  g_return_val_if_fail(nbLines == 1, FALSE);

  tokens = g_strsplit(g_strchug(lines[0]), " ", 0);
  pos = 0;
  if (!configFileRead_floatFromTokens(tokens, &pos, xyz, 3, position, error))
    {
      g_strfreev(tokens);
      return FALSE;      
    }
  if (!configFileRead_floatFromTokens(tokens, &pos, orientation, 3, position, error))
    {
      g_strfreev(tokens);
      return FALSE;      
    }
  if (!configFileRead_floatFromTokens(tokens, &pos, &len, 1, position, error))
    {
      g_strfreev(tokens);
      return FALSE;      
    }
  if (tokens[pos])
    remains = g_strjoinv(" ", &tokens[pos]);
  else
    remains = g_strdup(SCALE_LEGEND_DEFAULT);
  g_strfreev(tokens);

  legend = g_strstrip(remains);
  if (legend && !strcmp(legend, SCALE_AUTO_LEGEND))
    legend = (gchar*)0;
  scaleNew(xyz, orientation, len, legend);
  g_free(remains);

  return TRUE;
}

/* Export function that is called by visu_module to write the
   values of resources to a file. */
static gboolean exportResourcesScale(GString *data, 
				     int *nbLinesWritten,
				     VisuData *dataObj _U_)
{
  ScaleClass *klass;
  GList *tmpLst;
  Scale *scale;
  gchar *legend;

  klass = scalesGetClass();

  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_USED);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_USED);
  g_string_append_printf(data, "    %d\n", klass->ext->used);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_COLOR);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_COLOR);
  g_string_append_printf(data, "    %4.3f %4.3f %4.3f\n",
	  klass->rgba[0], klass->rgba[1], klass->rgba[2]);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_LINE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_LINE);
  g_string_append_printf(data, "    %4.0f\n", klass->width);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_STIPPLE);
  g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_STIPPLE);
  g_string_append_printf(data, "    %d\n", klass->stipple);
  g_string_append_printf(data, "# %s\n", DESC_RESOURCE_SCALE_DEFINITION);
  *nbLinesWritten += 13;
  tmpLst = klass->scales;
  while (tmpLst)
    {
      scale = SCALE(tmpLst->data);
      if (scale->legendPattern)
	legend = scale->legendPattern;
      else
	legend = SCALE_AUTO_LEGEND;
      g_string_append_printf(data, "%s:\n", FLAG_RESOURCE_SCALE_DEFINITION);
      g_string_append_printf(data, "    %g %g %g  %g %g %g  %g  %s\n",
			     scale->origin[0], scale->origin[1], scale->origin[2],
			     scale->direction[0], scale->direction[1], scale->direction[2],
			     scale->length, legend);
      *nbLinesWritten += 2;
      tmpLst = g_list_next(tmpLst);
    }
  g_string_append_printf(data, "\n");
  *nbLinesWritten += 1;

  return TRUE;

}



/* fonction qui dessine la legende*/
static void lineDraw(VisuData *dataObj _U_, int nlat, Scale *scale)
{
  float x2,y2,z2, norm;
  float angles[3]; /*declaration des tableaux angles et coordonnées a 3 cellules*/
  float coord[3];
  float radius = 0.3;
  GLUquadricObj *obj;

  obj = gluNewQuadric();

  norm = scale->length / sqrt(scale->direction[0] * scale->direction[0] +
		       scale->direction[1] * scale->direction[1] +
		       scale->direction[2] * scale->direction[2]);
  x2 = scale->origin[0] + scale->direction[0] * norm;
  y2 = scale->origin[1] + scale->direction[1] * norm;
  z2 = scale->origin[2] + scale->direction[2] * norm;

  coord[0] = x2 - scale->origin[0];
  coord[1] = y2 - scale->origin[1];
  coord[2] = z2 - scale->origin[2];

  matrix_cartesianToSpherical(angles, coord); /*appel de la fonction pour transformé les angles en coordonnée*/

  glPushMatrix(); /*aide damien*/
  glTranslated(scale->origin[0], scale->origin[1], scale->origin[2]); /*aide damien*/
  glRotated(angles[2], 0., 0., 1.);
  glRotated(angles[1], 0., 1., 0.);
  glTranslated(0., 0., scale->length); /*aide damien*/
  gluCylinder(obj, radius, 0., 1.0, nlat, 1); /* 20 NOMBRE DE FACETTES */
  glRotated(180., 1., 0., 0.);
  gluDisk(obj, 0, radius, nlat, 1);
  glPopMatrix();/*aide damien*/

  glBegin (GL_LINES); /*premier ligne */
  glVertex3fv(scale->origin);
  glVertex3f(x2, y2, z2);
  glEnd();

  gluDeleteQuadric(obj);   

  glRasterPos3fv(scale->origin);
  openGLText_drawChars(scale->legend->str); /*affichage de la  legende*/
}
