
#include <stdio.h>
#include <fcntl.h>
#include <errno.h>
#include <ctype.h>
#include <string.h>
#include <sys/ioctl.h>
#include "common.h"
#include "ups.h"
#include "mapping.h"

/* UPSMapping - the logical-to-physical I/O mapping. Filled in by upsd.conf */
static UPS_Map UPSMapping[] = {
  { 0x0, "", "UPSPWR", '\0' },
  { 0x0, "", "UPSOFF", '\0' },
  { 0x0, "", "ACFAIL", '\0' },
  { 0x0, "", "LOWBATT", '\0' },
  { 0x0, "", "UPSDET", '\0' },
  { 0x0, "", "", '\0' }
};

/* These two parallel arrays map RS-232 signal names as found in upsd.conf
   to (Linux-specific?) ioctl() codes. */
char serlines[][MAXLEN] = { "RTS", "CTS", "DSR", "DCD", "DTR", "RI", "" };
int sercodes[] = { TIOCM_RTS, TIOCM_CTS, TIOCM_DSR, TIOCM_CD,
                   TIOCM_DTR, TIOCM_RI, 0x0 };

/*-------------------------------------------------------------------*
     Map_Entry

     Map parameters from config file to internal line map structure.

     Parameters: char *line_param[3] - three strings
                 UPS_Map *mapping - mapping array for UPS I/O

     Returns: number of invalid parameter; 0 otherwise.
 *-------------------------------------------------------------------*/
int Map_Entry(char param[3][MAXLEN], UPS_Map *mapping)
{
  int upsindex;
  int serindex;

  /* Determine which UPS line we're mapping - 2nd param in config file*/
  for(upsindex=0;(mapping+upsindex)->ups_funcname[0] != '\0';upsindex++)
  {
    if( strncmp((mapping+upsindex)->ups_funcname, param[1], MAXLEN) == 0 )
      break;
  }
  if( (mapping+upsindex)->ups_funcname[0] == '\0' )
    return(2); /* Invalid UPS line parameter */

  /* Determine which serial line it's being mapped to - 1st param */
  for(serindex=0;serlines[serindex][0] != '\0';serindex++)
  {
    if( strncmp(serlines[serindex], param[0], MAXLEN) == 0 )
      break;
  }
  if( serlines[serindex][0] == '\0' )
    return(1); /* Invalid serial line parameter */

  switch( param[2][0] )
  {
    case 'H':
    case 'h':
      (mapping+upsindex)->active_state = 'H';
      break;
    case 'L':
    case 'l':
      (mapping+upsindex)->active_state = 'L';
      break;
    default:
      return(3); /* Invalid logic level parameter */
  }
  strncpy((mapping+upsindex)->serial_pin, serlines[serindex], MAXLEN);
  (mapping+upsindex)->ioctl_mask = sercodes[serindex];

  return(0);
} /* end Map_Entry() */

void Dump_Map(UPS_Map *mapping)
{
  int index;

  puts("UPS I/O Mapping:");
  for(index=0;(mapping+index)->ups_funcname[0] != '\0';index++)
  {
    printf("UPS Line %s == %s, active '%c', ioctl:%04x\n",
	   (mapping+index)->ups_funcname, (mapping+index)->serial_pin,
	   (mapping+index)->active_state, (mapping+index)->ioctl_mask);
  }
} /* end Dump_Map() */

/*-------------------------------------------------------------------*
     Config_Read

     Read UPS pin assignments from config file.

     Parameters: char *fname - filename (incl. path) of config file
                 int test_mode - whether we're in test mode or not

     Returns: FALSE if config file couldn't be read or has improper
              syntax; TRUE otherwise.
 *-------------------------------------------------------------------*/
int Config_Read(char *fname, int mode_test)
{
  int index;  /* generic loop var */
  FILE *configfile;
  char inbuff[MAXLINELEN+1];
  char line_param[3][MAXLEN];
  char line_errmsg[MAXLINELEN+MAXLEN+1];
  int status;
  int retval = TRUE;          /* be optimistic */

  /*  line_param[0][0] = line_param[1][0] = line_param[2][0] = '\0';*/

  if( (configfile = fopen(fname,"r")) == NULL )
  {
    LogError("config_read", "Couldn't open " CFGFILE);
    retval = FALSE;
  }
  else
  {
    while( fgets(inbuff, MAXLINELEN, configfile) != NULL )
    {
      if( (inbuff[0] != '#') &&
          (inbuff[0] != '\n') ) /* skip comments and blank lines */
      {
        if( (status = sscanf(inbuff, "%s %s %s", line_param[0], line_param[1],
                   line_param[2])) != 3 )
        {
          LogError("config_read", "Syntax error in " CFGFILE);
          retval = FALSE;
          break;
        }
	else
        {
          if( (status = Map_Entry(line_param, UPSMapping)) > 0 )
          {
	    strcpy(line_errmsg, "Unknown parameter '");
            strncat(line_errmsg, line_param[status-1], MAXLEN);
            LogError("config_read", line_errmsg);
	    retval = FALSE;
	    break;
	  }
        } /* end if/else (3 args in current line) */
      } /* end if (non-comment line) */
    } /* end while */
    fclose(configfile);
  } /* end if (config file accessible) */

  /* Now as a final check we ensure all UPS lines were given an assignment. */
  for(index=0;UPSMapping[index].ups_funcname[0] != '\0';index++)
  {
    if( UPSMapping[index].active_state == '\0' )
    {
      LogError("config_read", "One or more UPS lines undefined!");
      retval = FALSE;
      break;
    }
  }

  /* If things went OK so far and we're in test mode, dump mapping info */
  if( (retval == TRUE) && (mode_test) )
    Dump_Map(UPSMapping);
  return(retval);
} /* end Config_Read() */

/*-------------------------------------------------------------------*
     Serial_Set

     Set specified lines on the serial port.

     Parameters:  int fd   - File descriptor of monitor device.
                  int lstate - bit mask of handshake outputs to set.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void Serial_Set (int fd, int lstate)
{
  ioctl(fd, TIOCMBIS, &lstate);
} /* end Serial_Set() */

/*-------------------------------------------------------------------*
     Serial_Clear

     Clear specified lines on the serial port.

     Parameters:  int fd   - File descriptor of monitor device.
                  int lstate - bit mask of handshake outputs to clear.

     Returns:     Nothing.
 *-------------------------------------------------------------------*/
void Serial_Clear (int fd, int lstate)
{
  ioctl(fd, TIOCMBIC, &lstate);
} /* end Serial_Clear() */

/*-------------------------------------------------------------------*
     UPS_Assert

     Asserts a UPS line.

     Parameters: int fd - file handle of UPS device
                 int logical_line - logical UPS line to assert

     Returns: <none>

     NOTE: References global table UPSMapping[] to map logical_line to
           a physical UPS line.
 *-------------------------------------------------------------------*/
void UPS_Assert (int fd, int logical_line)
{
  int iomask;

  iomask = UPSMapping[logical_line].ioctl_mask;

  if( toupper(UPSMapping[logical_line].active_state) == 'H' )
    Serial_Set(fd, iomask);
  else
    Serial_Clear(fd, iomask);
} /* end UPS_Assert() */

/*-------------------------------------------------------------------*
     UPS_Deassert

     De-asserts a UPS line.

     Parameters: int fd - file handle of UPS device
                 int logical_line - logical UPS line to de-assert

     Returns: <none>

     NOTE: References global table UPSMapping[] to map logical_line to
           a physical UPS line.
 *-------------------------------------------------------------------*/
void UPS_Deassert (int fd, int logical_line)
{
  int iomask;

  iomask = UPSMapping[logical_line].ioctl_mask;

  if( toupper(UPSMapping[logical_line].active_state) == 'H' )
    Serial_Clear(fd, iomask);
  else
    Serial_Set(fd, iomask);
} /* end UPS_Deassert() */

/*-------------------------------------------------------------------*
     UPS_Read

     Read UPS physical lines and convert into active-high logical
     status.

     Parameters: fd - file handle of UPS device

     Returns: logical status  of each UPS line.
 *-------------------------------------------------------------------*/
int UPS_Read(int fd)
{
  int flagtemp;
  int flags;
  int index;

  flags = 0;

  ioctl (fd, TIOCMGET, &flagtemp);
  for(index=0;UPSMapping[index].active_state != '\0';index++)
  {
    if( toupper(UPSMapping[index].active_state) == 'H' )
    {
      if( (flagtemp & UPSMapping[index].ioctl_mask) != 0 )
	flags |= (0x1<<index);
    }
    else  /* active low physical line */
    {
      if( (flagtemp & UPSMapping[index].ioctl_mask) == 0 )
	flags |= (0x1<<index);
    }
  }
  return(flags);
} /* end UPS_Read() */

/*-------------------------------------------------------------------*
 *-------------------------------------------------------------------*/
/*-------------------------------------------------------------------*
 *-------------------------------------------------------------------*/
