/*
 * Copyright 2013 Canonical Ltd.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; version 3.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Authors:
 *      Nick Dedekind <nick.dedekind@canonical.com>
 */

import QtQuick 2.0
import Unity.Indicators 0.1 as Indicators
import Unity.Indicators.Network 0.1 as ICNetwork
import Unity.Indicators.Messaging 0.1 as ICMessaging
import QMenuModel 0.1 as QMenuModel

Item {
    id: menuFactory

    property var model: null

    property var _map:  {
        "unity.widgets.systemsettings.tablet.volumecontrol" : sliderMenu,
        "unity.widgets.systemsettings.tablet.switch"        : switchMenu,

        "com.canonical.indicator.button"    : buttonMenu,
        "com.canonical.indicator.div"       : divMenu,
        "com.canonical.indicator.section"   : sectionMenu,
        "com.canonical.indicator.progress"  : progressMenu,
        "com.canonical.indicator.slider"    : sliderMenu,
        "com.canonical.indicator.switch"    : switchMenu,

        "com.canonical.indicator.messages.messageitem"  : messageItem,
        "com.canonical.indicator.messages.sourceitem"   : groupedMessage,

        "com.canonical.unity.slider"    : sliderMenu,
        "com.canonical.unity.switch"    : switchMenu,

        "unity.widgets.systemsettings.tablet.wifisection" : wifiSection,
        "unity.widgets.systemsettings.tablet.accesspoint" : accessPoint,
    }

    Component { id: divMenu; Indicators.DivMenuItem {} }

    Component {
        id: sliderMenu;
        Indicators.SliderMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            icon: menu && menu.icon ? menu.icon : ""
            minIcon: menu && menu.ext.minIcon ? menu.ext.minIcon : ""
            maxIcon: menu && menu.ext.maxIcon ? menu.ext.maxIcon : ""

            minimumValue: menu.ext.minValue ? menu.ext.minValue : 0.0
            maximumValue: {
                var maximum = menu.ext.maxValue ? menu.ext.maxValue : 1.0
                if (maximum <= minimumValue) {
                        return minimumValue + 1;
                }
                return maximum;
            }
            value: menu ? menu.actionState : 0.0
            enabled: menu ? menu.sensitive : false

            Component.onCompleted: {
                model.loadExtendedAttributes(modelIndex, {'min-value': 'double',
                                                          'max-value': 'double',
                                                          'min-icon': 'icon',
                                                          'max-icon': 'icon'});
            }

            onChangeState: {
                model.changeState(modelIndex, value);
            }
        }
    }

    Component {
        id: buttonMenu;
        Indicators.ButtonMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            enabled: menu ? menu.sensitive : false

            onActivate: {
                model.activate(modelIndex);
            }
        }
    }
    Component {
        id: sectionMenu;
        Indicators.SectionMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
        }
    }

    Component {
        id: progressMenu;
        Indicators.ProgressMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            icon: menu ? menu.icon : ""
            value : menu ? menu.actionState : 0.0
            enabled: menu ? menu.sensitive : false
        }
    }

    Component {
        id: standardMenu;
        Indicators.StandardMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            icon: menu && menu.icon ? menu.icon : ""
            checkable: menu ? (menu.isCheck || menu.isRadio) : false
            checked: checkable ? menu.isToggled : false
            enabled: menu ? menu.sensitive : false

            onActivate: {
                model.activate(modelIndex);
            }
        }
    }

    Component {
        id: switchMenu;
        Indicators.SwitchMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            icon: menu && menu.icon ? menu.icon : ""
            checked: menu ? menu.isToggled : false
            enabled: menu ? menu.sensitive : false

            onActivate: {
                model.activate(modelIndex);
            }
        }
    }

    Component {
        id: wifiSection;
        Indicators.SectionMenuItem {
            property QtObject menu: null

            text: menu && menu.label ? menu.label : ""
            busy: menu ? menu.ext.xCanonicalBusyAction : false

            Component.onCompleted: {
                model.loadExtendedAttributes(modelIndex, {'x-canonical-busy-action': 'bool'});
            }
        }
    }

    Component {
        id: accessPoint;
        ICNetwork.AccessPoint {
            property QtObject menu: null
            property var strengthAction: QMenuModel.UnityMenuAction {
                model: menuFactory.model ? menuFactory.model : null
                index: modelIndex
                name: menu ? menu.ext.xCanonicalWifiApStrengthAction : ""
            }

            text: menu && menu.label ? menu.label : ""
            secure: menu ? menu.ext.xCanonicalWifiApIsSecure : false
            adHoc: menu ? menu.ext.xCanonicalWifiApIsAdhoc : false
            checked: menu ? menu.isToggled : false
            signalStrength: strengthAction.valid ? strengthAction.state : 0
            enabled: menu ? menu.sensitive : false

            Component.onCompleted: {
                model.loadExtendedAttributes(modelIndex, {'x-canonical-wifi-ap-is-adhoc': 'bool',
                                                          'x-canonical-wifi-ap-is-secure': 'bool',
                                                          'x-canonical-wifi-ap-strength-action': 'string'});
            }
            onActivate: {
                model.activate(modelIndex);
            }
        }
    }

    Component {
        id: messageItem
        ICMessaging.MessageMenuItemFactory {

            property QtObject menu: null
            model: menuFactory.model ? menuFactory.model : null

            Component.onCompleted: {
                model.loadExtendedAttributes(modelIndex, {'x-canonical-time': 'int64',
                                                          'x-canonical-text': 'string',
                                                          'x-canonical-message-actions': 'variant',
                                                          'icon': 'icon',
                                                          'x-canonical-app-icon': 'icon'});
            }
        }
    }

    Component {
        id: groupedMessage
        ICMessaging.GroupedMessage {
            property QtObject menu: null

            title: menu && menu.label ? menu.label : ""
            count: menu && menu.actionState[0] ? menu.actionState[0] : "0"
            appIcon: menu && menu.ext.icon !== undefined ? menu.ext.icon : "qrc:/indicators/artwork/messaging/default_app.svg"

            Component.onCompleted: {
                model.loadExtendedAttributes(modelIndex, {'icon': 'icon'});
            }
            onActivateApp: {
                model.activate(modelIndex, true);
            }
            onDismiss: {
                model.activate(modelIndex, false);
            }
        }
    }

    function load(modelData) {
        if (modelData.type !== undefined) {
            var component = _map[modelData.type];
            if (component !== undefined) {
                return component;
            }
        } else {
            if (modelData.isSeparator) {
                return divMenu;
            }
        }
        return standardMenu;
    }
}
