/*
 * util/config_file.h - reads and stores the config file for unbound.
 *
 * Copyright (c) 2007, NLnet Labs. All rights reserved.
 *
 * This software is open source.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * Redistributions of source code must retain the above copyright notice,
 * this list of conditions and the following disclaimer.
 * 
 * Redistributions in binary form must reproduce the above copyright notice,
 * this list of conditions and the following disclaimer in the documentation
 * and/or other materials provided with the distribution.
 * 
 * Neither the name of the NLNET LABS nor the names of its contributors may
 * be used to endorse or promote products derived from this software without
 * specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR
 * PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

/**
 * \file
 *
 * This file contains functions for the config file.
 */

#ifndef UTIL_CONFIG_FILE_H
#define UTIL_CONFIG_FILE_H
struct config_stub;
struct config_strlist;
struct config_str2list;

/**
 * The configuration options.
 * Strings are malloced.
 */
struct config_file {
	/** verbosity level as specified in the config file */
	int verbosity;

	/** statistics interval (in seconds) */
	int stat_interval;
	/** if false, statistics values are reset after printing them */
	int stat_cumulative;
	/** if true, the statistics are kept in greater detail */
	int stat_extended;

	/** number of threads to create */
	int num_threads;

	/** port on which queries are answered. */
	int port;
	/** do ip4 query support. */
	int do_ip4;
	/** do ip6 query support. */
	int do_ip6;
	/** do udp query support. */
	int do_udp;
	/** do tcp query support. */
	int do_tcp;

	/** outgoing port range number of ports (per thread) */
	int outgoing_num_ports;
	/** number of outgoing tcp buffers per (per thread) */
	size_t outgoing_num_tcp;
	/** number of incoming tcp buffers per (per thread) */
	size_t incoming_num_tcp;
	/** allowed udp port numbers, array with 0 if not allowed */
	int* outgoing_avail_ports;

	/** number of bytes buffer size for DNS messages */
	size_t msg_buffer_size;
	/** size of the message cache */
	size_t msg_cache_size;
	/** slabs in the message cache. */
	size_t msg_cache_slabs;
	/** number of queries every thread can service */
	size_t num_queries_per_thread;
	/** number of msec to wait before items can be jostled out */
	size_t jostle_time;
	/** size of the rrset cache */
	size_t rrset_cache_size;
	/** slabs in the rrset cache */
	size_t rrset_cache_slabs;
	/** host cache ttl in seconds */
	int host_ttl;
	/** host is lame for a zone ttl, in seconds */
	int lame_ttl;
	/** number of slabs in the infra host cache */
	size_t infra_cache_slabs;
	/** max number of hosts in the infra cache */
	size_t infra_cache_numhosts;
	/** max size of lame zones per host in the infra cache */
	size_t infra_cache_lame_size;

	/** the target fetch policy for the iterator */
	char* target_fetch_policy;

	/** automatic interface for incoming messages. Uses ipv6 remapping,
	 * and recvmsg/sendmsg ancillary data to detect interfaces, boolean */
	int if_automatic;

	/** number of interfaces to open. If 0 default all interfaces. */
	int num_ifs;
	/** interface description strings (IP addresses) */
	char **ifs;

	/** number of outgoing interfaces to open. 
	 * If 0 default all interfaces. */
	int num_out_ifs;
	/** outgoing interface description strings (IP addresses) */
	char **out_ifs;

	/** the root hints */
	struct config_strlist* root_hints;
	/** the stub definitions, linked list */
	struct config_stub* stubs;
	/** the forward zone definitions, linked list */
	struct config_stub* forwards;
	/** list of donotquery addresses, linked list */
	struct config_strlist* donotqueryaddrs;
	/** list of access control entries, linked list */
	struct config_str2list* acls;
	/** use default localhost donotqueryaddr entries */
	int donotquery_localhost;

	/** harden against very small edns buffer sizes */
	int harden_short_bufsize;
	/** harden against very large query sizes */
	int harden_large_queries;
	/** harden against spoofed glue (out of zone data) */
	int harden_glue;
	/** harden against receiving no DNSSEC data for trust anchor */
	int harden_dnssec_stripped;
	/** harden the referral path, query for NS,A,AAAA and validate */
	int harden_referral_path;
	/** use 0x20 bits in query as random ID bits */
	int use_caps_bits_for_id;
	/** strip away these private addrs from answers, no DNS Rebinding */
	struct config_strlist* private_address;
	/** allow domain (and subdomains) to use private address space */
	struct config_strlist* private_domain;
	/** what threshold for unwanted action. */
	size_t unwanted_threshold;

	/** chrootdir, if not "" or chroot will be done */
	char* chrootdir;
	/** username to change to, if not "". */
	char* username;
	/** working directory */
	char* directory;
	/** filename to log to. */
	char* logfile;
	/** pidfile to write pid to. */
	char* pidfile;

	/** should log messages be sent to syslogd */
	int use_syslog;
	/** log timestamp in ascii UTC */
	int log_time_ascii;

	/** do not report identity (id.server, hostname.bind) */
	int hide_identity;
	/** do not report version (version.server, version.bind) */
	int hide_version;
	/** identity, hostname is returned if "". */
	char* identity;
	/** version, package version returned if "". */
	char* version;

	/** the module configuration string */
	char* module_conf;
	
	/** files with trusted DS and DNSKEYs in zonefile format, list */
	struct config_strlist* trust_anchor_file_list;
	/** list of trustanchor keys, linked list */
	struct config_strlist* trust_anchor_list;
	/** files with trusted DNSKEYs in named.conf format, list */
	struct config_strlist* trusted_keys_file_list;
	/** DLV anchor file */
	char* dlv_anchor_file;
	/** DLV anchor inline */
	struct config_strlist* dlv_anchor_list;
	/** insecure domain list */
	struct config_strlist* domain_insecure;

	/** the number of seconds maximal TTL used for RRsets and messages */
	int max_ttl;
	/** the number of seconds minimum TTL used for RRsets and messages */
	int min_ttl;
	/** if not 0, this value is the validation date for RRSIGs */
	int32_t val_date_override;
	/** the minimum for signature clock skew */
	int32_t val_sig_skew_min;
	/** the maximum for signature clock skew */
	int32_t val_sig_skew_max;
	/** this value sets the number of seconds before revalidating bogus */
	int bogus_ttl; 
	/** should validator clean additional section for secure msgs */
	int val_clean_additional;
	/** log bogus messages by the validator */
	int val_log_level;
	/** should validator allow bogus messages to go through */
	int val_permissive_mode;
	/** nsec3 maximum iterations per key size, string */
	char* val_nsec3_key_iterations;

	/** size of the key cache */
	size_t key_cache_size;
	/** slabs in the key cache. */
	size_t key_cache_slabs;
	/** size of the neg cache */
	size_t neg_cache_size;

	/** local zones config */
	struct config_str2list* local_zones;
	/** local zones nodefault list */
	struct config_strlist* local_zones_nodefault;
	/** local data RRs configged */
	struct config_strlist* local_data;

	/** remote control section. enable toggle. */
	int remote_control_enable;
	/** the interfaces the remote control should listen on */
	struct config_strlist* control_ifs;
	/** port number for the control port */
	int control_port;
	/** private key file for server */
	char* server_key_file;
	/** certificate file for server */
	char* server_cert_file;
	/** private key file for unbound-control */
	char* control_key_file;
	/** certificate file for unbound-control */
	char* control_cert_file;

	/** Python script file */
	char* python_script;

	/** daemonize, i.e. fork into the background. */
	int do_daemonize;
};

/**
 * Stub config options
 */
struct config_stub {
	/** next in list */
	struct config_stub* next;
	/** domain name (in text) of the stub apex domain */
	char* name;
	/** list of stub nameserver hosts (domain name) */
	struct config_strlist* hosts;
	/** list of stub nameserver addresses (IP address) */
	struct config_strlist* addrs;
	/** if stub-prime is set */
	int isprime;
};

/**
 * List of strings for config options
 */
struct config_strlist {
	/** next item in list */
	struct config_strlist* next;
	/** config option string */
	char* str;
};

/**
 * List of two strings for config options
 */
struct config_str2list {
	/** next item in list */
	struct config_str2list* next;
	/** first string */
	char* str;
	/** second string */
	char* str2;
};

/**
 * Create config file structure. Filled with default values.
 * @return: the new structure or NULL on memory error.
 */
struct config_file* config_create();

/**
 * Create config file structure for library use. Filled with default values.
 * @return: the new structure or NULL on memory error.
 */
struct config_file* config_create_forlib();

/**
 * Read the config file from the specified filename.
 * @param config: where options are stored into, must be freshly created.
 * @param filename: name of configfile. If NULL nothing is done.
 * @param chroot: if not NULL, the chroot dir currently in use (for include).
 * @return: false on error. In that case errno is set, ENOENT means 
 * 	file not found.
 */
int config_read(struct config_file* config, const char* filename,
	const char* chroot);

/**
 * Destroy the config file structure.
 * @param config: to delete.
 */
void config_delete(struct config_file* config);

/**
 * Apply config to global constants; this routine is called in single thread.
 * @param config: to apply. Side effect: global constants change.
 */
void config_apply(struct config_file* config);

/**
 * Set the given keyword to the given value.
 * @param config: where to store config
 * @param option: option name, including the ':' character.
 * @param value: value, this string is copied if needed, or parsed.
 * 	The caller owns the value string.
 * @return 0 on error (malloc or syntax error).
 */
int config_set_option(struct config_file* config, const char* option,
	const char* value);

/**
 * Insert string into strlist.
 * @param head: pointer to strlist head variable.
 * @param item: new item. malloced by caller. If NULL the insertion fails.
 * @return: true on success.
 */
int cfg_strlist_insert(struct config_strlist** head, char* item);

/**
 * Insert string into str2list.
 * @param head: pointer to str2list head variable.
 * @param item: new item. malloced by caller. If NULL the insertion fails.
 * @param i2: 2nd string, malloced by caller. If NULL the insertion fails.
 * @return: true on success.
 */
int cfg_str2list_insert(struct config_str2list** head, char* item, char* i2);

/**
 * Delete items in config string list.
 * @param list: list.
 */
void config_delstrlist(struct config_strlist* list);

/**
 * Delete items in config double string list.
 * @param list: list.
 */
void config_deldblstrlist(struct config_str2list* list);

/**
 * Delete items in config stub list.
 * @param list: list.
 */
void config_delstubs(struct config_stub* list);

/**
 * Convert 14digit to time value
 * @param str: string of 14 digits
 * @return time value or 0 for error.
 */
uint32_t cfg_convert_timeval(const char* str);

/**
 * Count number of values in the string.
 * format ::= (sp num)+ sp
 * num ::= [-](0-9)+
 * sp ::= (space|tab)*
 *
 * @param str: string
 * @return: 0 on parse error, or empty string, else
 *	number of integer values in the string.
 */
int cfg_count_numbers(const char* str);

/**
 * Convert a 'nice' memory or file size into a bytecount
 * From '100k' to 102400. and so on. Understands kKmMgG.
 * k=1024, m=1024*1024, g=1024*1024*1024.
 * @param str: string
 * @param res: result is stored here, size in bytes.
 * @return: true if parsed correctly, or 0 on a parse error (and an error 
 * is logged).
 */
int cfg_parse_memsize(const char* str, size_t* res);

/**
 * Parse local-zone directive into two strings and register it in the config.
 * @param cfg: to put it in.
 * @param val: argument strings to local-zone, "example.com nodefault".
 * @return: false on failure
 */
int cfg_parse_local_zone(struct config_file* cfg, const char* val);

/**
 * Mark "number" or "low-high" as available or not in ports array.
 * @param str: string in input
 * @param allow: give true if this range is permitted.
 * @param avail: the array from cfg.
 * @param num: size of the array (65536).
 * @return: true if parsed correctly, or 0 on a parse error (and an error 
 * is logged).
 */
int cfg_mark_ports(const char* str, int allow, int* avail, int num);

/**
 * Get a condensed list of ports returned. allocated.
 * @param cfg: config file.
 * @param avail: the available ports array is returned here.
 * @return: number of ports in array or 0 on error.
 */
int cfg_condense_ports(struct config_file* cfg, int** avail);

/**
 * Scan ports available
 * @param avail: the array from cfg.
 * @param num: size of the array (65536).
 * @return the number of ports available for use.
 */
int cfg_scan_ports(int* avail, int num);

/** 
 * Convert a filename to full pathname in original filesys
 * @param fname: the path name to convert.
 *      Must not be null or empty.
 * @param cfg: config struct for chroot and chdir (if set).
 * @param use_chdir: if false, only chroot is applied.
 * @return pointer to malloced buffer which is: [chroot][chdir]fname
 *      or NULL on malloc failure.
 */
char* fname_after_chroot(const char* fname, struct config_file* cfg, 
	int use_chdir);

/**
 * Convert a ptr shorthand into a full reverse-notation PTR record.
 * @param str: input string, "IP name"
 * @return: malloced string "reversed-ip-name PTR name"
 */
char* cfg_ptr_reverse(char* str);

/**
 * Used during options parsing
 */
struct config_parser_state {
	/** name of file being parser */
	char* filename;
	/** line number in the file, starts at 1 */
	int line;
	/** number of errors encountered */
	int errors;
	/** the result of parsing is stored here. */
	struct config_file* cfg;
	/** the current chroot dir (or NULL if none) */
	const char* chroot;
};

/** global config parser object used during config parsing */
extern struct config_parser_state* cfg_parser;
/** parsing helpers: print error with file and line numbers */
void ub_c_error(const char* msg);
/** parsing helpers: print error with file and line numbers */
void ub_c_error_msg(const char* fmt, ...) ATTR_FORMAT(printf, 1, 2);

#endif /* UTIL_CONFIG_FILE_H */
