/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>
#include <core_api/L10n.h>

#include <gobject/uHMMObject.h>

#include <distributed_computing/SerializeUtils.h>
#include <util/uhmm3Utilities.h>
#include "uHMM3SearchLocalTask.h"

namespace GB2 {

/*************************************************
 * UHMM3SearchLocalTaskSettings
 *************************************************/

UHMM3SearchLocalTaskSettings::UHMM3SearchLocalTaskSettings( const UHMM3SearchTaskSettings & s, P7_HMM * h, const DNASequence & se  )
: settings( s ), hmm( h ), sequence( se ) {
}

UHMM3SearchLocalTaskSettings::UHMM3SearchLocalTaskSettings() : hmm( NULL ) {
}

UHMM3SearchLocalTaskSettings::~UHMM3SearchLocalTaskSettings() {
}

QVariant UHMM3SearchLocalTaskSettings::serialize() const {
    assert( NULL != hmm && !sequence.isNull() && checkUHMM3SearchSettings( (UHMM3SearchSettings*)&settings.inner ) );
    QVariantList res;
    res << SerializeUtils::serializeValue<DNASequence>( sequence );
    res << serializeHMM();
    res << serializeSettings();
    return res;
}

QVariant UHMM3SearchLocalTaskSettings::serializeHMM() const {
    assert( NULL != hmm );
    QVariantList res;
    
    res << SerializeUtils::serializeValue( hmm->M );
    res << SerializeUtils::serializeArray( hmm->t[0], p7H_NTRANSITIONS * ( hmm->M + 1 ) );
    res << SerializeUtils::serializeArray( hmm->mat[0], hmm->abc->K * ( hmm->M + 1 ) );
    res << SerializeUtils::serializeArray( hmm->ins[0], hmm->abc->K * ( hmm->M + 1 ) );
    res << SerializeUtils::serializeValue( hmm->name );
    res << SerializeUtils::serializeValue( hmm->acc );
    res << SerializeUtils::serializeValue( hmm->desc );
    res << SerializeUtils::serializeArray( hmm->rf, hmm->M + 2 );
    res << SerializeUtils::serializeArray( hmm->cs, hmm->M + 2 );
    res << SerializeUtils::serializeArray( hmm->ca, hmm->M + 2 );
    res << SerializeUtils::serializeValue( hmm->comlog );
    res << SerializeUtils::serializeValue( hmm->nseq );
    res << SerializeUtils::serializeValue( hmm->eff_nseq );
    res << SerializeUtils::serializeValue( hmm->ctime );
    res << SerializeUtils::serializeArray( hmm->map, hmm->M + 1 );
    res << SerializeUtils::serializeValue( hmm->checksum );
    res << SerializeUtils::serializeArray( hmm->evparam, p7_NEVPARAM );
    res << SerializeUtils::serializeArray( hmm->cutoff, p7_NCUTOFFS );
    res << SerializeUtils::serializeArray( hmm->compo, p7_MAXABET );
    res << SerializeUtils::serializeValue( (qlonglong)hmm->offset );
    res << SerializeUtils::serializeValue( hmm->abc->type );
    res << SerializeUtils::serializeValue( hmm->flags );
    
    return QVariant( res );
}

bool UHMM3SearchLocalTaskSettings::deserializeHMM( const QVariant & data ) {
    assert( NULL == hmm );
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_HMM_LIST_SZ != args.size() ) {
        return false;
    }
    
    int m = 0;
    if( !SerializeUtils::deserializeValue( args[0], &m ) ) { return false; }
    int alType = 0;
    if( !SerializeUtils::deserializeValue( args[20], &alType ) ) { return false; }
    int flags = 0;
    if( !SerializeUtils::deserializeValue( args[21], &flags ) ){ return false; }
    hmm = p7_hmm_Create( m, alType, flags );
    if( NULL == hmm ) {
        return false;
    }
    assert( NULL != hmm->abc );
    
    hmm->M = m;
    if( !SerializeUtils::deserializeArray( args[1], hmm->t[0], p7H_NTRANSITIONS * ( hmm->M + 1 ) ) ) { return false; }
    if( !SerializeUtils::deserializeArray( args[2], hmm->mat[0], hmm->abc->K * ( hmm->M + 1 ) ) ) { return false; }
    if( !SerializeUtils::deserializeArray( args[3], hmm->ins[0], hmm->abc->K * ( hmm->M + 1 ) ) ) { return false; }
    
    if( !SerializeUtils::deserializeValue( args[4], &hmm->name ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[5], &hmm->acc ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[6], &hmm->desc ) ) { return false; }
    if( hmm->flags & p7H_RF ) {
        assert( NULL != hmm->rf );
        if( !SerializeUtils::deserializeArray( args[7], hmm->rf, hmm->M + 2 ) ) { return false; }
    }
    if( hmm->flags & p7H_CS ) {
        assert( NULL != hmm->cs );
        if( !SerializeUtils::deserializeArray( args[8], hmm->cs, hmm->M + 2 ) ) { return false; }
    }
    if( hmm->flags & p7H_CA ) {
        assert( NULL != hmm->ca );
        if( !SerializeUtils::deserializeArray( args[9], hmm->ca, hmm->M + 2 ) ) { return false; }
    }
    if( !SerializeUtils::deserializeValue( args[10], &hmm->comlog ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[11], &hmm->nseq ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[12], &hmm->eff_nseq ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[13], &hmm->ctime ) ) { return false; }
    if( hmm->flags & p7H_MAP ) {
        if( !SerializeUtils::deserializeArray( args[14], hmm->map, hmm->M + 1 ) ) { return false; }
    }
    if( !SerializeUtils::deserializeValue( args[15], &hmm->checksum ) ) { return false; }
    if( !SerializeUtils::deserializeArray( args[16], hmm->evparam, p7_NEVPARAM ) ) { return false; }
    if( !SerializeUtils::deserializeArray( args[17], hmm->cutoff, p7_NCUTOFFS ) ) { return false; }
    if( !SerializeUtils::deserializeArray( args[18], hmm->compo, p7_MAXABET ) ) { return false; }
    {
        qlonglong tmp = 0;
        if( !SerializeUtils::deserializeValue( args[19], &tmp ) ) { return false; }
        hmm->offset = (off_t)tmp;
    }
    
    return true;
}

QVariant UHMM3SearchLocalTaskSettings::serializeSettings() const {
    QVariantList res;
    const UHMM3SearchSettings & set = settings.inner;
    
    res << SerializeUtils::serializeValue( set.e );
    res << SerializeUtils::serializeValue( set.t );
    res << SerializeUtils::serializeValue( set.z );
    res << SerializeUtils::serializeValue( set.domE );
    res << SerializeUtils::serializeValue( set.domT );
    res << SerializeUtils::serializeValue( set.domZ );
    res << SerializeUtils::serializeValue(    set.useBitCutoffs );
    res << SerializeUtils::serializeValue( set.incE );
    res << SerializeUtils::serializeValue( set.incT );
    res << SerializeUtils::serializeValue( set.incDomE );
    res << SerializeUtils::serializeValue( set.incDomT );
    res << SerializeUtils::serializeValue( set.f1 );
    res << SerializeUtils::serializeValue( set.f2 );
    res << SerializeUtils::serializeValue( set.f3 );
    res << SerializeUtils::serializeValue(    set.doMax );
    res << SerializeUtils::serializeValue(    set.noBiasFilter );
    res << SerializeUtils::serializeValue(    set.noNull2 );
    res << SerializeUtils::serializeValue(    set.seed );
    
    return res;
}

bool UHMM3SearchLocalTaskSettings::deserializeSettings( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_HMM_SEARCH_SETTINGS_LIST_SZ != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0],  &settings.inner.e ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[1],  &settings.inner.t ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[2],  &settings.inner.z ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[3],  &settings.inner.domE ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[4],  &settings.inner.domT ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[5],  &settings.inner.domZ ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[6],  &settings.inner.useBitCutoffs ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[7],  &settings.inner.incE ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[8],  &settings.inner.incT ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[9],  &settings.inner.incDomE ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[10], &settings.inner.incDomT ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[11], &settings.inner.f1 ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[12], &settings.inner.f2 ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[13], &settings.inner.f3 ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[14], &settings.inner.doMax ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[15], &settings.inner.noBiasFilter ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[16], &settings.inner.noNull2 ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[17], &settings.inner.seed ) ) { return false; }
    
    return true;
}

bool UHMM3SearchLocalTaskSettings::deserialize( const QVariant & data ) {
    assert( NULL == hmm && sequence.isNull() );
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_LOCAL_TASK_SETTINGS_LIST_SZ != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0], &sequence ) ) { return false; }
    if( !deserializeHMM( args[1] ) ) { return false; }
    if( !deserializeSettings( args[2] ) ) { return false; }
    return true;
}

UHMM3SearchTaskSettings UHMM3SearchLocalTaskSettings::getSearchTaskSettings() const {
    return settings;
}

P7_HMM * UHMM3SearchLocalTaskSettings::getHMM() const {
    return hmm;
}

DNASequence UHMM3SearchLocalTaskSettings::getDNASequence() const {
    return sequence;
}

/*************************************************
* UHMM3SearchLocalTaskResult
*************************************************/

UHMM3SearchLocalTaskResult::UHMM3SearchLocalTaskResult( const UHMM3SWSearchTaskResult & r ) : result( r ) {
}

UHMM3SearchLocalTaskResult::UHMM3SearchLocalTaskResult() {
}

UHMM3SearchLocalTaskResult::~UHMM3SearchLocalTaskResult() {
}

QVariant UHMM3SearchLocalTaskResult::serialize() const {
    QVariantList res;
    foreach( UHMM3SWSearchTaskDomainResult domainResult, result ) {
        QVariantList domainVariant;
        domainVariant << serializeHMM3SearchSeqDomainResult( domainResult.generalResult );
        domainVariant << SerializeUtils::serializeValue( domainResult.onCompl );
        domainVariant << SerializeUtils::serializeValue( domainResult.onAmino );
        res << QVariant( domainVariant );
    }
    return res;
}

QVariant UHMM3SearchLocalTaskResult::serializeHMM3SearchSeqDomainResult( const UHMM3SearchSeqDomainResult & domain ) const {
    QVariantList res;
    res << SerializeUtils::serializeValue( domain.score );
    res << SerializeUtils::serializeValue( domain.bias );
    res << SerializeUtils::serializeValue( domain.ival );
    res << SerializeUtils::serializeValue( domain.cval );
    res << SerializeUtils::serializeValue( domain.acc );
    res << SerializeUtils::serializeValue( domain.isSignificant );
    res << SerializeUtils::serializeValue( domain.queryRegion.startPos );
    res << SerializeUtils::serializeValue( domain.queryRegion.len );
    res << SerializeUtils::serializeValue( domain.seqRegion.startPos );
    res << SerializeUtils::serializeValue( domain.seqRegion.len );
    res << SerializeUtils::serializeValue( domain.envRegion.startPos );
    res << SerializeUtils::serializeValue( domain.envRegion.len );
    return res;
}

bool UHMM3SearchLocalTaskResult::deserialize( const QVariant & data ) {
    assert( result.isEmpty() );
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    int sz = args.size();
    int i = 0;
    for(; i < sz; ++i ) {
        QVariant cur = args.at( i );
        if( !cur.canConvert( QVariant::List ) ) {
            result.clear();
            return false;
        }
        QVariantList current = cur.toList();
        if( 3 != current.size() ) {
            result.clear();
            return false;
        }
        UHMM3SWSearchTaskDomainResult domainResult;
        if( !deserializeHMM3SearchSeqDomainResult( current[0], &domainResult.generalResult ) ) { return false; }
        if( !SerializeUtils::deserializeValue( current[1], &domainResult.onCompl ) ) { return false; }
        if( !SerializeUtils::deserializeValue( current[2], &domainResult.onAmino ) ) { return false; }
        result << domainResult;
    }
    return true;
}

bool UHMM3SearchLocalTaskResult::deserializeHMM3SearchSeqDomainResult( const QVariant & data, UHMM3SearchSeqDomainResult * domainRes ) {
    if( NULL == domainRes ) {
        return false;
    }
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( 12 != args.size() ) {
        return false;
    }
    
    if( !SerializeUtils::deserializeValue( args[0], &domainRes->score ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[1], &domainRes->bias ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[2], &domainRes->ival ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[3], &domainRes->cval ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[4], &domainRes->acc ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[5], &domainRes->isSignificant ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[6], &domainRes->queryRegion.startPos ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[7], &domainRes->queryRegion.len ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[8], &domainRes->seqRegion.startPos ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[9], &domainRes->seqRegion.len ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[10], &domainRes->envRegion.startPos ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[11], &domainRes->envRegion.len ) ) { return false; }
    return true;
}

void UHMM3SearchLocalTaskResult::setResult( const UHMM3SWSearchTaskResult & res ) {
    result = res;
}

UHMM3SWSearchTaskResult UHMM3SearchLocalTaskResult::getResult() const {
    return result;
}

/*************************************************
* UHMM3SearchLocalTask
*************************************************/

UHMM3SearchLocalTask::UHMM3SearchLocalTask( UHMM3SearchLocalTaskSettings * s )
: LocalTask( "", TaskFlags( TaskFlag_NoRun ) | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled  ), settings( s ), searchTask( NULL ) {
    if( NULL == settings ) {
        setTaskName( tr( "HMM3 search local task" ) );
        setError( tr( "No settings given" ) );
        return;
    }
    setTaskName( tr( "HMM3 search local task with amino and complement translations. Sequence: '%1'. HMM profile:'%2'" ).
        arg( settings->getDNASequence().getName() ).arg( settings->getHMM()->name ) );
}

UHMM3SearchLocalTask::~UHMM3SearchLocalTask() {
    if( NULL != settings ) {
        assert( NULL != settings->getHMM() );
        p7_hmm_Destroy( settings->getHMM() );
        delete settings;
    }
}

void UHMM3SearchLocalTask::prepare() {
    if( hasErrors() ) {
        return;
    }
    assert( NULL != settings && NULL != settings->getHMM() );
    searchTask = new UHMM3SWSearchTask( settings->getHMM(), settings->getDNASequence(), settings->getSearchTaskSettings() );
    addSubTask( searchTask );
}

Task::ReportResult UHMM3SearchLocalTask::report() {
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    if( searchTask->hasErrors() ) {
        setError( searchTask->getError() );
        return ReportResult_Finished;
    }
    result.setResult( searchTask->getResults() );
    return ReportResult_Finished;
}

QString UHMM3SearchLocalTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Alien HMM search task from remote machine.") + "</b></td><td>";
    res += "<tr><td><b>" + tr("HMM profile") + "</b></td><td>" + QString( settings->getHMM()->name ) + "</td></tr>";
    res += "<tr><td><b>" + tr("Sequence") + "</b></td><td>" + settings->getDNASequence().getName() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task finished with error") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    
    int nResults = result.getResult().size();
    res += "<tr><td><b>" + tr("Results count") +  "</b></td><td>" + QString::number( nResults )+ "</td></tr>";
    res += "</table>";
    return res;
}

const LocalTaskResult * UHMM3SearchLocalTask::getResult() const {
    return &result;
}

/*************************************************
* UHMM3RemoteSearchToAnnotationsTask
*************************************************/

UHMM3RemoteSearchToAnnotationsTask::UHMM3RemoteSearchToAnnotationsTask( const QString & hmmf, const DNASequence & seq,
                                                                        const UHMM3SearchTaskSettings & set, 
                                                                        RemoteMachineSettings * m, AnnotationTableObject * ato,
                                                                        const QString & gr, const QString & nm )
: Task( tr( "HMMER3 search task on remote machine" ), TaskFlags_NR_FOSCOE 
       | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ), 
hmmfile( hmmf ), sequence( seq ), searchSettings( set ), machineSettings( m ), 
annotationObj( ato ), agroup( gr ), aname( nm ), loadHmmTask( NULL ), searchTask( NULL ), 
createAnnotationsTask( NULL ), machine( NULL ), hmm( NULL ) {

    checkArgs();
    if( hasErrors() ) {
        return;
    }
    setTaskName( tr( "HMMER3 search task on remote machine. Profile: '%1', sequence: '%2'" ).arg( hmmfile ).arg( sequence.getName() ) );
    loadHmmTask = LoadDocumentTask::getDefaultLoadDocTask( hmmfile );
    if( NULL == loadHmmTask ) {
        setError( L10N::errorOpeningFileRead( hmmfile ) );
        return;
    }
    addSubTask( loadHmmTask );
}

UHMM3RemoteSearchToAnnotationsTask::~UHMM3RemoteSearchToAnnotationsTask() {
    delete machine;
}

void UHMM3RemoteSearchToAnnotationsTask::checkArgs() {
    if( hmmfile.isEmpty() ) {
        setError( L10N::badArgument( tr( "hmm profile path" ) ) );
        return;
    }
    if( sequence.isNull() ) {
        setError( L10N::badArgument( tr( "sequence to search in" ) ) );
        return;
    }
    if( NULL == machineSettings ) {
        setError( L10N::badArgument( tr( "remote machine settings" ) ) );
        return;
    }
    if( NULL == annotationObj.data() ) {
        stateInfo.setError( L10N::badArgument( tr("annotation object") ) );
        return;
    }
    if( agroup.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "annotations group name" ) ) );
        return;
    }
    if( aname.isEmpty() ) {
        stateInfo.setError( L10N::badArgument( tr( "annotations name" ) ) );
        return;
    }
}

QString UHMM3RemoteSearchToAnnotationsTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("HMM search task runned on remote machine.") + "</b></td><td>" + 
        machineSettings->toString() + "</td></tr>";
    res+="<tr><td width=200><b>" + tr("HMM profile used") + "</b></td><td>" + QFileInfo( hmmfile ).absoluteFilePath() + "</td></tr>";
    res+="<tr><td width=200><b>" + tr("Sequence") + "</b></td><td>" + sequence.getName() + "</td></tr>";

    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task was not finished") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    
    res += "<tr><td><b>" + tr("Result annotation table") + "</b></td><td>" + annotationObj->getDocument()->getName() + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation group") + "</b></td><td>" + agroup + "</td></tr>";
    res += "<tr><td><b>" + tr("Result annotation name") +  "</b></td><td>" + aname + "</td></tr>";
    
    int nResults = createAnnotationsTask == NULL ? 0 : createAnnotationsTask->getAnnotations().size();
    res += "<tr><td><b>" + tr("Results count") +  "</b></td><td>" + QString::number( nResults )+ "</td></tr>";
    res += "</table>";
    return res;
}

QList< Task* > UHMM3RemoteSearchToAnnotationsTask::onSubTaskFinished( Task * subTask ) {
    assert( subTask );
    QList< Task* > res;
    propagateSubtaskError();
    if( hasErrors() ) {
        return res;
    }
    
    if( loadHmmTask == subTask ) {
        hmm = UHMM3Utilities::getHmmFromDocument( loadHmmTask->getDocument(), stateInfo );
        if( hasErrors() ) {
            return res;
        }
        assert( NULL != hmm );
        machine = AppContext::getProtocolInfoRegistry()->getProtocolInfo( machineSettings->getProtocolId() )
            ->getRemoteMachineFactory()->createInstance( machineSettings );
        UHMM3SearchLocalTaskSettings localSettings( searchSettings, hmm, sequence );
        searchTask = new RemoteTask( UHMM3SearchLocalTaskFactory::ID, localSettings, machine );
        res << searchTask;
    } else if( searchTask == subTask ) {
        UHMM3SearchLocalTaskResult * result = dynamic_cast< UHMM3SearchLocalTaskResult* >( searchTask->getResult() );
        if( NULL == result ) {
            setError( tr( "remote task didn't create results" ) );
            return res;
        }
        UHMM3SWSearchTaskResult swResult = result->getResult();
        QList< SharedAnnotationData > annotations = UHMM3SWSearchTask::getResultsAsAnnotations( swResult, hmm, aname );
        if( annotations.isEmpty() ) {
            return res;
        }
        
        createAnnotationsTask = new CreateAnnotationsTask( annotationObj, agroup, annotations );
        res << createAnnotationsTask;
    } else if( createAnnotationsTask != subTask ) {
        assert( false && "undefined task finished" );
    }
    
    return res;
}

/*************************************************
* UHMM3SearchLocalTaskFactory
*************************************************/

template<> const QString UHMM3SearchLocalTaskFactory::ID = "HMMER3 search task";

} // GB2
