/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtXml/QDomDocument>

#include <core_api/L10n.h>

#include <distributed_computing/SerializeUtils.h>
#include <workflow_support/WorkflowIOTasks.h>
#include <workflow_support/SchemaSerializer.h>
#include <workflow_support/CoreLibConstants.h>
#include <workflow_support/CoreDataTypes.h>

#include "WorkflowSimpleLocalTask.h"

namespace GB2 {

/***************************************
* WorkflowSimpleLocalTaskSettings
***************************************/

const QString WorkflowSimpleLocalTaskSettings::ITERATIONS_ROOT_ELEMENT_NAME = "root";

WorkflowSimpleLocalTaskSettings::WorkflowSimpleLocalTaskSettings() {
}

WorkflowSimpleLocalTaskSettings::~WorkflowSimpleLocalTaskSettings() {
}

WorkflowSimpleLocalTaskSettings::WorkflowSimpleLocalTaskSettings( const Schema & sc, const QList< Iteration > & it, 
                                                                  const VirtualFileSystem & v, const QString & n )
: schema( sc ), iterations( it ), vfs( v ), outVfsName( n ) {
}

QVariant WorkflowSimpleLocalTaskSettings::serialize() const {
    QVariantList res;
    
    res << SerializeUtils::serializeValue( WorkflowIOUtils::schema2DomDocument( const_cast<Schema*>( &schema ) ).toByteArray() );
    
    QDomDocument xmlDoc(SchemaSerializer::WORKFLOW_DOC);
    QDomElement elem = xmlDoc.createElement( ITERATIONS_ROOT_ELEMENT_NAME );
    xmlDoc.appendChild( elem );
    SchemaSerializer::saveIterations( iterations, elem );
    QByteArray iterationsBytes = xmlDoc.toByteArray();
    res << SerializeUtils::serializeValue( iterationsBytes );
    
    res << SerializeUtils::serializeValue( vfs );
    res << SerializeUtils::serializeValue( outVfsName );
    return res;
}

bool WorkflowSimpleLocalTaskSettings::deserialize( const QVariant & data ) {
    if( !data.canConvert( QVariant::List ) ) {
        return false;
    }
    QVariantList args = data.toList();
    if( SERIALIZED_LIST_SZ != args.size() ) {
        return false;
    }
    
    QByteArray schemaBytes;
    if( !SerializeUtils::deserializeValue( args[0], &schemaBytes ) ) { return false; }
    QDomDocument xml;
    xml.setContent( schemaBytes );
    QMap<ActorId, ActorId> remapping;
    QString err = SchemaSerializer::xml2schema( xml.documentElement(), &schema, remapping );
    if( !err.isEmpty() ) {
        // TODO: error to log
        schema.reset();
        return false;
    }
    
    QByteArray iterationsBytes;
    if( !SerializeUtils::deserializeValue( args[1], &iterationsBytes ) ) { return false; }
    xml.setContent( iterationsBytes );
    SchemaSerializer::readIterations( iterations, xml.documentElement(), remapping );
    
    if( !SerializeUtils::deserializeValue( args[2], &vfs ) ) { return false; }
    if( !SerializeUtils::deserializeValue( args[3], &outVfsName ) ) { return false; }
    return true;
}

Schema WorkflowSimpleLocalTaskSettings::getSchema() const {
    return schema;
}

QList<Iteration> WorkflowSimpleLocalTaskSettings::getIterations() const {
    return iterations;
}

VirtualFileSystem & WorkflowSimpleLocalTaskSettings::getVFS() {
    return vfs;
}

QString WorkflowSimpleLocalTaskSettings::getOutVfsName() const {
    return outVfsName;
}

/***************************************
* WorkflowSimpleLocalTaskResult
***************************************/

WorkflowSimpleLocalTaskResult::WorkflowSimpleLocalTaskResult() {
}

WorkflowSimpleLocalTaskResult::~WorkflowSimpleLocalTaskResult() {
}

WorkflowSimpleLocalTaskResult::WorkflowSimpleLocalTaskResult( const VirtualFileSystem & v ) : vfs( v ) {
}

QVariant WorkflowSimpleLocalTaskResult::serialize() const {
    return SerializeUtils::serializeValue( vfs );
}

bool WorkflowSimpleLocalTaskResult::deserialize( const QVariant & data ) {
    return SerializeUtils::deserializeValue( data, &vfs );
}

void WorkflowSimpleLocalTaskResult::setResult( const VirtualFileSystem & v ) {
    vfs = v;
}

VirtualFileSystem WorkflowSimpleLocalTaskResult::getVfs() const {
    return vfs;
}

/***************************************
* WorkflowSimpleLocalTask
***************************************/

WorkflowSimpleLocalTask::WorkflowSimpleLocalTask( WorkflowSimpleLocalTaskSettings * s ) 
: LocalTask( tr( "Workflow simple local task" ), TaskFlags( TaskFlag_NoRun ) | TaskFlag_ReportingIsSupported | TaskFlag_ReportingIsEnabled ), 
  settings( s ), workflowTask( NULL ) {
    
    if( NULL == settings ) {
        setError( tr( "No settings given" ) );
        return;
    }
    VirtualFileSystemRegistry * vfsReg = AppContext::getVirtualFileSystemRegistry();
    assert( NULL != vfsReg );
    vfsReg->registerFileSystem( &settings->getVFS() );
    vfsReg->registerFileSystem( new VirtualFileSystem( settings->getOutVfsName() ) );
}

WorkflowSimpleLocalTask::~WorkflowSimpleLocalTask() {
    if( NULL != settings ) {
        VirtualFileSystemRegistry * vfsReg = AppContext::getVirtualFileSystemRegistry();
        assert( NULL != vfsReg );
        vfsReg->unregisterFileSystem( settings->getVFS().getId() );
        delete settings;
    }
}

void WorkflowSimpleLocalTask::prepare() {
    if( hasErrors() || stateInfo.cancelFlag ) {
        return;
    }
    assert( NULL != settings );
    workflowTask = new WorkflowRunTask( settings->getSchema(), settings->getIterations() );
    addSubTask( workflowTask );
}

Task::ReportResult WorkflowSimpleLocalTask::report() {
    propagateSubtaskError();
    if( !hasErrors() && ( workflowTask->isCanceled() || isCanceled() )  ) {
        setError( tr( "Workflow local task was canceled" ) );
    }
    if( hasErrors() ) {
        return ReportResult_Finished;
    }
    
    VirtualFileSystemRegistry * vfsReg = AppContext::getVirtualFileSystemRegistry();
    assert( NULL != vfsReg );
    VirtualFileSystem * outVfs = vfsReg->unregisterFileSystem( settings->getOutVfsName() );
    result.setResult( *outVfs );
    delete outVfs;
    
    return ReportResult_Finished;
}

QString WorkflowSimpleLocalTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Alien Workflow simple task from remote machine.") + "</b></td><td>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task finished with error") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    res += "</table>";
    return res;
}

const LocalTaskResult * WorkflowSimpleLocalTask::getResult() const {
    return &result;
}

/***************************************
 * WorkflowSimpleLocalTaskFactory
 ***************************************/

template<> const QString WorkflowSimpleLocalTaskFactory::ID = "Workflow schema simple run task";

/***************************************
* WorkflowRemoteRunTask
***************************************/

const QString OUTPUT_VFS_PREFIX = "OUT_VFS:";

WorkflowRemoteRunTask::WorkflowRemoteRunTask( RemoteMachineSettings * m, const Schema & sc, const QList<Iteration> & its ) 
: Task( tr( "Workflow run task on remote machine" ), TaskFlags_NR_FOSCOE | TaskFlag_ReportingIsSupported
| TaskFlag_ReportingIsEnabled ), machineSettings( m ), machine( NULL ), schema( sc ), iterations( its ), workflowTask( NULL ) {
    
    if( NULL == machineSettings ) {
        setError( L10N::badArgument( tr( "remote machine settings" ) ) );
        return;
    }
}

WorkflowRemoteRunTask::~WorkflowRemoteRunTask() {
    delete machine;
}

void WorkflowRemoteRunTask::preprocessSchema() {
    foreach( Actor * actor, schema.procs ) {
        assert( NULL != actor );
        if( actor->getParameter( CoreLibConstants::URL_IN_ATTR_ID ) != NULL &&
            actor->getParameter( CoreLibConstants::URL_LOCATION_ATTR_ID ) == NULL ) {
            
            actor->addParameter( CoreLibConstants::URL_LOCATION_ATTR_ID, 
                new Attribute( CoreLibConstants::URL_LOCATION_ATTR(), CoreDataTypes::BOOL_TYPE(), false, true ) );
        }
        
        QList<Iteration>::iterator it = iterations.begin();
        ActorId id = actor->getId();
        while( it != iterations.end() ) {
            QList<QString> parameterNames = actor->getParameters().keys();
            foreach( const QString & paramName, parameterNames ) {
                if( !it->cfg[id].contains( paramName ) ) {
                    it->cfg[id][paramName] = actor->getParameter( paramName )->getAttributePureValue();
                }
            }
            ++it;
        }
    }
}

void WorkflowRemoteRunTask::prepare() {
    if( hasErrors() || isCanceled() ) {
        return;
    }
    
    machine = AppContext::getProtocolInfoRegistry()->getProtocolInfo( machineSettings->getProtocolId() )
        ->getRemoteMachineFactory()->createInstance( machineSettings );
    if( NULL == machine ) {
        setError( tr( "Cannot create remote machine from remote machine settings: %1" ).arg( machineSettings->toString() ) );
        return;
    }
    
    preprocessSchema();
    
    QString inputVfsName = QString::number( getTaskId() );
    QString outputVfsName = OUTPUT_VFS_PREFIX + inputVfsName;
    VirtualFileSystem inputVfs( inputVfsName );
    foreach( Actor * actor, schema.procs ) {
        assert( NULL != actor );
        ActorId actorId = actor->getId();
        
        Attribute * urlInAttr = actor->getParameter( CoreLibConstants::URL_IN_ATTR_ID );
        if( NULL != urlInAttr ) {
            QList<Iteration>::iterator it = iterations.begin();
            while( it != iterations.end() ) {
                if( it->cfg[actorId].value( CoreLibConstants::URL_LOCATION_ATTR_ID ).value<bool>() ) { // file located on this computer
                    QString filePath = it->cfg[actorId].value( CoreLibConstants::URL_IN_ATTR_ID ).value<QString>();
                    inputVfs.mapFile( filePath, filePath );
                    it->cfg[actorId][CoreLibConstants::URL_IN_ATTR_ID] = 
                        VirtualFileSystem::URL_PREFIX + inputVfsName + VirtualFileSystem::URL_NAME_SEPARATOR + filePath;
                }
                ++it;
            }
        }
        
        Attribute * urlOutAttr = actor->getParameter( CoreLibConstants::URL_OUT_ATTR_ID );
        if( NULL != urlOutAttr ) {
            assert( NULL == actor->getParameter( CoreLibConstants::URL_LOCATION_ATTR_ID ) );
            QList<Iteration>::iterator it = iterations.begin();
            while( it != iterations.end() ) {
                QVariantMap cfg = it->getParameters( actorId );
                QString filePath = cfg.value( CoreLibConstants::URL_OUT_ATTR_ID ).value<QString>();
                it->cfg[actorId][CoreLibConstants::URL_OUT_ATTR_ID] = 
                    VirtualFileSystem::URL_PREFIX + outputVfsName + VirtualFileSystem::URL_NAME_SEPARATOR + filePath;
                ++it;
            }
        }
    }
    
    WorkflowSimpleLocalTaskSettings localSettings( schema, iterations, inputVfs, outputVfsName );
    workflowTask = new RemoteTask( WorkflowSimpleLocalTaskFactory::ID, localSettings, machine );
    addSubTask( workflowTask );
}

Task::ReportResult WorkflowRemoteRunTask::report() {
    propagateSubtaskError();
    if( hasErrors() || isCanceled() ) {
        return ReportResult_Finished;
    }
    
    WorkflowSimpleLocalTaskResult * result = dynamic_cast< WorkflowSimpleLocalTaskResult* >( workflowTask->getResult() );
    if( NULL == result ) {
        setError( tr( "remote task didn't produce result" ) );
        return ReportResult_Finished;
    }
    
    VirtualFileSystem outputVfs = result->getVfs();
    foreach( const QString & filename, outputVfs.getAllFilenames() ) {
        outputVfs.mapBack( filename, filename );
    }
    return ReportResult_Finished;
}

QString WorkflowRemoteRunTask::generateReport() const {
    QString res;
    res += "<table>";
    res+="<tr><td width=200><b>" + tr("Workflow run task runned on remote machine.") + "</b></td><td>" + 
        machineSettings->toString() + "</td></tr>";
    
    if( hasErrors() || isCanceled() ) {
        res += "<tr><td width=200><b>" + tr("Task finished with error") + "</b></td><td></td></tr>";
        res += "</table>";
        return res;
    }
    res += "</table>";
    return res;
}

} // GB2
