/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtCore/QFile>

#include <core_api/AppContext.h>
#include <core_api/Log.h>
#include <core_api/Settings.h>
#include <core_api/Counter.h>
#include <cmdline/CMDLineRegistry.h>
#include <cmdline/CMDLineUtils.h>
#include <cmdline/CMDLineCoreOptions.h>
#include <distributed_computing/SerializeUtils.h>

#include <workflow/WorkflowEnv.h>

#include "WorkflowDesignerPlugin.h"
#include "WorkflowCMDLineTasks.h"

#define WORKFLOW_CMDLINE_TASK_LOG_CAT "Workflow cmdline tasks"

namespace GB2 {

static LogCategory log( WORKFLOW_CMDLINE_TASK_LOG_CAT );

/*******************************************
* WorkflowRunFromCMDLineBase
*******************************************/
WorkflowRunFromCMDLineBase::WorkflowRunFromCMDLineBase() 
: Task( tr( "Workflow run from cmdline" ), TaskFlag_NoRun ), schema(NULL), optionsStartAt(-1), loadTask(NULL) {
    GCOUNTER(cvar,tvar,"workflow_run_from_cmdline");
    
    CMDLineRegistry * cmdLineRegistry = AppContext::getCMDLineRegistry();

    // try to process schema without 'task' option (it can only be the first one)
    QStringList pureValues = CMDLineRegistryUtils::getPureValues();
    if( !pureValues.isEmpty() ) {
        QString schemaName = pureValues.first();
        processLoadSchemaTask( schemaName, 1 ); // because after program name
    }
    if( loadTask != NULL ) {
        addSubTask( loadTask );
        return;
    }

    // process schema with 'task' option
    int taskOptionIdx = CMDLineRegistryUtils::getParameterIndex( WorkflowDesignerPlugin::RUN_WORKFLOW );
    if(taskOptionIdx != -1) {
        processLoadSchemaTask( cmdLineRegistry->getParameterValue( WorkflowDesignerPlugin::RUN_WORKFLOW, taskOptionIdx ), taskOptionIdx );
    }
    if( loadTask == NULL ) {
        setError( tr( "no task to run" ) );
        return;
    }
    addSubTask( loadTask );
}

void WorkflowRunFromCMDLineBase::processLoadSchemaTask( const QString & schemaName, int optionIdx ) {
    loadTask = prepareLoadSchemaTask( schemaName );
    if( loadTask != NULL ) {
        optionsStartAt = optionIdx + 1;
    }
}

LoadWorkflowTask * WorkflowRunFromCMDLineBase::prepareLoadSchemaTask( const QString & schemaName ) {
    QString pathToSchema = getPathToSchemaFile( schemaName );
    if( pathToSchema.isEmpty() ) {
        log.error( tr( "Cannot find schema: %1" ).arg( schemaName ) );
        return NULL;
    }

    schema = new Schema();
    schema->deepCopy = true;
    return new LoadWorkflowTask( schema, NULL, pathToSchema );
}

WorkflowRunFromCMDLineBase::~WorkflowRunFromCMDLineBase() {
    delete schema;
}

static Actor * findActorByParamAlias( Schema * schema, const QString & alias, QString & attrName ) {
    assert( schema != NULL );
    QList<Actor*> actors;
    foreach( Actor * actor, schema->procs ) {
        assert( actor != NULL );
        if( actor->getParamAliases().values().contains( alias ) ) {
            actors << actor;
        }
    }

    if ( actors.isEmpty() ) {
        return NULL;
    } else if( actors.size() > 1 ) {
        log.error( WorkflowRunFromCMDLineBase::tr( "%1 actors in schema have '%2' alias" ).arg( actors.size() ).arg( alias ) );
    }

    Actor * ret = actors.first();
    attrName = ret->getParamAliases().key( alias );
    return ret;
}

static void setSchemaCMDLineOptions( Schema * schema, int optionsStartAtIdx ) {
    assert( schema != NULL && optionsStartAtIdx > 0 );

    QList<StringPair> parameters = AppContext::getCMDLineRegistry()->getParameters();
    int sz = parameters.size();
    for( int i = optionsStartAtIdx; i < sz; ++i ) {
        const StringPair & param = parameters.at(i);
        if( param.first.isEmpty() ) { // TODO: unnamed parameters not supported yet
            continue;
        }

        QString paramAlias = param.first;
        QString paramName;
        Actor * actor = findActorByParamAlias( schema, paramAlias, paramName );
        if( actor == NULL ) {
            assert( paramName.isEmpty() );
            log.info( WorkflowRunFromCMDLineBase::tr( "alias '%1' not set in schema" ).arg( paramAlias ) );
            continue;
        }

        Attribute * attr = actor->getParameter( paramName );
        if( attr == NULL ) {
            log.error( WorkflowRunFromCMDLineBase::tr( "actor parameter '%1' not found" ).arg( paramName ) );
            continue;
        }

        DataTypeValueFactory * valueFactory = WorkflowEnv::getDataTypeValueFactoryRegistry()->
            getById( attr->getAttributeType()->getId() );
        if( valueFactory == NULL ) {
            log.error( WorkflowRunFromCMDLineBase::tr( "cannot parse value from '%1'" ).arg( param.second ) );
            continue;
        }

        ActorId id = actor->getId();
        bool isOk;
        QVariant value = valueFactory->getValueFromString( param.second, &isOk );
        if(!isOk){
            log.error( WorkflowRunFromCMDLineBase::tr( "Incorrect value for '%1', null or default value passed to schema" ).arg( param.first ) );
            continue;
        }
        QList<Iteration>::iterator it = schema->iterations.begin();
        while( it != schema->iterations.end() ) { // TODO: make different values for different iterations
            it->cfg[id].insert( paramName, value );
            ++it;
        }
    }
}

QList<Task*> WorkflowRunFromCMDLineBase::onSubTaskFinished( Task* subTask ) {
    assert( subTask != NULL );
    QList<Task*> res;

    propagateSubtaskError();
    if( hasErrors() || isCanceled() ) {
        return res;
    }
    assert( !hasErrors() ); // if error, we won't be here

    if( loadTask == subTask ) {
        Schema * schema = loadTask->getSchema();
        assert( schema != NULL );

        setSchemaCMDLineOptions( schema, optionsStartAt );
        if( schema->domain.isEmpty() ) {
            schema->domain = WorkflowEnv::getDomainRegistry()->getAllIds().value(0);
        }
        res << getWorkflowRunTask();
    }
    return res;
}

QString WorkflowRunFromCMDLineBase::getPathToSchemaFile(const QString & name ) const {
    // full path given
    if( QFile::exists( name ) ) {
        return name;
    }
    // search schema in data dir
    QString filenameWithDataPrefix = QString( PATH_PREFIX_DATA ) + ":" + "cmdline/" + name;
    if( QFile::exists( filenameWithDataPrefix ) ) {
        return filenameWithDataPrefix;
    }
    QString filenameWithDataPrefixAndExt = QString( PATH_PREFIX_DATA ) + ":" + "cmdline/" + name + ".uws";
    if( QFile::exists( filenameWithDataPrefixAndExt ) ) {
        return filenameWithDataPrefixAndExt;
    }
    
    // if no such file found -> search name in settings. user saved schemas
    Settings * settings = AppContext::getSettings();
    assert( settings != NULL );
    QVariantMap pathsMap = settings->getValue( SaveWorkflowTask::SCHEMA_PATHS_SETTINGS_TAG ).toMap();
    QString path = pathsMap.value( name ).toString();
    if( QFile::exists( path ) ) {
        return path;
    }
    return QString();
}

/*******************************************
* WorkflowRunFromCMDLineTask
*******************************************/
Task * WorkflowRunFromCMDLineTask::getWorkflowRunTask() const {
    return new WorkflowRunTask(*schema, schema->iterations);
}

/*******************************************
* WorkflowRemoteRunFromCMDLineTask
*******************************************/
WorkflowRemoteRunFromCMDLineTask::WorkflowRemoteRunFromCMDLineTask() {
    CMDLineRegistry * cmdlineReg = AppContext::getCMDLineRegistry();
    assert(cmdlineReg != NULL);
    QString filePath = cmdlineReg->getParameterValue(WorkflowDesignerPlugin::REMOTE_MACHINE);
    if( filePath.isEmpty() ) {
        stateInfo.setError(tr("%1 parameter excpected, but not set").arg(WorkflowDesignerPlugin::REMOTE_MACHINE));
        return;
    }
    
    if( !SerializeUtils::deserializeRemoteMachineSettingsFromFile(filePath, &settings) ) {
        assert(settings == NULL);
        stateInfo.setError(tr("Cannot read remote machine settings from %2").arg(filePath));
        return;
    }
    assert(settings != NULL);
}

Task * WorkflowRemoteRunFromCMDLineTask::getWorkflowRunTask() const {
    assert(settings != NULL);
    return new WorkflowRemoteRunTask( settings, *schema, schema->iterations);
}

} // GB2
