/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WeightMatrixIOWorkers.h"
#include "WeightMatrixWorkers.h"
#include "WeightMatrixIO.h"
#include "PWMSearchDialogController.h"
#include <workflow/Datatype.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/Log.h>
#include <core_api/GUrlUtils.h>
#include <util_tasks/FailTask.h>


/* TRANSLATOR GB2::WeightMatrixIO */

namespace GB2 {
namespace LocalWorkflow {

const QString PWMatrixReader::ACTOR_ID("pwmatrix.read");
const QString PWMatrixWriter::ACTOR_ID("pwmatrix.write");
const QString PFMatrixReader::ACTOR_ID("pfmatrix.read");
const QString PFMatrixWriter::ACTOR_ID("pfmatrix.write");

const QString PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE_ID("pwmatrix.model");
const QString PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE_ID("pfmatrix.model");

DataTypePtr const PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(WEIGHT_MATRIX_MODEL_TYPE_ID, WeightMatrixIO::tr("Weight matrix"), "")));
        startup = false;
    }
    return dtr->getById(WEIGHT_MATRIX_MODEL_TYPE_ID);
}

const Descriptor PWMatrixWorkerFactory::WEIGHT_MATRIX_CATEGORY() {return Descriptor("hweightmatrix", WeightMatrixIO::tr("Weight matrix"), "");}

static LogCategory log(ULOG_CAT_WD);

PWMatrixIOProto::PWMatrixIOProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
    : BusActorPrototype(_desc, _ports, _attrs) {
}

bool PWMatrixIOProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = GUrlUtils::getUncompressedExtension(GUrl(url, GUrl_File));
            if (WeightMatrixIO::WEIGHT_MATRIX_EXT == ext) {
                if (params) {
                    params->insert(urlAttrId, url);
                }
                return true;
            }
        }
    }
    return false;
}

ReadPWMatrixProto::ReadPWMatrixProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
    : PWMatrixIOProto( _desc, _ports, _attrs ) {

        attrs << new Attribute(CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true);
        QMap<QString, PropertyDelegate*> delegateMap;
        delegateMap[CoreLibConstants::URL_IN_ATTR_ID] = new URLDelegate(WeightMatrixIO::getPWMFileFilter(), WeightMatrixIO::WEIGHT_MATRIX_ID, true);
        setEditor(new DelegateEditor(delegateMap));
        setIconPath(":weight_matrix/images/weight_matrix.png");
}

bool ReadPWMatrixProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return PWMatrixIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR_ID );
}

WritePWMatrixProto::WritePWMatrixProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
    : PWMatrixIOProto( _desc, _ports, _attrs ) {
        attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), true );
        attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

        QMap<QString, PropertyDelegate*> delegateMap;
        delegateMap[CoreLibConstants::URL_OUT_ATTR_ID] = new URLDelegate(WeightMatrixIO::getPWMFileFilter(), WeightMatrixIO::WEIGHT_MATRIX_ID, false );
        delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(false);

        setEditor(new DelegateEditor(delegateMap));
        setIconPath(":weight_matrix/images/weight_matrix.png");
        setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR_ID, ports.first()->getId(), CoreLibConstants::URL_SLOT_ID));
        setPortValidator(CoreLibConstants::IN_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT_ID));
}

bool WritePWMatrixProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return PWMatrixIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR_ID );
}

void PWMatrixWorkerFactory::init() 
{
    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    DataTypePtr dt = WEIGHT_MATRIX_MODEL_TYPE();
    {        
        QMap<Descriptor, DataTypePtr> m;
        Descriptor sd(CoreLibConstants::URL_SLOT_ID, WeightMatrixIO::tr("Location"), WeightMatrixIO::tr("Location hint for the target file."));
        m[sd] = CoreDataTypes::STRING_TYPE();
        m[*dt] = dt;
        DataTypePtr t(new MapDataType(Descriptor("write.pwmatrix.content"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::IN_PORT_ID, WeightMatrixIO::tr("Weight matrix"), WeightMatrixIO::tr("Input weight matrix"));
        p << new PortDescriptor(pd, t, true /*input*/);
        Descriptor desc(PWMatrixWriter::ACTOR_ID, WeightMatrixIO::tr("Write weight matrix"), WeightMatrixIO::tr("Saves all input weight matrices to specified location."));
        BusActorPrototype* proto = new WritePWMatrixProto(desc, p, a);
        proto->setPrompter(new PWMatrixWritePrompter());
        r->registerProto(WEIGHT_MATRIX_CATEGORY(), proto);
    }
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::OUT_PORT_ID, WeightMatrixIO::tr("Weight matrix"), WeightMatrixIO::tr("Loaded weight matrices data."));
        p << new PortDescriptor(pd, dt, false /*input*/, true /*multi*/);
        Descriptor desc(PWMatrixReader::ACTOR_ID, WeightMatrixIO::tr("Read weight matrix"), WeightMatrixIO::tr("Reads weight matrices from file(s). The files can be local or Internet URLs."));
        BusActorPrototype* proto = new ReadPWMatrixProto(desc, p, a);
        proto->setPrompter(new PWMatrixReadPrompter());
        r->registerProto(WEIGHT_MATRIX_CATEGORY(), proto);
    }

    PWMatrixBuildWorker::registerProto();
    PWMatrixSearchWorker::registerProto();

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new PWMatrixWorkerFactory(PWMatrixReader::ACTOR_ID));
    localDomain->registerEntry(new PWMatrixWorkerFactory(PWMatrixWriter::ACTOR_ID));
    localDomain->registerEntry(new PWMatrixWorkerFactory(PWMatrixSearchWorker::ACTOR_ID));
    localDomain->registerEntry(new PWMatrixWorkerFactory(PWMatrixBuildWorker::ACTOR_ID));
}

Worker* PWMatrixWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (PWMatrixReader::ACTOR_ID == a->getProto()->getId()) {
        w = new PWMatrixReader(a);
    } 
    else if (PWMatrixWriter::ACTOR_ID == a->getProto()->getId()) {
        w = new PWMatrixWriter(a);
    }
    else if (PWMatrixBuildWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new PWMatrixBuildWorker(a);
    }
    else if (PWMatrixSearchWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new PWMatrixSearchWorker(a);
    }

    return w;    
}

QString PWMatrixReadPrompter::composeRichDoc() {
    return tr("Read model(s) from <u>%1</u>").arg(getURL(CoreLibConstants::URL_IN_ATTR_ID));
}

QString PWMatrixWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE_ID);
    if (!producer) {
        return getURL(CoreLibConstants::URL_OUT_ATTR_ID);
    }
    QString url = getScreenedURL(input, CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID); 
    QString doc = tr("Save the profile(s) from <u>%1</u> to %2.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void PWMatrixReader::init() {
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    urls = DesignerUtils::expandToUrls(actor->getParameter(CoreLibConstants::URL_IN_ATTR_ID)->getAttributeValue<QString>());
    mtype = PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE();
}

Task* PWMatrixReader::tick() {
    Task* t = new PWMatrixReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    tasks.append(t);
    return t;
}

void PWMatrixReader::sl_taskFinished() {
    PWMatrixReadTask* t = qobject_cast<PWMatrixReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    tasks.removeAll(t);
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<PWMatrix>(t->getResult());
            output->put(Message(mtype, v));
        }
        if (urls.isEmpty() && tasks.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded weight matrix from %1").arg(t->getURL()));
    }
}

void PWMatrixWriter::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
}

Task* PWMatrixWriter::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    url = actor->getParameter(CoreLibConstants::URL_OUT_ATTR_ID)->getAttributeValue<QString>();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->getAttributeValue<uint>();
    QVariantMap data = inputMessage.getData().toMap();
    
    PWMatrix model = data.value(PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE_ID).value<PWMatrix>();
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    }
    if (anUrl.isEmpty()) {
        QString err = tr("Unspecified URL for writing weight matrix");
        if (failFast) {
            return new FailTask(err);
        } else {
            log.error(err);
            return NULL;
        }
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = GUrlUtils::prepareFileName(anUrl, count, QStringList(WeightMatrixIO::WEIGHT_MATRIX_EXT));
    } else {
        anUrl = GUrlUtils::ensureFileExt( anUrl, QStringList(WeightMatrixIO::WEIGHT_MATRIX_EXT)).getURLString();
    }
    log.info(tr("Writing weight matrix to %1").arg(anUrl));
    return new PWMatrixWriteTask(anUrl, model, fileMode);
}

//////////////////////////////////////////////////////////////////////////
// PFMatrix workers
//////////////////////////////////////////////////////////////////////////
DataTypePtr const PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE() {
    DataTypeRegistry* dtr = WorkflowEnv::getDataTypeRegistry();
    assert(dtr);
    static bool startup = true;
    if (startup)
    {
        dtr->registerEntry(DataTypePtr(new DataType(FREQUENCY_MATRIX_MODEL_TYPE_ID, WeightMatrixIO::tr("Frequency matrix"), "")));
        startup = false;
    }
    return dtr->getById(FREQUENCY_MATRIX_MODEL_TYPE_ID);
}

const Descriptor PFMatrixWorkerFactory::FREQUENCY_MATRIX_CATEGORY() {return Descriptor("hweightmatrix", WeightMatrixIO::tr("Weight matrix"), "");}

PFMatrixIOProto::PFMatrixIOProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: BusActorPrototype(_desc, _ports, _attrs) {
}

bool PFMatrixIOProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params, const QString & urlAttrId ) const {
    if (md->hasUrls()) {
        QList<QUrl> urls = md->urls();
        if (urls.size() == 1)
        {
            QString url = urls.at(0).toLocalFile();
            QString ext = GUrlUtils::getUncompressedExtension(GUrl(url, GUrl_File));
            if (WeightMatrixIO::FREQUENCY_MATRIX_EXT == ext) {
                if (params) {
                    params->insert(urlAttrId, url);
                }
                return true;
            }
        }
    }
    return false;
}

ReadPFMatrixProto::ReadPFMatrixProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: PFMatrixIOProto( _desc, _ports, _attrs ) {

    attrs << new Attribute(CoreLibConstants::URL_IN_ATTR(), CoreDataTypes::STRING_TYPE(), true);
    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_IN_ATTR_ID] = new URLDelegate(WeightMatrixIO::getPFMFileFilter(), WeightMatrixIO::FREQUENCY_MATRIX_ID, true);
    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":weight_matrix/images/weight_matrix.png");
}

bool ReadPFMatrixProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return PFMatrixIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_IN_ATTR_ID );
}

WritePFMatrixProto::WritePFMatrixProto(const Descriptor& _desc, const QList<PortDescriptor*>& _ports, const QList<Attribute*>& _attrs ) 
: PFMatrixIOProto( _desc, _ports, _attrs ) {
    attrs << new Attribute(CoreLibConstants::URL_OUT_ATTR(), CoreDataTypes::STRING_TYPE(), true );
    attrs << new Attribute(BioActorLibrary::FILE_MODE_ATTR(), CoreDataTypes::NUM_TYPE(), false, SaveDoc_Roll);

    QMap<QString, PropertyDelegate*> delegateMap;
    delegateMap[CoreLibConstants::URL_OUT_ATTR_ID] = new URLDelegate(WeightMatrixIO::getPFMFileFilter(), WeightMatrixIO::FREQUENCY_MATRIX_ID, false );
    delegateMap[BioActorLibrary::FILE_MODE_ATTR_ID] = new FileModeDelegate(false);

    setEditor(new DelegateEditor(delegateMap));
    setIconPath(":weight_matrix/images/weight_matrix.png");
    setValidator(new ScreenedParamValidator(CoreLibConstants::URL_OUT_ATTR_ID, ports.first()->getId(), CoreLibConstants::URL_SLOT_ID));
    setPortValidator(CoreLibConstants::IN_PORT_ID, new ScreenedSlotValidator(CoreLibConstants::URL_SLOT_ID));
}

bool WritePFMatrixProto::isAcceptableDrop(const QMimeData * md, QVariantMap * params ) const {
    return PFMatrixIOProto::isAcceptableDrop( md, params, CoreLibConstants::URL_OUT_ATTR_ID );
}

void PFMatrixWorkerFactory::init() 
{
    WProtoRegistry* r = WorkflowEnv::getProtoRegistry();
    assert(r);
    DataTypePtr dt = FREQUENCY_MATRIX_MODEL_TYPE();
    {        
        QMap<Descriptor, DataTypePtr> m;
        Descriptor sd(CoreLibConstants::URL_SLOT_ID, WeightMatrixIO::tr("Location"), WeightMatrixIO::tr("Location hint for the target file."));
        m[sd] = CoreDataTypes::STRING_TYPE();
        m[*dt] = dt;
        DataTypePtr t(new MapDataType(Descriptor("write.pfmatrix.content"), m));

        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::IN_PORT_ID, WeightMatrixIO::tr("Frequency matrix"), WeightMatrixIO::tr("Input frequency matrix"));
        p << new PortDescriptor(pd, t, true /*input*/);
        Descriptor desc(PFMatrixWriter::ACTOR_ID, WeightMatrixIO::tr("Write frequency matrix"), WeightMatrixIO::tr("Saves all input frequency matrices to specified location."));
        BusActorPrototype* proto = new WritePFMatrixProto(desc, p, a);
        proto->setPrompter(new PFMatrixWritePrompter());
        r->registerProto(FREQUENCY_MATRIX_CATEGORY(), proto);
    }
    {
        QList<PortDescriptor*> p; QList<Attribute*> a;
        Descriptor pd(CoreLibConstants::OUT_PORT_ID, WeightMatrixIO::tr("Frequency matrix"), WeightMatrixIO::tr("Loaded weight matrices data."));
        p << new PortDescriptor(pd, dt, false /*input*/, true /*multi*/);
        Descriptor desc(PFMatrixReader::ACTOR_ID, WeightMatrixIO::tr("Read frequency matrix"), WeightMatrixIO::tr("Reads frequency matrices from file(s). The files can be local or Internet URLs."));
        BusActorPrototype* proto = new ReadPFMatrixProto(desc, p, a);
        proto->setPrompter(new PFMatrixReadPrompter());
        r->registerProto(FREQUENCY_MATRIX_CATEGORY(), proto);
    }

    PFMatrixBuildWorker::registerProto();
    PFMatrixConvertWorker::registerProto();

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new PFMatrixWorkerFactory(PFMatrixReader::ACTOR_ID));
    localDomain->registerEntry(new PFMatrixWorkerFactory(PFMatrixWriter::ACTOR_ID));
    localDomain->registerEntry(new PFMatrixWorkerFactory(PFMatrixBuildWorker::ACTOR_ID));
    localDomain->registerEntry(new PFMatrixWorkerFactory(PFMatrixConvertWorker::ACTOR_ID));
}

Worker* PFMatrixWorkerFactory::createWorker(Actor* a) {
    BaseWorker* w = NULL;
    if (PFMatrixReader::ACTOR_ID == a->getProto()->getId()) {
        w = new PFMatrixReader(a);
    } 
    else if (PFMatrixWriter::ACTOR_ID == a->getProto()->getId()) {
        w = new PFMatrixWriter(a);
    }
    else if (PFMatrixBuildWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new PFMatrixBuildWorker(a);
    } 
    else if (PFMatrixConvertWorker::ACTOR_ID == a->getProto()->getId()) {
        w = new PFMatrixConvertWorker(a);
    }

    return w;    
}

QString PFMatrixReadPrompter::composeRichDoc() {
    return tr("Read model(s) from <u>%1</u>").arg(getURL(CoreLibConstants::URL_IN_ATTR_ID));
}

QString PFMatrixWritePrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE_ID);
    if (!producer) {
        return getURL(CoreLibConstants::URL_OUT_ATTR_ID);
    }
    QString url = getScreenedURL(input, CoreLibConstants::URL_OUT_ATTR_ID, CoreLibConstants::URL_SLOT_ID); 
    QString doc = tr("Save the profile(s) from <u>%1</u> to %2.")
        .arg(producer->getLabel())
        .arg(url);
    return doc;
}

void PFMatrixReader::init() {
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    urls = DesignerUtils::expandToUrls(actor->getParameter(CoreLibConstants::URL_IN_ATTR_ID)->getAttributeValue<QString>());
    mtype = PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE();
}

Task* PFMatrixReader::tick() {
    Task* t = new PFMatrixReadTask(urls.takeFirst());
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    tasks.append(t);
    return t;
}

void PFMatrixReader::sl_taskFinished() {
    PFMatrixReadTask* t = qobject_cast<PFMatrixReadTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    tasks.removeAll(t);
    if (output) {
        if (!t->hasErrors()) {
            QVariant v = qVariantFromValue<PFMatrix>(t->getResult());
            output->put(Message(mtype, v));
        }
        if (urls.isEmpty() && tasks.isEmpty()) {
            output->setEnded();
        }
        log.info(tr("Loaded frequency matrix from %1").arg(t->getURL()));
    }
}

void PFMatrixWriter::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
}

Task* PFMatrixWriter::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    url = actor->getParameter(CoreLibConstants::URL_OUT_ATTR_ID)->getAttributeValue<QString>();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->getAttributeValue<uint>();
    QVariantMap data = inputMessage.getData().toMap();
    PFMatrix model = data.value(PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE_ID).value<PFMatrix>();
    
    QString anUrl = url;
    if (anUrl.isEmpty()) {
        anUrl = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    }
    if (anUrl.isEmpty()) {
        QString err = tr("Unspecified URL for writing frequency matrix");
        if (failFast) {
            return new FailTask(err);
        } else {
            log.error(err);
            return NULL;
        }
    }
    assert(!anUrl.isEmpty());
    int count = ++counter[anUrl];
    if (count != 1) {
        anUrl = GUrlUtils::prepareFileName(anUrl, count, QStringList(WeightMatrixIO::FREQUENCY_MATRIX_EXT));
    } else {
        anUrl = GUrlUtils::ensureFileExt( anUrl, QStringList(WeightMatrixIO::FREQUENCY_MATRIX_EXT)).getURLString();
    }
    log.info(tr("Writing frequency matrix to %1").arg(anUrl));
    return new PFMatrixWriteTask(anUrl, model, fileMode);
}


} //namespace LocalWorkflow
} //namespace GB2