/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WeightMatrixWorkers.h"
#include "WeightMatrixIOWorkers.h"
#include "WeightMatrixPlugin.h"
#include "PWMBuildDialogController.h"

#include <util_weight_matrix/BuiltInPWMConversionAlgorithms.h>
#include <util_weight_matrix/PWMConversionAlgorithmRegistry.h>

#include <workflow/Datatype.h>
#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>

#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <datatype/MAlignment.h>

/* TRANSLATOR GB2::WeightMatrixIO */
/* TRANSLATOR GB2::LocalWorkflow::PWMatrixBuildWorker */

namespace GB2 {
namespace LocalWorkflow {

static const QString ALG_ATTR("a_algo");
static const QString TYPE_ATTR("b_type");

const QString PWMatrixBuildWorker::ACTOR_ID("pwmatrix.build");
const QString PFMatrixBuildWorker::ACTOR_ID("pfmatrix.build");
const QString PFMatrixConvertWorker::ACTOR_ID("pfmatrix.convert");

static LogCategory log(ULOG_CAT_WD);

//////////////////////////////////////////////////////////////////////////
// PWMatrix build worker
//////////////////////////////////////////////////////////////////////////

void PWMatrixBuildWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    QMap<Descriptor, DataTypePtr> m;
    Descriptor id(CoreLibConstants::IN_PORT_ID, PWMatrixBuildWorker::tr("Input alignment"), PWMatrixBuildWorker::tr("Input multiple sequence alignment for building statistical model."));
    m[id] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    DataTypePtr t(new MapDataType(Descriptor("build.pwmatrix.content"), m));

    Descriptor od(CoreLibConstants::OUT_PORT_ID, PWMatrixBuildWorker::tr("Weight matrix"), PWMatrixBuildWorker::tr("Produced statistical model of specified TFBS data."));
    p << new PortDescriptor(id, t, true /*input*/);
    p << new PortDescriptor(od, PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE(), false /*input*/, true /*multi*/);
    
    {
        Descriptor ad(ALG_ATTR, PWMatrixBuildWorker::tr("Weight algorithm"), QApplication::translate("PWMBuildDialog", "algo_tip", 0, QApplication::UnicodeUTF8));
        a << new Attribute(ad, CoreDataTypes::STRING_TYPE(), true, BuiltInPWMConversionAlgorithms::BVH_ALGO);
    }

    {
        Descriptor td(TYPE_ATTR, PWMatrixBuildWorker::tr("Matrix type"), QApplication::translate("PWMBuildDialog", "type_tip", 0, QApplication::UnicodeUTF8));
        a << new Attribute(td, CoreDataTypes::BOOL_TYPE(), true, false /* false = mononucleic, true = dinucleic */);
    }

    Descriptor desc(ACTOR_ID, tr("Build weight matrix"),
        tr("Builds weight matrix. Weight matrices are used for probabilistic recognition of transcription factor binding sites."));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    

    {
        QVariantMap modeMap;
        QStringList algo = AppContext::getPWMConversionAlgorithmRegistry()->getAlgorithmIds();
        foreach (QString curr, algo) {
            modeMap[curr] = QVariant(curr);
        }
        delegates[ALG_ATTR] = new ComboBoxDelegate(modeMap);
    }

    {
        QVariantMap modeMap;
        modeMap[tr("Mononucleic")] = QVariant(false);
        modeMap[tr("Dinucleic")] = QVariant(true);
        delegates[TYPE_ATTR] = new ComboBoxDelegate(modeMap);
    }
    
    proto->setPrompter(new PWMatrixBuildPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":weight_matrix/images/weight_matrix.png");
    WorkflowEnv::getProtoRegistry()->registerProto(PWMatrixWorkerFactory::WEIGHT_MATRIX_CATEGORY(), proto);
}

QString PWMatrixBuildPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* msaProducer = input->getProducer(CoreLibConstants::IN_PORT_ID);

    QString msaName = msaProducer ? tr("For each MSA from <u>%1</u>,").arg(msaProducer->getLabel()) : "";
    QString doc = tr("%1 build weight matrix.")
        .arg(msaName);

    return doc;
}

void PWMatrixBuildWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
}

bool PWMatrixBuildWorker::isReady() {
    return (input && input->hasMessage());
}

Task* PWMatrixBuildWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    mtype = PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE();
    QVariantMap data = inputMessage.getData().toMap();
    PWMatrix model = data.value(PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE_ID).value<PWMatrix>();
    QString url = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    cfg.algo = actor->getParameter(ALG_ATTR)->getAttributeValue<QString>();
    cfg.type = actor->getParameter(TYPE_ATTR)->getAttributeValue<bool>() ? PM_DINUCLEOTIDE : PM_MONONUCLEOTIDE;
    const MAlignment& ma = data.value(CoreLibConstants::IN_PORT_ID).value<MAlignment>();
    Task* t = new PWMatrixBuildTask(cfg, ma);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void PWMatrixBuildWorker::sl_taskFinished() {
    PWMatrixBuildTask* t = qobject_cast<PWMatrixBuildTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    PWMatrix model = t->getResult();
    QVariant v = qVariantFromValue<PWMatrix>(model);
    output->put(Message(mtype, v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Built weight matrix"));
}

bool PWMatrixBuildWorker::isDone() {
    return !input || input->isEnded();
}

//////////////////////////////////////////////////////////////////////////
// PFMatrix build worker
//////////////////////////////////////////////////////////////////////////

void PFMatrixBuildWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    QMap<Descriptor, DataTypePtr> m;
    Descriptor id(CoreLibConstants::IN_PORT_ID, PFMatrixBuildWorker::tr("Input alignment"), PFMatrixBuildWorker::tr("Input multiple sequence alignment for building statistical model."));
    m[id] = BioDataTypes::MULTIPLE_ALIGNMENT_TYPE();
    DataTypePtr t(new MapDataType(Descriptor("build.pfmatrix.content"), m));

    Descriptor od(CoreLibConstants::OUT_PORT_ID, PFMatrixBuildWorker::tr("Frequency matrix"), PFMatrixBuildWorker::tr("Produced statistical model of specified TFBS data."));
    p << new PortDescriptor(id, t, true /*input*/);
    p << new PortDescriptor(od, PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE(), false /*input*/, true /*multi*/);

    {
        Descriptor td(TYPE_ATTR, PWMatrixBuildWorker::tr("Matrix type"), QApplication::translate("PWMBuildDialog", "type_tip", 0, QApplication::UnicodeUTF8));
        a << new Attribute(td, CoreDataTypes::BOOL_TYPE(), true, false /* false = mononucleic, true = dinucleic */);
    }

    Descriptor desc(ACTOR_ID, tr("Build frequency matrix"),
        tr("Builds frequency matrix. Frequency matrices are used for probabilistic recognition of transcription factor binding sites."));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;

    {
        QVariantMap modeMap;
        modeMap[tr("Mononucleic")] = QVariant(false);
        modeMap[tr("Dinucleic")] = QVariant(true);
        delegates[TYPE_ATTR] = new ComboBoxDelegate(modeMap);
    }

    proto->setPrompter(new PFMatrixBuildPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":weight_matrix/images/weight_matrix.png");
    WorkflowEnv::getProtoRegistry()->registerProto(PFMatrixWorkerFactory::FREQUENCY_MATRIX_CATEGORY(), proto);
}

QString PFMatrixBuildPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* msaProducer = input->getProducer(CoreLibConstants::IN_PORT_ID);

    QString msaName = msaProducer ? tr("For each MSA from <u>%1</u>,").arg(msaProducer->getLabel()) : "";
    QString doc = tr("%1 build frequency matrix.")
        .arg(msaName);

    return doc;
}

void PFMatrixBuildWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
}

bool PFMatrixBuildWorker::isReady() {
    return (input && input->hasMessage());
}

Task* PFMatrixBuildWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    mtype = PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE();
    QVariantMap data = inputMessage.getData().toMap();
    PFMatrix model = data.value(PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE_ID).value<PFMatrix>();
    QString url = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    cfg.type = actor->getParameter(TYPE_ATTR)->getAttributeValue<bool>() ? PM_DINUCLEOTIDE : PM_MONONUCLEOTIDE;

    const MAlignment& ma = data.value(CoreLibConstants::IN_PORT_ID).value<MAlignment>();
    Task* t = new PFMatrixBuildTask(cfg, ma);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void PFMatrixBuildWorker::sl_taskFinished() {
    PFMatrixBuildTask* t = qobject_cast<PFMatrixBuildTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    PFMatrix model = t->getResult();
    QVariant v = qVariantFromValue<PFMatrix>(model);
    output->put(Message(mtype, v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Built frequency matrix"));
}

bool PFMatrixBuildWorker::isDone() {
    return !input || input->isEnded();
}

//////////////////////////////////////////////////////////////////////////
// PFMatrix convert worker
//////////////////////////////////////////////////////////////////////////

void PFMatrixConvertWorker::registerProto() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    QMap<Descriptor, DataTypePtr> m;
    Descriptor id(CoreLibConstants::IN_PORT_ID, PFMatrixConvertWorker::tr("Weight matrix"), PFMatrixConvertWorker::tr("Frequency matrix to convert."));
    m[id] = PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE();
    DataTypePtr t(new MapDataType(Descriptor("convert.pfmatrix.content"), m));

    Descriptor od(CoreLibConstants::OUT_PORT_ID, PFMatrixConvertWorker::tr("Weight matrix"), PFMatrixConvertWorker::tr("Produced statistical model of specified TFBS data."));
    p << new PortDescriptor(id, t, true /*input*/);
    p << new PortDescriptor(od, PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE(), false /*input*/, true /*multi*/);

    {
        Descriptor ad(ALG_ATTR, PWMatrixBuildWorker::tr("Weight algorithm"), QApplication::translate("PWMBuildDialog", "algo_tip", 0, QApplication::UnicodeUTF8));
        a << new Attribute(ad, CoreDataTypes::STRING_TYPE(), true, BuiltInPWMConversionAlgorithms::BVH_ALGO);
    }

    {
        Descriptor td(TYPE_ATTR, PWMatrixBuildWorker::tr("Matrix type"), QApplication::translate("PWMBuildDialog", "type_tip", 0, QApplication::UnicodeUTF8));
        a << new Attribute(td, CoreDataTypes::BOOL_TYPE(), true, false /* false = mononucleic, true = dinucleic */);
    }

    Descriptor desc(ACTOR_ID, tr("Convert frequency matrix"),
        tr("Converts frequency matrix to weight matrix. Weight matrices are used for probabilistic recognition of transcription factor binding sites."));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);
    QMap<QString, PropertyDelegate*> delegates;    

    {
        QVariantMap modeMap;
        QStringList algo = AppContext::getPWMConversionAlgorithmRegistry()->getAlgorithmIds();
        foreach (QString curr, algo) {
            modeMap[curr] = QVariant(curr);
        }
        delegates[ALG_ATTR] = new ComboBoxDelegate(modeMap);
    }

    {
        QVariantMap modeMap;
        modeMap[tr("Mononucleic")] = QVariant(false);
        modeMap[tr("Dinucleic")] = QVariant(true);
        delegates[TYPE_ATTR] = new ComboBoxDelegate(modeMap);
    }

    proto->setPrompter(new PFMatrixConvertPrompter());
    proto->setEditor(new DelegateEditor(delegates));
    proto->setIconPath(":weight_matrix/images/weight_matrix.png");
    WorkflowEnv::getProtoRegistry()->registerProto(PFMatrixWorkerFactory::FREQUENCY_MATRIX_CATEGORY(), proto);
}

QString PFMatrixConvertPrompter::composeRichDoc() {
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* msaProducer = input->getProducer(CoreLibConstants::IN_PORT_ID);

    QString msaName = msaProducer ? tr("For each frequency matrix from <u>%1</u>,").arg(msaProducer->getLabel()) : "";
    QString doc = tr("%1 build weight matrix.")
        .arg(msaName);

    return doc;
}

void PFMatrixConvertWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
}

bool PFMatrixConvertWorker::isReady() {
    return (input && input->hasMessage());
}

Task* PFMatrixConvertWorker::tick() {
    Message inputMessage = getMessageAndSetupScriptValues(input);
    mtype = PFMatrixWorkerFactory::FREQUENCY_MATRIX_MODEL_TYPE();
    QVariantMap data = inputMessage.getData().toMap();
    PWMatrix model = data.value(PWMatrixWorkerFactory::WEIGHT_MATRIX_MODEL_TYPE_ID).value<PWMatrix>();
    QString url = data.value(CoreLibConstants::URL_SLOT_ID).toString();
    cfg.algo = actor->getParameter(ALG_ATTR)->getAttributeValue<QString>();
    cfg.type = actor->getParameter(TYPE_ATTR)->getAttributeValue<bool>() ? PM_DINUCLEOTIDE : PM_MONONUCLEOTIDE;
    const PFMatrix& ma = data.value(CoreLibConstants::IN_PORT_ID).value<PFMatrix>();
    Task* t = new PWMatrixBuildTask(cfg, ma);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    return t;
}

void PFMatrixConvertWorker::sl_taskFinished() {
    PWMatrixBuildTask* t = qobject_cast<PWMatrixBuildTask*>(sender());
    if (t->getState() != Task::State_Finished) return;
    PWMatrix model = t->getResult();
    QVariant v = qVariantFromValue<PWMatrix>(model);
    output->put(Message(mtype, v));
    if (input->isEnded()) {
        output->setEnded();
    }
    log.info(tr("Built weight matrix"));
}

bool PFMatrixConvertWorker::isDone() {
    return !input || input->isEnded();
}

} //namespace LocalWorkflow
} //namespace GB2
