/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "PWMBuildDialogController.h"

#include "WeightMatrixPlugin.h"
#include "WeightMatrixIO.h"

#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <core_api/DocumentModel.h>
#include <core_api/DNAAlphabet.h>
#include <core_api/Settings.h>
#include <core_api/Counter.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/MAlignmentObject.h>
#include <gobjects/DNASequenceObject.h>

#include <document_format/DocumentFormatUtils.h>
#include <core_api/DocumentUtils.h>

#include <util_gui/DialogUtils.h>
#include <util_tasks/LoadDocumentTask.h>

#include <util_weight_matrix/DIProperties.h>
#include <util_weight_matrix/PWMConversionAlgorithm.h>
#include <util_weight_matrix/PWMConversionAlgorithmRegistry.h>

#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#define SETTINGS_ROOT   QString("plugin_weight_matrix/")

namespace GB2 {

PWMBuildDialogController::PWMBuildDialogController(QWidget* w) 
: QDialog(w), logoArea(NULL)
{
    task = NULL;
    setupUi(this);

    QStringList algo = AppContext::getPWMConversionAlgorithmRegistry()->getAlgorithmIds();
    algorithmCombo->addItems(algo);

    this->resize(this->width(), this->minimumHeight());

    connect(inputButton, SIGNAL(clicked()), SLOT(sl_inFileButtonClicked()));
    connect(outputButton, SIGNAL(clicked()), SLOT(sl_outFileButtonClicked()));
    connect(okButton, SIGNAL(clicked()), SLOT(sl_okButtonClicked()));
    connect(weightButton, SIGNAL(toggled(bool)), SLOT(sl_matrixTypeChanged(bool)));
}


void PWMBuildDialogController::sl_inFileButtonClicked() {
    LastOpenDirHelper lod;
    lod.url = QFileDialog::getOpenFileName(this, tr("Select file with alignment"), lod, 
        DialogUtils::prepareDocumentsFileFilterByObjType(GObjectTypes::MULTIPLE_ALIGNMENT, true).append("\n").append(
        DialogUtils::prepareDocumentsFileFilterByObjType(GObjectTypes::SEQUENCE, false)));
    if (lod.url.isEmpty()) {
        return;
    }

    QString inFile = QFileInfo(lod.url).absoluteFilePath();
    inputEdit->setText(inFile);
    
    QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(inFile);
    if (formats.isEmpty()) {
        return;
    }
    DocumentFormat* format = formats.first();
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(inFile));
    TaskStateInfo ti;
    QVariantMap hints;
    Document *doc = format->loadDocument(iof, inFile, ti, hints);
    if (ti.hasErrors()) {
        return;
    }

    assert (doc != NULL);

    int bitsize = 30;
    int logoheight = 150;

    QList<GObject*> mobjs = doc->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
    if (!mobjs.isEmpty()) {
        MAlignmentObject* mobj =  qobject_cast<MAlignmentObject*>(mobjs.first());
        MAlignment ma = mobj->getMAlignment();
        if (ma.getLength() < 50) {
            AlignmentLogoSettings logoSettings(ma);
            logoSettings.bitSize = bitsize;
            logoWidget->resize(logoWidget->width(), logoheight);
            logoWidget->setMinimumHeight(logoheight);
            if (logoArea != NULL) {
                logoArea->replaceSettings(logoSettings);   
            } else {
                logoArea = new AlignmentLogoRenderArea(logoSettings, logoWidget);
            }
            logoArea->repaint();
        }
    } else {
        mobjs = doc->findGObjectByType(GObjectTypes::SEQUENCE);
        if (!mobjs.isEmpty()) {
            QList<MAlignmentRow> rows;
            foreach (GObject* obj, mobjs) {
                DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
                if (dnaObj->getAlphabet()->getType() != DNAAlphabet_NUCL) {
                    ti.setError(  tr("Wrong sequence alphabet") );
                }
                rows.append(MAlignmentRow(dnaObj->getDNASequence().getName(), dnaObj->getSequence()));
            }
            DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(mobjs.first());
            MAlignment ma(dnaObj->getDNASequence().getName(), dnaObj->getAlphabet(), rows);
            if (ma.getLength() < 50) {
                AlignmentLogoSettings logoSettings(ma);
                logoSettings.bitSize = bitsize;
                logoWidget->resize(logoWidget->width(), logoheight);
                logoWidget->setMinimumHeight(logoheight);
                if (logoArea != NULL) {
                    logoArea->replaceSettings(logoSettings);   
                } else {
                    logoArea = new AlignmentLogoRenderArea(logoSettings, logoWidget);
                }
                logoArea->repaint();
            }
        } else {
            PFMatrix pfm = WeightMatrixIO::readPFMatrix(iof, lod.url, ti);
            if (ti.hasErrors()) {
                return;
            }
            int len = pfm.getLength();
            if (len == 0) {
                ti.setError(tr("Zero length matrix is not allowed"));
                return;
            }
            int size = 0;
            
            for (int i = 0, n = pfm.getType() == PFM_MONONUCLEOTIDE ? 4 : 16; i < n; i++) {
                size += pfm.getValue(i, 0);
            }
            QList<MAlignmentRow> rows;
            for (int i = 0; i < size; i++) {
                QByteArray arr;
                for (int j = 0; j < len; j++) {
                    int row = 0;
                    int sum = i;
                    while (sum >= pfm.getValue(row, j)) {
                        sum -= pfm.getValue(row, j);
                        row++;
                    }   
                    if (pfm.getType() == PFM_MONONUCLEOTIDE) {
                        arr.append(DiProperty::fromIndex(row));
                    } else {
                        arr.append(DiProperty::fromIndexHi(row));
                        if (j == len - 1) {
                            arr.append(DiProperty::fromIndexLo(row));
                        }
                    }
                }
                rows.append(MAlignmentRow("", arr));
            }
            DNAAlphabet* al = AppContext::getDNAAlphabetRegistry()->findById(BaseDNAAlphabetIds::NUCL_DNA_DEFAULT);
            MAlignment ma(QString("Temporary alignment"), al, rows);
            if (ma.getLength() < 50) {
                AlignmentLogoSettings logoSettings(ma);
                logoSettings.bitSize = bitsize;
                logoWidget->resize(logoWidget->width(), logoheight);
                logoWidget->setMinimumHeight(logoheight);
                if (logoArea != NULL) {
                    logoArea->replaceSettings(logoSettings);   
                } else {
                    logoArea = new AlignmentLogoRenderArea(logoSettings, logoWidget);
                }
                logoArea->repaint();
            }
            
        }
    }
}

void PWMBuildDialogController::sl_outFileButtonClicked() {
    LastOpenDirHelper lod(WeightMatrixIO::WEIGHT_MATRIX_ID);
    if (frequencyButton->isChecked()) {
        lod.url = QFileDialog::getSaveFileName(this, tr("Select file to save frequency matrix to..."), lod, WeightMatrixIO::getPFMFileFilter(false));
    } else {
        lod.url = QFileDialog::getSaveFileName(this, tr("Select file to save weight matrix to..."), lod, WeightMatrixIO::getPWMFileFilter(false));
    }
    if (lod.url.isEmpty()) {
        return;
    }
    outputEdit->setText(QFileInfo(lod.url).absoluteFilePath());
}

void PWMBuildDialogController::sl_matrixTypeChanged(bool matrixType) {
    QStringList nameParts = outputEdit->text().split(".");
    if (matrixType) {
        for (int i = nameParts.length() - 1; i >= 0; --i) {
            if (nameParts[i] == WeightMatrixIO::FREQUENCY_MATRIX_EXT) {
                nameParts[i] = WeightMatrixIO::WEIGHT_MATRIX_EXT;
                break;
            }
        }
    } else {
        for (int i = nameParts.length() - 1; i >= 0; --i) {
            if (nameParts[i] == WeightMatrixIO::WEIGHT_MATRIX_EXT) {
                nameParts[i] = WeightMatrixIO::FREQUENCY_MATRIX_EXT;
                break;
            }
        }
    }
    QString name = nameParts.join(".");
    if (QFile::exists(name)) {
        if (QMessageBox::No == QMessageBox::question(this, tr("Overwrite existing file"),
            tr("File with this name already exists.\nDo you want to write over this file?"), QMessageBox::Yes | QMessageBox::No)) {
            emit sl_outFileButtonClicked();
            return;
        }
    }
    outputEdit->setText(name);
}

void PWMBuildDialogController::sl_okButtonClicked() {
    if (task != NULL) {
        accept(); //go to background
        return;
    }

    // try prepare model

    PMBuildSettings s;

    QString errMsg;

    QString inFile = inputEdit->text();
    if (inFile.isEmpty() || !QFile::exists(inFile)) {
        statusLabel->setText(tr("Illegal input file name"));
        inputEdit->setFocus();
        return;
    }

    QString outFile = outputEdit->text();
    if (outFile.isEmpty()) {
        statusLabel->setText(tr("Illegal output file name"));
        outputEdit->setFocus();
        return;
    }

    if (frequencyButton->isChecked()) {
        s.target = FREQUENCY_MATRIX;
    } else {
        s.target = WEIGHT_MATRIX;
    }

    //save settings
    //AppContext::getSettings()->setValue(SETTINGS_ROOT + WEIGHT_ALG, weightAlgCombo->currentIndex());

    if (mononucleicButton->isChecked()) {
        s.type = PM_MONONUCLEOTIDE;
    } else {
        s.type = PM_DINUCLEOTIDE;
    }

    // run task
    if (frequencyButton->isChecked()) {
        task = new PFMatrixBuildToFileTask(inFile, outFile, s);
    } else {
        s.algo = algorithmCombo->currentText();
        task = new PWMatrixBuildToFileTask(inFile, outFile, s);
    }
    connect(task, SIGNAL(si_stateChanged()), SLOT(sl_onStateChanged()));
    connect(task, SIGNAL(si_progressChanged()), SLOT(sl_onProgressChanged()));
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
    statusLabel->setText(tr("Counting frequency statistics"));

    //update buttons
    okButton->setText(tr("Hide"));
    cancelButton->setText(tr("Cancel"));
}


void PWMBuildDialogController::sl_onStateChanged() {
    Task* t = qobject_cast<Task*>(sender());
    assert(task!=NULL);
    if (task != t || t->getState() != Task::State_Finished) {
        return;
    }
    task->disconnect(this);
    const TaskStateInfo& si = task->getStateInfo();
    if (si.hasErrors()) {
        statusLabel->setText(tr("Build finished with errors: %1").arg(si.getError()));
        lastURL = "";
    } else if (task->isCanceled()) {
        statusLabel->setText(tr("Build canceled"));
        lastURL = "";
    } else {
        statusLabel->setText(tr("Build finished successfuly"));
        lastURL = outputEdit->text();
    }
    okButton->setText(tr("Start"));
    cancelButton->setText(tr("Close"));
    task = NULL;
}

void PWMBuildDialogController::sl_onProgressChanged() {
    assert(task==sender());
    statusLabel->setText(tr("Running state %1 progress %2%").arg(task->getStateInfo().getStateDesc()).arg(task->getProgress()));
}

void PWMBuildDialogController::reject() {
    if (task!=NULL) {
        task->cancel();
    }
    if (lastURL != "") {
        QDialog::accept();    
    } else {
        QDialog::reject();
    }
}


//////////////////////////////////////////////////////////////////////////
// tasks

PFMatrixBuildTask::PFMatrixBuildTask(const PMBuildSettings& s, const MAlignment& ma) 
: Task (tr("Build frequency matrix"), TaskFlag_None), settings(s), ma(ma)
{
    GCOUNTER( cvar, tvar, "PFMatrixBuildTask" );
    tpm = Task::Progress_Manual;
}

void PFMatrixBuildTask::run() {
    if (ma.hasGaps()) {
        stateInfo.setError(  tr("Alignment has gaps") );
        return;
    }
    if (ma.isEmpty()) {
        stateInfo.setError(  tr("Alignment is empty") );
        return;
    }
    if (!ma.getAlphabet()->isNucleic()) {
        stateInfo.setError(  tr("Alignment is not nucleic") );
        return;
    }
    stateInfo.setStateDesc( tr("Calculating frequencies of nucleotids") );
    if (settings.type == PM_MONONUCLEOTIDE) {
        m = PFMatrix(ma, PFM_MONONUCLEOTIDE);
    } else {
        m = PFMatrix(ma, PFM_DINUCLEOTIDE);
    }
    stateInfo.progress+=50;
    if (stateInfo.hasErrors() || isCanceled()) {
        return;
    }
    return;
}

PFMatrixBuildToFileTask::PFMatrixBuildToFileTask(const QString& inFile, const QString& _outFile, const PMBuildSettings& s) 
: Task (tr("Build weight matrix"), TaskFlag_NoRun), loadTask(NULL), buildTask(NULL), outFile(_outFile), settings(s)
{
    tpm = Task::Progress_SubTasksBased;
    
    DocumentFormatConstraints c;
    c.checkRawData = true;
    c.supportedObjectTypes += GObjectTypes::MULTIPLE_ALIGNMENT;
    c.supportedObjectTypes += GObjectTypes::SEQUENCE;
    c.rawData = BaseIOAdapters::readFileHeader(inFile);
    QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(inFile);
    if (formats.isEmpty()) {
        stateInfo.setError(  tr("Input format error") );
        return;
    }
    DocumentFormatId format = formats.first()->getFormatId();
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(inFile));
    loadTask = new LoadDocumentTask(format, inFile, iof);
    loadTask->setSubtaskProgressWeight(0.03F);
    stateInfo.progress = 0;
    stateInfo.setStateDesc(tr("Loading alignment"));
    addSubTask(loadTask);
}

QList<Task*> PFMatrixBuildToFileTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    if (isCanceled()) {
        return res;
    }
    if (subTask->getStateInfo().hasErrors()) {
        stateInfo.setError(  subTask->getStateInfo().getError() );
        return res;
    }
    if (subTask == loadTask) {
        setUseDescriptionFromSubtask(true);
        Document* d = loadTask->getDocument();
        assert(d != NULL);
        QList<GObject*> mobjs = d->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
        if (!mobjs.isEmpty()) {
            MAlignmentObject* mobj =  qobject_cast<MAlignmentObject*>(mobjs.first());
            MAlignment ma = mobj->getMAlignment();
            buildTask = new PFMatrixBuildTask(settings, ma);
            res.append(buildTask);
        } else {
            mobjs = d->findGObjectByType(GObjectTypes::SEQUENCE);
            if (!mobjs.isEmpty()) {
                QList<MAlignmentRow> rows;
                foreach (GObject* obj, mobjs) {
                    DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
                    if (dnaObj->getAlphabet()->getType() != DNAAlphabet_NUCL) {
                        stateInfo.setError(  tr("Wrong sequence alphabet") );
                    }
                    rows.append(MAlignmentRow(dnaObj->getDNASequence().getName(), dnaObj->getSequence()));
                }
                DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(mobjs.first());
                QString baseName = d->getURL().baseFileName();
                MAlignment ma(baseName, dnaObj->getAlphabet(), rows);
                buildTask = new PFMatrixBuildTask(settings, ma);
                res.append(buildTask);
            } else {
                stateInfo.setError(  tr("No alignments or sequences found") );
            }
        }
    } else if (subTask == buildTask) {
        Task* t = new PFMatrixWriteTask(outFile, buildTask->getResult());
        t->setSubtaskProgressWeight(0);
        res.append(t);
    }
    return res;
}

PWMatrixBuildTask::PWMatrixBuildTask(const PMBuildSettings& s, const MAlignment& ma) 
: Task (tr("Build weight matrix"), TaskFlag_None), settings(s), ma(ma)
{
    GCOUNTER( cvar, tvar, "PWMatrixBuildTask" );
    tpm = Task::Progress_Manual;
}

PWMatrixBuildTask::PWMatrixBuildTask(const PMBuildSettings& s, const PFMatrix& ma) 
: Task (tr("Build weight matrix"), TaskFlag_None), settings(s), tempMatrix(ma)
{
    GCOUNTER( cvar, tvar, "PWMatrixBuildTask" );
    tpm = Task::Progress_Manual;
}

void PWMatrixBuildTask::run() {
    if (tempMatrix.getLength() > 0) {
        if (settings.type == PM_DINUCLEOTIDE && tempMatrix.getType() == PFM_MONONUCLEOTIDE) {
            stateInfo.setError( tr("Can't convert mononucleotide matrix to dinucleotide one"));
            return;
        }
        if (settings.type == PM_MONONUCLEOTIDE && tempMatrix.getType() == PFM_DINUCLEOTIDE) {
            tempMatrix = PFMatrix::convertDi2Mono(tempMatrix);
        }
        stateInfo.progress+=40;
        if (stateInfo.hasErrors() || isCanceled()) {
            return;
        }
    } else {
        if (ma.hasGaps()) {
            stateInfo.setError(  tr("Alignment has gaps") );
            return;
        }
        if (ma.isEmpty()) {
            stateInfo.setError(  tr("Alignment is empty") );
            return;
        }
        if (!ma.getAlphabet()->isNucleic()) {
            stateInfo.setError(  tr("Alignment is not nucleic") );
            return;
        }

        if (settings.type == PM_MONONUCLEOTIDE) {
            tempMatrix = PFMatrix(ma, PFM_MONONUCLEOTIDE);
        } else {
            tempMatrix = PFMatrix(ma, PFM_DINUCLEOTIDE);
        }
        stateInfo.progress+=40;
        if (stateInfo.hasErrors() || isCanceled()) {
            return;
        }
    }
    PWMConversionAlgorithmFactory* factory = AppContext::getPWMConversionAlgorithmRegistry()->getAlgorithmFactory(settings.algo);
    PWMConversionAlgorithm* algo = factory->createAlgorithm();

    m = algo->convert(tempMatrix);
    stateInfo.progress+=40;
    return;
}

PWMatrixBuildToFileTask::PWMatrixBuildToFileTask(const QString& inFile, const QString& _outFile, const PMBuildSettings& s) 
: Task (tr("Build weight matrix"), TaskFlag_NoRun), loadTask(NULL), buildTask(NULL), outFile(_outFile), settings(s)
{
    tpm = Task::Progress_SubTasksBased;
    
    DocumentFormatConstraints c;
    c.checkRawData = true;
    c.supportedObjectTypes += GObjectTypes::MULTIPLE_ALIGNMENT;
    c.supportedObjectTypes += GObjectTypes::SEQUENCE;
    c.rawData = BaseIOAdapters::readFileHeader(inFile);
    QList<DocumentFormat*> formats = DocumentFormatUtils::detectFormat(inFile);
    if (formats.isEmpty()) {
        stateInfo.setError(  tr("Input format error") );
        return;
    }
    DocumentFormatId format = formats.first()->getFormatId();
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(inFile));
    loadTask = new LoadDocumentTask(format, inFile, iof);
    loadTask->setSubtaskProgressWeight(0.03F);
    stateInfo.progress = 0;
    stateInfo.setStateDesc(tr("Loading alignment"));
    addSubTask(loadTask);
}

QList<Task*> PWMatrixBuildToFileTask::onSubTaskFinished(Task* subTask) {
    QList<Task*> res;
    if (isCanceled()) {
        return res;
    }
    if (subTask->getStateInfo().hasErrors()) {
        stateInfo.setError(  subTask->getStateInfo().getError() );
        return res;
    }
    if (subTask == loadTask) {
        setUseDescriptionFromSubtask(true);
        Document* d = loadTask->getDocument();
        assert(d != NULL);
        QList<GObject*> mobjs = d->findGObjectByType(GObjectTypes::MULTIPLE_ALIGNMENT);
        if (!mobjs.isEmpty()) {
            MAlignmentObject* mobj =  qobject_cast<MAlignmentObject*>(mobjs.first());
            MAlignment ma = mobj->getMAlignment();
            buildTask = new PWMatrixBuildTask(settings, ma);
            res.append(buildTask);
        } else {
            mobjs = d->findGObjectByType(GObjectTypes::SEQUENCE);
            if (!mobjs.isEmpty()) {
                QList<MAlignmentRow> rows;
                foreach (GObject* obj, mobjs) {
                    DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(obj);
                    if (dnaObj->getAlphabet()->getType() != DNAAlphabet_NUCL) {
                        stateInfo.setError(  tr("Wrong sequence alphabet") );
                    }
                    rows.append(MAlignmentRow(dnaObj->getDNASequence().getName(), dnaObj->getSequence()));
                }
                DNASequenceObject* dnaObj = qobject_cast<DNASequenceObject*>(mobjs.first());
                QString baseName = d->getURL().baseFileName();
                MAlignment ma(baseName, dnaObj->getAlphabet(), rows);
                buildTask = new PWMatrixBuildTask(settings, ma);
                res.append(buildTask);
            } else {
                stateInfo.setError(  tr("No alignments or sequences found") );
            }
        }
    } else if (subTask == buildTask) {
        Task* t = new PWMatrixWriteTask(outFile, buildTask->getResult());
        t->setSubtaskProgressWeight(0);
        res.append(t);
    }
    return res;
}

}//namespace
