/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <workflow/IntegralBusModel.h>
#include <workflow/WorkflowEnv.h>
#include <workflow/WorkflowRegistry.h>
#include <workflow_support/CoreDataTypes.h>
#include <workflow_library/BioDatatypes.h>
#include <workflow_library/BioActorLibrary.h>
#include <workflow_support/DelegateEditors.h>
#include <workflow_support/CoreLibConstants.h>
#include <core_api/Log.h>
#include <core_api/GUrl.h>
#include <gobjects/DNASequenceObject.h>
#include <util_gui/DialogUtils.h>

#include "ImportQualityScoresWorker.h"


/* TRANSLATOR GB2::LocalWorkflow::ImportPhredQualityWorker */

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

/*************************************
 * ImportPhredQualityWorkerFactory
 *************************************/
const QString ImportPhredQualityWorkerFactory::ACTOR_ID("import.phred.quality");
const QString QUALITY_TYPE_ATTR("qual-type");
const QString QUALITY_URL_ATTR("qual-file-url");

void ImportPhredQualityWorkerFactory::init() {
    QList<PortDescriptor*> p; QList<Attribute*> a;
    Descriptor ind(CoreLibConstants::IN_PORT_ID, ImportPhredQualityWorker::tr("DNA sequences"), ImportPhredQualityWorker::tr("The PHRED scores will be imported to these sequences"));
    Descriptor oud(CoreLibConstants::OUT_PORT_ID, ImportPhredQualityWorker::tr("DNA sequences with imported qualities"), ImportPhredQualityWorker::tr("These sequences have quality scores."));
    p << new PortDescriptor(ind, BioDataTypes::DNA_SEQUENCE_TYPE(), true /*input*/);
    p << new PortDescriptor(oud, BioDataTypes::DNA_SEQUENCE_TYPE(), false /*input*/, true /*multi*/);
    Descriptor qualUrl(QUALITY_URL_ATTR, ImportPhredQualityWorker::tr("PHRED input"), 
         ImportPhredQualityWorker::tr("Path to file with PHRED quality scores."));
    Descriptor qualType(QUALITY_TYPE_ATTR, ImportPhredQualityWorker::tr("Quality format"), 
        ImportPhredQualityWorker::tr("Choose format to encode quality scores."));

    a << new Attribute(qualUrl, CoreDataTypes::STRING_TYPE(), true /*required*/, QString());
    a << new Attribute(qualType, CoreDataTypes::STRING_TYPE(), false/*required*/, DNAQuality::getDNAQualityNameByType(DNAQualityType_Sanger) );    

    Descriptor desc(ACTOR_ID, ImportPhredQualityWorker::tr("Import PHRED qualities"), 
        ImportPhredQualityWorker::tr("Add corresponding PHRED quality scores to the sequences.\nYou can use this worker to convert .fasta and .qual pair to fastq format"));
    ActorPrototype* proto = new BusActorPrototype(desc, p, a);

    QMap<QString, PropertyDelegate*> delegates;    

    delegates[QUALITY_URL_ATTR] = new URLDelegate(DialogUtils::prepareDocumentsFileFilter(true), QString(), true);
    
    QVariantMap m;
    QStringList qualFormats = DNAQuality::getDNAQualityTypeNames();
    foreach( const QString& name, qualFormats  ) {
        m[name] = name;
    }
    delegates[QUALITY_TYPE_ATTR] = new ComboBoxDelegate(m);
    
    proto->setEditor(new DelegateEditor(delegates));
    proto->setPrompter(new ImportPhredQualityPrompter());
    WorkflowEnv::getProtoRegistry()->registerProto(BioActorLibrary::CATEGORY_BASIC(), proto);

    DomainFactory* localDomain = WorkflowEnv::getDomainRegistry()->getById(LocalDomainFactory::ID);
    localDomain->registerEntry(new ImportPhredQualityWorkerFactory());
}

/*************************************
 * ImportPhredQualityPrompter
 *************************************/
QString ImportPhredQualityPrompter::composeRichDoc() {
    
    BusPort* input = qobject_cast<BusPort*>(target->getPort(CoreLibConstants::IN_PORT_ID));
    Actor* producer = input->getProducer(CoreLibConstants::IN_PORT_ID);
    QString producerName = producer ? tr(" from <u>%1</u>").arg(producer->getLabel()) : "";
    QString qualUrl = getParameter(QUALITY_URL_ATTR).toString();
    QString qualSeq = (qualUrl.isEmpty() ? "<font color='red'>"+tr("unset")+"</font>" : QString("<u>%1</u>").arg(GUrl(qualUrl).fileName()) );

    QString doc = tr("Import PHRED quality scores in file %1  to the sequences %2 and send the sequences to the output.")
        .arg(qualSeq).arg(producerName);

    return doc;
}

/*************************************
 * ImportPhredQualityWorker
 *************************************/
ImportPhredQualityWorker::ImportPhredQualityWorker(Actor* a) : BaseWorker(a), input(NULL), output(NULL) {
}

void ImportPhredQualityWorker::init() {
    input = ports.value(CoreLibConstants::IN_PORT_ID);
    output = ports.value(CoreLibConstants::OUT_PORT_ID);
    cfg.fileName = actor->getParameter(QUALITY_URL_ATTR)->getAttributeValue<QString>();
    cfg.type = DNAQuality::getDNAQualityTypeByName( actor->getParameter(QUALITY_TYPE_ATTR)->getAttributeValue<QString>() );
}

bool ImportPhredQualityWorker::isReady() {
    return (input && input->hasMessage());
}

Task* ImportPhredQualityWorker::tick() {
    while (!input->isEnded()) {
       DNASequence dna = input->get().getData().value<DNASequence>();
       seqObjs.append(new DNASequenceObject(dna.getName(), dna));
    }
    
    if( seqObjs.isEmpty() ) {
         log.error( tr("Sequence list is empty.") );
         return NULL;
    }

    Task* t = new ImportPhredQualityScoresTask(seqObjs, cfg);
    connect(t, SIGNAL(si_stateChanged()), SLOT(sl_taskFinished()));
    
    return t;
}

void ImportPhredQualityWorker::sl_taskFinished() {
    ImportPhredQualityScoresTask* t = qobject_cast<ImportPhredQualityScoresTask*>(sender());
    if (t->getState() != Task::State_Finished) {
       return;
    }
    
    foreach (DNASequenceObject* obj, seqObjs) {
        DNASequence dna = obj->getDNASequence();
        QVariant v = qVariantFromValue<DNASequence>(dna);
        output->put(Message(BioDataTypes::DNA_SEQUENCE_TYPE(), v));
    }
    
    if (input->isEnded()) {
        output->setEnded();
    }
    
    log.trace(tr("Import of qualities is finished.") );
}

bool ImportPhredQualityWorker::isDone() {
    return !input || input->isEnded();
}

void ImportPhredQualityWorker::cleanup() {
    qDeleteAll(seqObjs);
}

} //namespace LocalWorkflow
} //namespace GB2
