/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_EXPORT_PLUGIN_TASKS_H_
#define _GB2_EXPORT_PLUGIN_TASKS_H_

#include <core_api/DocumentFormats.h>

#include <core_api/Task.h>
#include <core_api/DocumentModel.h>
#include <datatype/MAlignment.h>
#include <memory>


namespace GB2 {

class DNAAlphabet;
class DNATranslation;

class ExportSequencesTaskSettings {
public:
    ExportSequencesTaskSettings() {
        merge = false;
        mergeGap = 0;
        strand = TriState_Yes;
        allAminoStrands = false;
		formatId = BaseDocumentFormats::PLAIN_FASTA;
        backTranslate = false;
        mostProbable = true;
        translationTable = "";
        useSpecificTable = false;
    }

    QStringList             names;      // name of every sequence
    QList<QByteArray>       sequences;  // the sequences to copy
    QList<DNAAlphabet*>     alphabets;  // alphabets for the sequences
    QList<DNATranslation*>  complTranslations; // complement translations for a sequence. Used only if 'strand' is 'compl' or 'both'
    QList<DNATranslation*>  aminoTranslations; // amino translation for a sequence. If not NULL -> sequence is translated
    QList<DNATranslation*>  nucleicTranslations; // nucleic translation for a sequence. If not NULL -> sequence is back translated
    
    QString     fileName; // file to export
    bool        merge;    // if true -> multiple sequences are merged
    int         mergeGap; // the gap between sequences if merged

    TriState    strand; //Yes -> direct, No -> complement, Unknown -> Both
    bool        allAminoStrands; // for every sequence and (every complement if needed) generates 3-frame amino translations

    bool        useSpecificTable; // if true -> use translation table, which user choose in dialog, else use tables from hints
    QString     translationTable;

    bool		backTranslate; // for every sequence generates reverse protein translations
    bool        mostProbable; // True - use most probable codon, False - use distribution according to frequency

	DocumentFormatId formatId;
};

/** Base class for all export tasks */
class AbstractExportTask : public Task {
public:
	AbstractExportTask(const QString& _name, TaskFlags f) : Task(_name, f){}
    virtual Document* getDocument() const = 0;
};

/** A task to adds exported document to project and open view*/
//TODO: make this task a general purpose routine
class AddDocumentAndOpenViewTask: public Task {
	Q_OBJECT
public:
	AddDocumentAndOpenViewTask(AbstractExportTask* t);
	QList<Task*> onSubTaskFinished( Task* subTask );
private:
	AbstractExportTask* exportTask;
};

/** Exports sequences to FASTA file */
class ExportSequencesTask: public AbstractExportTask {
    Q_OBJECT
public:
    ExportSequencesTask(const ExportSequencesTaskSettings& s);
    
	void run();
    
	virtual Document* getDocument() const {return doc.get();}

private:
    std::auto_ptr<Document> doc; 
    ExportSequencesTaskSettings config;
    
};

/** A task to save alignment to CLUSTAL */
class ExportAlignmentTask : public AbstractExportTask  {
    Q_OBJECT
public:
    ExportAlignmentTask(const MAlignment& ma, const QString& fileName, DocumentFormatId f);

    void run();

	virtual Document* getDocument() const {return doc.get();}

private:
    MAlignment              ma;
    QString                 fileName;
    DocumentFormatId        format;
	std::auto_ptr<Document> doc; 
};


/** A task to export alignment to FASTA */
class ExportMSA2SequencesTask : public AbstractExportTask {
    Q_OBJECT
public:
    ExportMSA2SequencesTask(const MAlignment& ma, const QString& url, bool trimAli, DocumentFormatId format);

	void run();

	virtual Document* getDocument() const {return doc.get();}

private:
    MAlignment				ma;
    QString					url;
    bool					trimAli;
    QString                 format;
	std::auto_ptr<Document> doc; 
};

class DNAChromatogramObject;
class LoadDocumentTask;

/** A task to export chromatogram to SCF */

struct ExportChromatogramTaskSettings {
    ExportChromatogramTaskSettings() : reverse(false), complement(false), loadDocument(false) {}
    QString url;
    bool reverse;
    bool complement;
    bool loadDocument;
};


class ExportDNAChromatogramTask : public AbstractExportTask {
    Q_OBJECT
public:
    ExportDNAChromatogramTask(DNAChromatogramObject* chromaObj, const ExportChromatogramTaskSettings& url);
    void prepare();
    virtual Document* getDocument() const;
private:
    DNAChromatogramObject*  cObj;
    ExportChromatogramTaskSettings settings;
    LoadDocumentTask* loadTask;
};




}//namespace

#endif
