/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "CircularViewSplitter.h"
#include "CircularView.h"

#include <core_api/Settings.h>
#include <core_api/GObject.h>
#include <core_api/L10n.h>

#include <gobjects/GObjectTypes.h>

#include <util_ov_annotated_dna/ADVSingleSequenceWidget.h>
#include <util_gui/DialogUtils.h>

#include <QtGui/QFileDialog>
#include <QtGui/QVBoxLayout>
#include <QtGui/QHBoxLayout>
#include <QtSvg/QSvgGenerator>
#include <QtGui/QPrinter>
#include <QtGui/QPixmap>
#include <QtGui/QPainter>
#include <QtGui/QMessageBox>

#include "gobjects/AnnotationTableObject.h"

namespace GB2 {

CircularViewSplitter::CircularViewSplitter( AnnotatedDNAView* view)
	: ADVSplitWidget(view)
{   
    QHBoxLayout *layout = new QHBoxLayout;
	splitter = new QSplitter(Qt::Horizontal);
    header = new CircularViewHeaderWidget(this);
	layout->setSpacing(0);
	layout->setContentsMargins(0,0,3,0);
    layout->addWidget(header);
	layout->addWidget(splitter);
	setSizePolicy(QSizePolicy::Expanding, QSizePolicy::Preferred);
	setBaseSize(600,600);
	setAcceptDrops(false);

	setLayout(layout);
}

void CircularViewSplitter::updateState( const QVariantMap& m) {
    Q_UNUSED(m);
	//TODO:
}

void CircularViewSplitter::saveState( QVariantMap& m ) {
    Q_UNUSED(m);
	//TODO:
}

void CircularViewSplitter::addView(CircularView* view) {
	circularViewList.append(view);
	splitter->insertWidget(0, view);
	adaptSize();
}

void CircularViewSplitter::removeView(CircularView* view) {
	circularViewList.removeAll(view);
}

bool CircularViewSplitter::isEmpty() {
	return circularViewList.isEmpty();
}

void CircularViewSplitter::sl_save2file() {
    header->sl_save2file();
}

void CircularViewSplitter::adaptSize()
{
	setMaximumHeight(1000);
	QWidget* widget = parentWidget();
	Q_ASSERT(widget != NULL);
	QSplitter* parentSplitter = qobject_cast<QSplitter* > (widget);
	
	int index = parentSplitter->indexOf(this);
	QList<int> sizes = parentSplitter->sizes();
	sizes[index] = 500;
	parentSplitter->setSizes(sizes);
}

#define CV_HEADER_WIDTH 24
CircularViewHeaderWidget::CircularViewHeaderWidget(CircularViewSplitter* p) : QWidget(p), ctx(p) {
    setFixedWidth(CV_HEADER_WIDTH);
    toolBar = new HBar(this);

    saveButton = new QToolButton(this);
    saveButton->setIcon(QIcon(":/core/images/cam2.png"));
    saveButton->setToolTip(tr("Save circular view as image"));
    connect(saveButton, SIGNAL(pressed()), SLOT(sl_save2file()));

    toolBar->addWidget(saveButton);
    //////////////////////////////////////////////////////////////////////////
    setVisible(false);
}

void CircularViewHeaderWidget::sl_save2file() {
    const QList<CircularView*>& viewList = ctx->getViewList();
    int x_offset=0;
    foreach(CircularView* cv, viewList) {
        x_offset += cv->getRenderArea()->width();
        //populate scene with copies of items
        CircularRuler* ruler = cv->getRuler();
        CircularRuler* rulerCopy = ruler->clone();
        QPointF curPos = rulerCopy->pos();
        rulerCopy->setPos(curPos + QPointF(x_offset, 0));
        scene.addItem(rulerCopy);

        const QMap<Annotation*, CircularAnnotationItem*>& circItems = cv->getCircularItems();
        const QList<CircularAnnotationItem*>& items = circItems.values();
        foreach(CircularAnnotationItem* it, items) {
            CircularAnnotationItem* item = it->clone();
            QPointF curPos = item->pos();
            item->setPos(curPos + QPointF(x_offset, 0));
            scene.addItem(item);
        }

        const QList<CircularAnnotationLabel*>& annLabels = cv->getLabelList();
        foreach(CircularAnnotationLabel* annLabel, annLabels) {
            CircularAnnotationLabel* labelCopy = annLabel->clone();
            QPointF curPos = labelCopy->pos();
            labelCopy->setPos(curPos + QPointF(x_offset, 0));
            scene.addItem(labelCopy);
        }

        TextItem* seqNameLabel = cv->getSeqNameLabel();
        TextItem* seqLenLabel = cv->getSeqLenLabel();
        TextItem* nameCopy = seqNameLabel->clone();
        curPos = nameCopy->pos();
        nameCopy->setPos(curPos.toPoint() + QPoint(x_offset, 0));
        TextItem* lenCopy = seqLenLabel->clone();
        curPos = lenCopy->pos();
        lenCopy->setPos(curPos.toPoint() + QPoint(x_offset, lenCopy->boundingRect().height()));
        scene.addItem(nameCopy);
        scene.addItem(lenCopy);

        CircularSelectionItem* selIt = cv->getSelItem()->clone();
        curPos = selIt->pos();
        selIt->setPos(curPos + QPoint(x_offset, 0));
        scene.addItem(selIt);
    }

    scene2file();
    scene.clear();
    saveButton->setDown(false);
}

void CircularViewHeaderWidget::scene2file() {
    QString filter = tr("Raster image (*.png *.bmp *.jpg *.jpeg *.ppm *.xbm *.xpm)");
    filter += "\n" + tr("Vector image (*.svg)");
    filter += "\n"+ tr("Portable document (*.pdf *.ps)");
    LastOpenDirHelper lod("image");
    lod.url = QFileDialog::getSaveFileName(this, tr("Export circular view to image"), lod.dir, filter);
    if (!lod.url.isEmpty()) {
        bool result = false;
        QRectF bounds = scene.itemsBoundingRect();
        if (lod.url.endsWith(".svg", Qt::CaseInsensitive)) {
            QSvgGenerator svg;
            svg.setFileName(lod.url);
            svg.setSize(bounds.size().toSize());
            QPainter painter(&svg);
            painter.setRenderHint(QPainter::Antialiasing);
            scene.render(&painter, QRectF(), bounds);
            result = painter.end();
        } else if (lod.url.endsWith(".pdf", Qt::CaseInsensitive) || lod.url.endsWith(".ps", Qt::CaseInsensitive)) {
            QPrinter printer;
            printer.setOutputFileName(lod.url);
            QPainter painter(&printer);
            painter.setRenderHint(QPainter::Antialiasing);
            scene.render(&painter, QRectF(), bounds);
            result = painter.end();
        } else {
            QPixmap pixmap(bounds.size().toSize());
            QPainter painter(&pixmap);
            painter.fillRect(pixmap.rect(), Qt::white);
            painter.setRenderHint(QPainter::Antialiasing);
            scene.render(&painter, QRectF(), bounds);
            result = painter.end() & pixmap.save(lod.url);
        }
        if (!result) {
            QMessageBox::critical(this, L10N::errorTitle(), tr("Unexpected error while exporting image!"));
        }
    }
}

} //namespace GB2