/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _CIRCULAR_ITEMS_H_
#define _CIRCULAR_ITEMS_H_

#include <QtGui/QGraphicsPathItem>
#include <QtGui/QFont>

namespace GB2 {

class Annotation;
class AnnotationSettings;
class CircularAnnotationItem;
class CircularViewRenderArea;
class CircularAnnotationLabel;
class CircurlarAnnotationRegionItem;

/************************************************************************/
/* CircularAnnotationItem                                               */
/************************************************************************/

class CircularAnnotationItem : public QGraphicsItem {
friend class CircurlarAnnotationRegionItem;
friend class CircularAnnotationLabel;
public:
    CircularAnnotationItem(Annotation* ann, CircurlarAnnotationRegionItem* region, CircularViewRenderArea* _ra);
    CircularAnnotationItem(Annotation* ann, QList<CircurlarAnnotationRegionItem*>& regions, CircularViewRenderArea* _ra);
    ~CircularAnnotationItem();
    virtual void paint(QPainter *p,const QStyleOptionGraphicsItem *item,QWidget *widget);
    Annotation* getAnnotation() const;
    virtual QRectF boundingRect() const;
    virtual bool contains(const QPointF &point);
    int containsRegion(const QPointF &point);
    void setSelected(bool isSelected);
    const QList<CircurlarAnnotationRegionItem*>& getRegions();
    CircularAnnotationItem* clone();
protected:
    bool isSelected;
    QColor color;
    Annotation *annotation;
    QList<CircurlarAnnotationRegionItem*> regions;
    QRectF _boundingRect;
    CircularViewRenderArea* ra;
};

/************************************************************************/
/* CircurlarAnnotationRegionItem                                        */
/************************************************************************/
class CircurlarAnnotationRegionItem : public QGraphicsPathItem{
friend class CircularAnnotationItem;
public:
    CircurlarAnnotationRegionItem(const QPainterPath& path, bool isShort, int number );
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget /* = 0 */);
    CircularAnnotationLabel* getLabel() const;
    void setLabel(CircularAnnotationLabel* label);
    CircurlarAnnotationRegionItem* clone();
protected:
    CircularAnnotationItem *parent;
    CircularAnnotationLabel *label;
    int number;
    bool isShort;
};

/************************************************************************/
/* CircularAnnotationLabel                                              */
/************************************************************************/
class CircularAnnotationLabel : public QGraphicsItem {
public:
    CircularAnnotationLabel(Annotation* ann, int _region, int sequenceLength, const QFont& font, CircularViewRenderArea* renderArea );
    virtual void paint(QPainter *p,const QStyleOptionGraphicsItem *item,QWidget *widget);
    Annotation* getAnnotation() const;
    virtual QRectF boundingRect() const;
    virtual bool contains(const QPointF &point);
    int getRegion() const;
    void setLabelPosition();
    static void prepareLabels(QList<CircularAnnotationLabel*>& labelItems);

    bool operator<(const CircularAnnotationLabel& a);
    CircularAnnotationLabel* clone();
private:
    int findClosestPoint(const QPoint& targetPoint, const QVector<QRect>& rects, QVector<int> indexes);
    void getVacantPositions(const QVector<QRect>& rects, QVector<int>& result);
    void getVacantInnerPositions(const QVector<QRect>& rects, QVector<int>& result);
    void getConnectionPoint();

    Annotation* annotation;
    float annotationAngle;
    QFont labelFont;
    QRectF midRect;
    QPointF midRegionPoint;
    int region;

    qreal startA;
    qreal endA;
    qreal spanA;
    QPoint labelPos;
    QPoint connectionPoint;
    CircularViewRenderArea* ra;
    bool hasPosition;
    QString labelText;
    bool inner;
    int seqLen;
};

/************************************************************************/
/* Circular Ruler                                                       */
/************************************************************************/

class CircularRuler : public QGraphicsItem {
public:
    CircularRuler(QPoint _center, int _rulerEllipseSize, qreal _rotationDegree, int _start, int _len, QFont _rulerFont, int _notchSize);
    void setRotationDegree(qreal degree);
    void setRulerSize(int diam);
    QRectF boundingRect() const;
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget /* = 0 */);
    CircularRuler* clone();

private:
    QPoint          center;
    int             rulerEllipseSize;
    qreal           rotationDegree;
    int             start;
    int             len;
    QFont           rulerFont;
    int             notchSize;
};

/************************************************************************/
/* Text Item                                                            */
/************************************************************************/

class TextItem : public QGraphicsItem {
public:
    TextItem() : itemText(""), textPos(QPoint(0,0)), font(QFont()) {}
    TextItem(QString _text, QPoint _pos, QFont f) : itemText(_text), textPos(_pos), font(f) {}
    QRectF boundingRect() const;
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget /* = 0 */);
    TextItem* clone();
    void setText(const QString& txt) { itemText=txt; }
    void setPos(const QPoint& _pos) { textPos=_pos; }
private:
    QString itemText;
    QPoint  textPos;
    QFont   font;
};

/************************************************************************/
/* Circular Selection Item                                              */
/************************************************************************/

class CircularSelectionItem : public QGraphicsItem {
public:
    //CircularSelectionItem(QPainterPath* path, CircularViewRenderArea* _ra) : QGraphicsPathItem(path), ra(_ra) {}
    CircularSelectionItem() {}
    CircularSelectionItem(const QList<QPainterPath*>& _paths) : paths(_paths) {}
    QRectF boundingRect() const;
    void paint(QPainter *painter, const QStyleOptionGraphicsItem *option, QWidget *widget /* = 0 */);
    void addPath(QPainterPath& path);
    void clear();
    CircularSelectionItem* clone();
private:
    QList<QPainterPath*>    paths;
};

} // namespace GB2

#endif // _CIRCULAR_ITEMS_H_
