/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "ExportImageDialog.h"
#include "BioStruct3DGLWidget.h"
#include "gl2ps/gl2ps.h"

#include <core_api/Log.h>
#include <core_api/GUrlUtils.h>
#include <core_api/L10n.h>
#include <util_gui/DialogUtils.h>

#include <QtGui/QImageWriter>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

namespace GB2 {

static LogCategory log(ULOG_CAT_PLUGIN_BIOSTRUCT_3D);

int ExportImageDialog::fileCounter = 1;

ExportImageDialog::ExportImageDialog( BioStruct3DGLWidget* widget ) : QDialog(widget), glWidget(widget)
{
    setupUi(this);
    setupFormats();
    setupComponents();    
}

#define IMAGE_DIR   "image"
#define SVG_FORMAT  "svg"
#define PS_FORMAT   "ps"

void ExportImageDialog::setupComponents()
{
    widthSpinBox->setValue(glWidget->width());
    heightSpinBox->setValue(glWidget->height());
    
    foreach (QString formatName, supportedFormats) {
        formatsBox->addItem(formatName);
    }

    LastOpenDirHelper lod(IMAGE_DIR);
    QString count = QString("%1").arg(fileCounter);
    while (count.length() != 4) {
        count.push_front('0');
    }
    QString imageName = "/" + glWidget->getPDBId() + count + ".";
    fileNameEdit->setText(QFileInfo(lod.dir + imageName + formatsBox->currentText()).absoluteFilePath());
   
    connect(browseFileButton, SIGNAL(clicked()), this, SLOT(sl_onBrowseButtonClick()));
    connect(formatsBox, SIGNAL(currentIndexChanged(const QString&)), this, SLOT(sl_onFormatsBoxItemChanged(const QString&)));

}

void ExportImageDialog::setupFormats() {
    supportedFormats.append(SVG_FORMAT);
    supportedFormats.append("png");
    supportedFormats.append(PS_FORMAT);
    supportedFormats.append("jpg");
    supportedFormats.append("tiff");
}

void ExportImageDialog::accept() 
{
    QCursor cursor = this->cursor();
    ++fileCounter;
    QString fileName(fileNameEdit->text());
    // TODO: exclude saving functionality for dialog reuse
    if (fileName.isEmpty()) {
        QMessageBox::warning(this, tr("Error"),tr("The filename is empty!"));
        return;
    }
//     QString format = getFileExtension(fileName);
//     if (!supportedFormats.contains(format)) {
//         QMessageBox::error(this, tr("Error"),tr(QString("Format \"%1\" is not supported!").arg(format)),QMessageBox::Ok);
//         return;
//     }

    log.info(tr("Saving image to '%1'...").arg(fileName));
    setCursor(Qt::WaitCursor);
    QString formatName = formatsBox->currentText();
    if (isVectorGraphicFormat(formatName)) {
        int opt = GL2PS_NONE;
        int format = getVectorFormatIdByName(formatName);
        glWidget->writeImage2DToFile(format, opt, 2, qPrintable(fileName));
    } else {
        QPixmap image = glWidget->renderPixmap(widthSpinBox->value(), heightSpinBox->value());
        bool result = image.save(fileName, qPrintable(formatName));
        if (!result) {
            QMessageBox::critical(this, L10N::errorTitle(), L10N::errorImageSave(fileName, formatName));
            return;
        }

    }
    log.info("Done!");
    setCursor(cursor);
    QDialog::accept();
}


void ExportImageDialog::sl_onBrowseButtonClick() {
    QString fileFormats;
    QMap<QString,QString>::const_iterator iter;
    foreach (QString formatName, supportedFormats) {
        fileFormats += formatName.toUpper() + " format ( *." + formatName + ");;"; 
    }
    LastOpenDirHelper lod(IMAGE_DIR);
    lod.url = QFileDialog::getSaveFileName(this, tr("Save image to..."), lod, fileFormats);
    if (lod.url.isEmpty()) {
        return;
    }
    fileNameEdit->setText(lod.url);
    QString formatSuffix = QFileInfo(lod.url).suffix();
    Q_ASSERT(supportedFormats.contains(formatSuffix));
    int index = formatsBox->findText(formatSuffix);
    formatsBox->setCurrentIndex(index);
}

void ExportImageDialog::sl_onFormatsBoxItemChanged( const QString& text )
{
    QString format("");
    foreach (QString formatName, supportedFormats) {
        if (formatName == text) {
            format = text;
            break;
        }
    } 
    Q_ASSERT(!format.isEmpty());
    QString fileName = QFileInfo(fileNameEdit->text()).absoluteFilePath();
    QString suffix = QFileInfo(fileName).suffix();
    //Remove old suffix
    if (!suffix.isEmpty()) {
        int index = fileName.indexOf(suffix);
        fileName.remove(index - 1, suffix.size() + 1); // "." + suffix -> size() + 1
    }
    fileName = GUrlUtils::ensureFileExt(GUrl(fileName, GUrl_File), QStringList(format)).getURLString();
    fileNameEdit->setText(fileName);
}

bool ExportImageDialog::isVectorGraphicFormat( const QString& formatName )
{
    if ((formatName == SVG_FORMAT) || (formatName == PS_FORMAT)) {
        return true;
    }
    return false;

}

int ExportImageDialog::getVectorFormatIdByName( const QString& formatName )
{
    if (formatName == SVG_FORMAT) {
        return GL2PS_SVG;
    } else if (formatName == PS_FORMAT) {
        return GL2PS_EPS;
    } else 
        return -1;
    
}
} // namespace

