/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BioStruct3DViewPlugin.h"
#include "BioStruct3DSplitter.h"
#include "BioStruct3DGLWidget.h"
#include "LoadRemotePDBFileDialog.h"

#include <core_api/GObject.h>
#include <core_api/DocumentFormats.h>
#include <core_api/DocumentModel.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/BioStruct3DObject.h>
#include <core_api/MainWindow.h>
#include <selection/DocumentSelection.h>

#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <util_ov_annotated_dna/ADVSingleSequenceWidget.h>
#include <util_gui/GUIUtils.h>
#include <util_tasks/LoadRemoteDocumentTask.h>

#include <QtGui/QMessageBox>
#include <QtGui/QMenu>

namespace GB2 {


/*!
* \mainpage BioStruct3D Viewer Plugin Documentation
*
* \section viewer Introduction
*
* BioStruct3D Viewer is a macromolecular viewing / editing tool.
* It is activated whenever UGENE loads document that contains BioStruct3DObject.
* 
*
* \subsection main Main Classes
*
* plugin classes:
* - BioStruct3DGLWidget : Widget for rendering 3d representations of macromolecular structure.
* - BioStruct3DSplitter : Multiple glWidgets layout and manipulation  
* - GLFrame : Class for manipulating the 3d viewpoint
*
* plugin interfaces: 
* - BioStruct3DGLRenderer : General interface for structure 3d graphical styles
* - BioStruct3DColorScheme : Interface for coloring atoms, bonds, etc.
*
*
*/ 

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    if (AppContext::getMainWindow()) {
        BioStruct3DViewPlugin* plug = new BioStruct3DViewPlugin();
        return plug;
    }
    return NULL;
}

BioStruct3DViewPlugin::BioStruct3DViewPlugin() : Plugin(tr("biostruct3d_view_plugin"), tr("biostruct3d_view_plugin_desc")) {
    // Init plugin view context
    viewContext = new BioStruct3DViewContext(this);
    viewContext->init();
}

BioStruct3DViewPlugin::~BioStruct3DViewPlugin()
{

}
BioStruct3DViewContext::BioStruct3DViewContext(QObject* p)
: GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)
{
    
}

void BioStruct3DViewContext::initViewContext(GObjectView* v)  {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(v);

    DNASequenceObject* dna=av->getSequenceInFocus()->getSequenceObject();
    Document* doc = dna->getDocument();
    QList<GObject*> biostructObjs = doc->findGObjectByType(GObjectTypes::BIOSTRUCTURE_3D);
    if (biostructObjs.isEmpty()) {
        return;
    }
    QList<ADVSequenceWidget*> seqWidgets = av->getSequenceWidgets();
    foreach(ADVSequenceWidget* w, seqWidgets) {
        ADVSingleSequenceWidget* aw = qobject_cast<ADVSingleSequenceWidget*>(w);
        if (aw!=NULL) {
            aw->setDetViewCollapsed(true);
			aw->setOverviewCollapsed(true);
        }
    }
    foreach(GObject* obj, biostructObjs) {
        v->addObject(obj);
    }
}

bool BioStruct3DViewContext::canHandle(GObjectView* v, GObject* o) {
    Q_UNUSED(v);
    bool res = qobject_cast<BioStruct3DObject*>(o) != NULL;
    return res;
}

void BioStruct3DViewContext::onObjectAdded(GObjectView* view, GObject* obj) {
    //todo: add sequence & all objects associated with sequence to the view?

    BioStruct3DObject* obj3d = qobject_cast<BioStruct3DObject*>(obj);
    if (obj3d == NULL || view == NULL) {
        return;
    }
    
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    if (splitterMap.contains(view)) {
        BioStruct3DSplitter* splitter = splitterMap.value(view);
        splitter->addObject(obj3d);
        return;
    }
    BioStruct3DSplitter* splitter = new BioStruct3DSplitter(getClose3DViewAction(view), av);
    av->insertWidgetIntoSplitter(splitter);
    splitter->addObject(obj3d);
    splitterMap.insert(view,splitter);
}

void BioStruct3DViewContext::onObjectRemoved(GObjectView* v, GObject* obj) {
     BioStruct3DObject* obj3d = qobject_cast<BioStruct3DObject*>(obj);
     if (obj3d == NULL) {
         return;
     }
     BioStruct3DSplitter* splitter = splitterMap.value(v);
     bool close = splitter->removeObject(obj3d);
     if (close) {
         remove3DView(v, splitter);
     }
}

void BioStruct3DViewContext::remove3DView(GObjectView* view, BioStruct3DSplitter* splitter) {
   
    assert(splitter->getChildWidgets().isEmpty());
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    av->unregisterSplitWidget(splitter);
    splitterMap.remove(view);
    delete splitter;
}

QAction* BioStruct3DViewContext::getClose3DViewAction(GObjectView* view) {
    QList<QObject*> resources = viewResources.value(view);
    foreach(QObject* r, resources) {
        GObjectViewAction* a= qobject_cast<GObjectViewAction*>(r);
        if (a!=NULL) {
            return a;
        }
    }
    QAction* a = new GObjectViewAction(this, view, tr("Close 3D view"));
    connect(a, SIGNAL(triggered()), SLOT(sl_close3DView()));
    resources.append(a);
    return a;
}

void BioStruct3DViewContext::sl_close3DView() {
    GObjectViewAction* action = qobject_cast<GObjectViewAction*>(sender());
    GObjectView* ov = action->getObjectView();
    QList<GObject*> objects = ov->getObjects();
    foreach(GObject* obj, objects) {
        if (obj->getGObjectType() == GObjectTypes::BIOSTRUCTURE_3D) {
            ov->removeObject(obj);
        }
    }
//     assert(get3DView(action->getObjectView(), false) == NULL);
}

void BioStruct3DViewContext::sl_windowClosing(MWMDIWindow* w) {
	splitterMap.clear();
	GObjectViewWindowContext::sl_windowClosing(w);
}

}//namespace
