/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "RemovePartFromSequenceTask.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/IOAdapter.h>
#include <core_api/GObject.h>
#include <core_api/DocumentFormats.h>
#include <core_api/Counter.h>

#include <util_tasks/MultiTask.h>
#include <util_tasks/AddDocumentTask.h>
#include <util_tasks/SaveDocumentTask.h>

#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/GObjectUtils.h>

namespace GB2 {

static LogCategory log(ULOG_CAT_USER_INTERFACE);

RemovePartFromSequenceTask::RemovePartFromSequenceTask(DocumentFormatId _dfId, DNASequenceObject *_seqObj, LRegion _regionTodelete, AnnotationStrategyForRemove _str, 
                                                       const GUrl& _url, bool _mergeAnnotations )
:Task(tr("Remove part from sequence"), TaskFlag_NoRun), dfId(_dfId), mergeAnnotations(_mergeAnnotations), save(true),
url(_url), strat(_str), seqObj(_seqObj), regionToDelete(_regionTodelete) {
    GCOUNTER( cvar, tvar, "RemovePartFromSequenceTask" );
    curDoc = seqObj->getDocument();
    if(url == curDoc->getURL() || _url.isEmpty()){
        save = false;
        return;
    }
}

Task::ReportResult RemovePartFromSequenceTask::report(){
    if(regionToDelete == LRegion(0,0)) {
        return ReportResult_Finished;
    }
    DNASequence sequence = seqObj->getDNASequence();

    LRegion allSeq(0, sequence.length());
    if(!allSeq.contains(regionToDelete)){
        log.error(tr("Region to delete larger then whole sequence"));
        return ReportResult_Finished;
    }

    Project *p = AppContext::getProject();
    if(p != NULL){
        if(p->isStateLocked()){
            return ReportResult_CallMeAgain;
        }
        docs = p->getDocuments();
    }
    if(docs.isEmpty()){
        docs.append(curDoc);
    }

    if(curDoc->isStateLocked()){
        log.error(tr("Document is in locked state"));
        return ReportResult_Finished;
    }    

    if(save){
        preparationForSave();
    }
    sequence.seq.remove(regionToDelete.startPos, regionToDelete.len);
    seqObj->setSequence(sequence);

    fixAnnotations();
    
    if(save){
        QList<Task*> tasks;
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
        tasks.append(new SaveDocumentTask(seqObj->getDocument(), iof, url.getURLString()));              
        Project *p = AppContext::getProject();
        if(p != NULL){
            tasks.append(new AddDocumentTask(newDoc));
        }
        AppContext::getTaskScheduler()->registerTopLevelTask(new MultiTask("Save document and add it to project (optional)", tasks));
    } 
    return ReportResult_Finished;
}


void RemovePartFromSequenceTask::fixAnnotations(){
    foreach(Document *d, docs){
        QList<GObject*> annotationTablesList = d->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
        foreach(GObject *table, annotationTablesList){
            AnnotationTableObject *ato = qobject_cast<AnnotationTableObject*>(table);
            if(ato->hasObjectRelation(seqObj, GObjectRelationRole::SEQUENCE)){
                QList<Annotation*> annList = ato->getAnnotations();
                foreach(Annotation *an, annList){
                    QList<LRegion> regionList =  an->getLocation(), toReplace, replaceList;
                    foreach(LRegion reg, regionList){
                        if(reg.endPos() <= regionToDelete.startPos){
                            toReplace.append(reg);
                        }else{
                            if (strat == AnnotationStrategyForRemove_Resize){
                                if(reg.contains(regionToDelete)){
                                    reg.len -= regionToDelete.len;
                                    toReplace.append(reg);
                                }else if(reg.intersects(regionToDelete)){
                                    if(reg.startPos <= regionToDelete.startPos){
                                        reg.len -= reg.endPos() - regionToDelete.startPos;
                                        toReplace.append(reg);
                                    }else{
                                        reg.len = reg.len - (regionToDelete.endPos() - reg.startPos);
                                        reg.startPos = regionToDelete.startPos;
                                        toReplace.append(reg);
                                    }
                                }else if(reg.startPos >= regionToDelete.endPos()){
                                    reg.startPos -= regionToDelete.len;
                                    toReplace.append(reg);
                                }
                            }else if(strat == AnnotationStrategyForRemove_Remove){
                                if(reg.intersects(regionToDelete) || regionToDelete.contains(reg)){
                                }else if(reg.startPos >= regionToDelete.endPos()){
                                    reg.startPos -= regionToDelete.len;
                                    toReplace.append(reg);
                                }
                            }
                        }
                    }
                    foreach(LRegion rrr, toReplace){
                        assert(rrr.startPos >=0);
                        if(rrr.len > 0){
                            replaceList.append(rrr);
                        }
                    }
                    if(!replaceList.isEmpty()){
                        an->replaceLocationRegions(replaceList);
                    }else{
                        ato->removeAnnotation(an);
                    }
                }
            }
        }
    }
}


void RemovePartFromSequenceTask::preparationForSave(){
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(dfId);
    if (iof == NULL) {
        return;
    }
    QList<GObject*> objList = curDoc->getObjects();
    if(mergeAnnotations){
        DNASequenceObject *oldObj = seqObj;
        newDoc = df->createNewDocument(iof, url, curDoc->getGHintsMap());
        foreach(GObject* go, objList){
            if(df->isObjectOpSupported(newDoc, DocumentFormat::DocObjectOp_Add, go->getGObjectType()) && 
                (go->getGObjectType() != GObjectTypes::SEQUENCE || go == seqObj) &&
                go->getGObjectType() != GObjectTypes::ANNOTATION_TABLE){
                    GObject *cl = go->clone();
                    newDoc->addObject(cl);
                    if(go == seqObj){
                        seqObj = qobject_cast<DNASequenceObject *>(cl);
                    }
                    GObjectUtils::updateRelationsURL(cl, curDoc->getURL(), url);
            }
        }
        AnnotationTableObject *newDocAto = new AnnotationTableObject("Annotations");
        newDoc->addObject(newDocAto);
        newDocAto->addObjectRelation(seqObj, GObjectRelationRole::SEQUENCE);
        foreach(Document *d, docs){
            QList<GObject*> annotationTablesList = d->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
            foreach(GObject *table, annotationTablesList){
                AnnotationTableObject *ato = (AnnotationTableObject*)table;
                if(ato->hasObjectRelation(oldObj, GObjectRelationRole::SEQUENCE)){
                    foreach(Annotation *ann, ato->getAnnotations()){
                        QStringList groupNames;
                        foreach(AnnotationGroup* gr,ann->getGroups()){
                            groupNames.append(gr->getGroupName());
                        }
                        newDocAto->addAnnotation(new Annotation(ann->data()), groupNames);
                    }
                }
            }
        }
    }else{
        newDoc = df->createNewDocument(iof, url, curDoc->getGHintsMap());
        foreach(GObject* go, objList){
            if(df->isObjectOpSupported(newDoc, DocumentFormat::DocObjectOp_Add, go->getGObjectType())){
                GObject *cl = go->clone();
                newDoc->addObject(cl);
                if(go == seqObj){
                    seqObj = qobject_cast<DNASequenceObject *>(cl);
                }
                GObjectUtils::updateRelationsURL(cl, curDoc->getURL(), url);
            }
        }
    }
    docs.append(newDoc);
}


}//ns
