/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "AddPartToSequenceTask.h"

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/IOAdapter.h>
#include <core_api/GObject.h>
#include <core_api/Task.h>
#include <core_api/DocumentFormats.h>
#include <core_api/Counter.h>

#include <gobjects/AnnotationTableObject.h>
#include <gobjects/GObjectRelationRoles.h>
#include <gobjects/GObjectUtils.h>

#include <util_tasks/SaveDocumentTask.h>
#include <util_tasks/AddPartToSequenceTask.h>
#include <util_tasks/MultiTask.h>
#include <util_tasks/AddDocumentTask.h>

#include <util_ov_annotated_dna/ADVSingleSequenceWidget.h>
#include <util_ov_annotated_dna/GSequenceLineView.h>

namespace GB2{

static LogCategory log(ULOG_CAT_USER_INTERFACE);

AddPartToSequenceTask::AddPartToSequenceTask(DocumentFormatId _dfId, DNASequenceObject *_seqObj, int _insertPos, 
                                             DNASequence  _seqPart, AnnotationStrategyForAdd _strat, 
											 const GUrl& _url, bool _mergeAnnotations)
:Task(tr("Add part to sequence"), TaskFlag_NoRun), dfId(_dfId), 
mergeAnnotations(_mergeAnnotations), save(true), url(_url), strat(_strat),
seqObj(_seqObj), insertPos(_insertPos), seqPart(_seqPart) 
{
    curDoc = seqObj->getDocument();
    GCOUNTER( cvar, tvar, "Add part to sequence" );
    if (url == curDoc->getURL() || _url.isEmpty()){
        save = false;
        return;
    }
}

Task::ReportResult AddPartToSequenceTask::report(){
    if(insertPos > seqObj->getSequenceLen()){
        log.error(tr("Insertion position is out of sequence bounds"));
        return ReportResult_Finished;
    }
    Project *p = AppContext::getProject();
    if(p != NULL){
        docs = p->getDocuments();
    }
    if(docs.isEmpty()){
        docs.append(curDoc);
    }
    if(curDoc->isStateLocked()){
        log.error(tr("Document is in locked state"));
        return ReportResult_Finished;
    }
    if(save){
        preparationForSave();
    }

    DNASequence sequence = seqObj->getDNASequence();
    if(seqPart.length() == 0) {
        return ReportResult_Finished;
    }

    assert(sequence.alphabet == seqPart.alphabet);
    sequence.seq.insert(insertPos, seqPart.constData());
    seqObj->setSequence(sequence);

    fixAnnotations();
        
    if(save){
        QList<Task*> tasks;
        IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
        tasks.append(new SaveDocumentTask(seqObj->getDocument(), iof, url));              
        Project *p = AppContext::getProject();
        if(p != NULL){
            tasks.append(new AddDocumentTask(newDoc));
        }
        AppContext::getTaskScheduler()->registerTopLevelTask(new MultiTask("Save document and add it to project (optional)", tasks));
    } 
    return ReportResult_Finished;
}

void AddPartToSequenceTask::fixAnnotations(){
    int len = seqPart.length();
    foreach(Document *d, docs){
        QList<GObject*> annotationTablesList = d->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
        foreach(GObject *table, annotationTablesList){
            AnnotationTableObject *ato = (AnnotationTableObject*)table;
            if(ato->hasObjectRelation(seqObj, GObjectRelationRole::SEQUENCE)){
                QList<Annotation*> annList = ato->getAnnotations();
                foreach(Annotation *an, annList){
                    QList<LRegion> regionList =  an->getLocation(), toReplace;
                    foreach(LRegion reg, regionList){
                        if(reg.endPos() <= insertPos){
                            toReplace.append(reg);
                        }else{
                            if (strat == AnnotationStrategyForAdd_Resize){
                                if(reg.endPos() > insertPos && reg.startPos <= insertPos ){
                                    reg.len += len;
                                    toReplace.append(reg);
                                }else if(reg.startPos > insertPos){
                                    reg.startPos += len;
                                    toReplace.append(reg);
                                }                                
                            }else if(strat == AnnotationStrategyForAdd_Split){
                                if(reg.endPos() > insertPos && reg.startPos <= insertPos){
                                    LRegion firstPart, secondPart;
                                    firstPart.startPos = reg.startPos;
                                    firstPart.len = insertPos - reg.startPos;
                                    secondPart.startPos = firstPart.endPos() + len;
                                    secondPart.len = reg.len - firstPart.len;
                                    toReplace.append(firstPart);
                                    toReplace.append(secondPart);
                                }else if(reg.startPos > insertPos){
                                    reg.startPos += len;
                                    toReplace.append(reg);
                                }                                
                            }else if(strat == AnnotationStrategyForAdd_Remove){
                                if(reg.endPos() > insertPos && reg.startPos <= insertPos){

                                }else if(reg.startPos > insertPos){
                                    reg.startPos += len;
                                    toReplace.append(reg);
                                }                                
                            }
                        }
                    }
                    an->replaceLocationRegions(toReplace);
                }
            }
        }
    }
}

void AddPartToSequenceTask::preparationForSave(){
    IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
    DocumentFormat *df = AppContext::getDocumentFormatRegistry()->getFormatById(dfId);;
    if (iof == NULL) {
        return;
    }
    QList<GObject*> objList = curDoc->getObjects();
    if(mergeAnnotations){
        DNASequenceObject *oldObj = seqObj;
        newDoc = df->createNewDocument(iof, url, curDoc->getGHintsMap());
        foreach(GObject* go, objList){
            if(df->isObjectOpSupported(newDoc, DocumentFormat::DocObjectOp_Add, go->getGObjectType()) && 
                (go->getGObjectType() != GObjectTypes::SEQUENCE || go == seqObj) &&
                go->getGObjectType() != GObjectTypes::ANNOTATION_TABLE){
                    GObject *cl = go->clone();
                    newDoc->addObject(cl);
                    if(go == seqObj){
                        seqObj = qobject_cast<DNASequenceObject *>(cl);
                    }
                    GObjectUtils::updateRelationsURL(cl, curDoc->getURL(), url);
            }
        }
        AnnotationTableObject *newDocAto = new AnnotationTableObject("Annotations");
        newDoc->addObject(newDocAto);
        newDocAto->addObjectRelation(seqObj, GObjectRelationRole::SEQUENCE);
        foreach(Document *d, docs){
            QList<GObject*> annotationTablesList = d->findGObjectByType(GObjectTypes::ANNOTATION_TABLE);
            foreach(GObject *table, annotationTablesList){
                AnnotationTableObject *ato = (AnnotationTableObject*)table;
                if(ato->hasObjectRelation(oldObj, GObjectRelationRole::SEQUENCE)){
                    foreach(Annotation *ann, ato->getAnnotations()){
                        QStringList groupNames;
                        foreach(AnnotationGroup* gr,ann->getGroups()){
                            groupNames.append(gr->getGroupName());
                        }
                        newDocAto->addAnnotation(new Annotation(ann->data()), groupNames);
                    }
                }
            }
        }
    }else{
        newDoc = df->createNewDocument(iof, url, curDoc->getGHintsMap());
        foreach(GObject* go, objList){
            if(df->isObjectOpSupported(newDoc, DocumentFormat::DocObjectOp_Add, go->getGObjectType())){
                GObject *cl = go->clone();
                newDoc->addObject(cl);
                if(go == seqObj){
                    seqObj = qobject_cast<DNASequenceObject *>(cl);
                }
                GObjectUtils::updateRelationsURL(cl, curDoc->getURL(), url);
            }
        }
    }
    docs.append(newDoc);
}

}//ns
