/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SimpleTextObjectView.h"
#include "SimpleTextObjectViewTasks.h"

#include <core_api/DocumentFormats.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/AppContext.h>
#include <core_api/DocumentModel.h>

#include <gobjects/TextObject.h>
#include <selection/SelectionUtils.h>


#include <QtGui/QVBoxLayout>
#include <QtGui/QScrollBar>


namespace GB2 {

/* TRANSLATOR GB2::SimpleTextObjectViewFactory */    

static LogCategory log(ULOG_CAT_TEXT);


//////////////////////////////////////////////////////////////////////////
/// Factory
const GObjectViewFactoryId SimpleTextObjectViewFactory::ID("SimpleTextView");


SimpleTextObjectViewFactory::SimpleTextObjectViewFactory(QObject* p)
: GObjectViewFactory(ID, tr("Text editor"), p)
{
}

SimpleTextObjectViewFactory::~SimpleTextObjectViewFactory() {
	QList<GObjectViewWindow*> views = GObjectViewUtils::findViewsByFactoryId(ID);
	MWMDIManager* mdi = AppContext::getMainWindow()->getMDIManager();
	foreach(GObjectViewWindow* v, views) {
		mdi->closeMDIWindow(v);
	}
}

bool SimpleTextObjectViewFactory::canCreateView(const MultiGSelection& multiSelection) {
	bool hasTextDocuments = !SelectionUtils::findDocumentsWithObjects(GObjectTypes::TEXT, &multiSelection, UOF_LoadedAndUnloaded, true).isEmpty();
	return hasTextDocuments;
}

bool SimpleTextObjectViewFactory::isStateInSelection(const MultiGSelection& multiSelection, const QVariantMap& stateData) {
	QString documentURL = SimpleTextObjectView::getDocumentUrl(stateData);
	QSet<Document*> documents = SelectionUtils::findDocumentsWithObjects(GObjectTypes::TEXT, &multiSelection, UOF_LoadedAndUnloaded, true);
	foreach(Document* doc, documents) {
		if (doc->getURL() == documentURL) {
			return true;
		}
	}
	return false;
}

Task* SimpleTextObjectViewFactory::createViewTask(const MultiGSelection& multiSelection, bool single) {
	QSet<Document*> documents = SelectionUtils::findDocumentsWithObjects(GObjectTypes::TEXT, &multiSelection, UOF_LoadedAndUnloaded, true);
	if (documents.size() == 0) {
		return NULL;
	}
	Task* result = (single || documents.size() == 1) ? NULL : new Task(tr("open_multiple_views"), TaskFlag_NoRun);
	foreach(Document* d, documents) {
		Task* t = new OpenSimpleTextObjectViewTask(d);
		if (result == NULL) {
			return t;
		} 
		//todo: limit number of views?
		result->addSubTask(t);
	}
	return result;
}	

Task* SimpleTextObjectViewFactory::createViewTask(const QString& viewName, const QVariantMap& state) {
	return new OpenSavedTextObjectViewTask(viewName, state);	
}


//////////////////////////////////////////////////////////////////////////
/// Simple Text View


SimpleTextObjectView::SimpleTextObjectView(const QString& name, TextObject* to, const QVariantMap& _state)  
: GObjectView(SimpleTextObjectViewFactory::ID, name), textObject(to), openState(_state), selection(to)
{
	textEdit = NULL;
	firstShow = true;
	assert(to);
	objects.append(to);
	requiredObjects.append(to);
}

QWidget* SimpleTextObjectView::createWidget() {
	assert(textEdit == NULL);
	textEdit = new QPlainTextEdit();
    textEdit->setLineWrapMode(QPlainTextEdit::NoWrap);
    textEdit->setWordWrapMode(QTextOption::NoWrap);
    textEdit->setPlainText(textObject->getText());
    if (textObject->isStateLocked()) {
        textEdit->setReadOnly(true);
    }
    connect(textEdit, SIGNAL(textChanged()), SLOT(sl_onTextEditTextChanged()));
    connect(textObject, SIGNAL(si_lockedStateChanged()), SLOT(sl_onTextObjStateLockChanged()));
	textEdit->installEventFilter(this);
	return textEdit;
}

bool SimpleTextObjectView::eventFilter(QObject* o, QEvent *e) {
	if (firstShow && o == textEdit && e->type() == QEvent::ShowToParent) {
		firstShow = false;
		updateView(openState);
	}
	return false;
}

void SimpleTextObjectView::sl_onTextEditTextChanged() {
	textObject->setText(textEdit->toPlainText());
}

void SimpleTextObjectView::sl_onTextObjStateLockChanged() {
    textEdit->setReadOnly(textObject->isStateLocked());
}


Task* SimpleTextObjectView::updateViewTask(const QString& stateName, const QVariantMap& state) {
	return new UpdateSimpleTextObjectViewTask(this, stateName, state);
}

bool SimpleTextObjectView::checkAddToView(const MultiGSelection& s) {
    Q_UNUSED(s);
	return false;
}

Task* SimpleTextObjectView::addToViewTask(const MultiGSelection& s) {
    Q_UNUSED(s);
	assert(0);
	return NULL;
}

#define URL_KEY				"url"
#define OBJ_KEY				"obj"
#define CURS_POS_KEY		"cursor_pos"
#define HBAR_POS_KEY		"hbar_pos"
#define VBAR_POS_KEY		"vbar_pos"


void SimpleTextObjectView::updateView(const QVariantMap& data) {
	int cursPos=  data.value(CURS_POS_KEY, 0).toInt();
	int hScrollPos = data.value(HBAR_POS_KEY, 0).toInt();
	int vScrollPos = data.value(VBAR_POS_KEY, 0).toInt();


	QTextCursor c = textEdit->textCursor();
	c.setPosition(cursPos);//, QTextCursor::KeepAnchor
	textEdit->setTextCursor(c);
	textEdit->verticalScrollBar()->setSliderPosition(vScrollPos);
	textEdit->horizontalScrollBar()->setSliderPosition(hScrollPos);
}

QVariantMap SimpleTextObjectView::saveState() {
	QVariantMap data;
	//Content:
	//file url
	data[URL_KEY] = textObject->getDocument()->getURLString();
    data[OBJ_KEY] = textObject->getGObjectName();
	//GUI:
	//start vis pos
	data[CURS_POS_KEY] = textEdit->textCursor().position();
	data[HBAR_POS_KEY] = textEdit->horizontalScrollBar()->sliderPosition();
	data[VBAR_POS_KEY] = textEdit->verticalScrollBar()->sliderPosition();
	return data;
}

QString SimpleTextObjectView::getDocumentUrl(const QVariantMap& savedState) {
	return savedState.value(URL_KEY).toString();
}

void SimpleTextObjectView::setDocumentUrl(QVariantMap& savedState, const QString& url) {
	savedState[URL_KEY] = url;
}

QString SimpleTextObjectView::getObjectName( const QVariantMap& savedState )
{
    return savedState.value(OBJ_KEY).toString();
}

}//namespace
