/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <document_format/PDBFormat.h>
#include <core_api/AppContext.h>

#include "MolecularSurface.h"
#include "MolecularSurfaceFactoryRegistry.h"

namespace GB2 {

// MolecularSurface

const float MolecularSurface::TOLERANCE = 1.0f;

const QVector<Face> &MolecularSurface::getFaces()const
{
    return faces;
}

QList<SharedAtom> MolecularSurface::findAtomNeighbors( const SharedAtom& a, const QList<SharedAtom>& atoms )
{
    QList<SharedAtom> neighbors;
    // maximum covalent radius in angstroms
    static const float maxAtomRadius = 1.0;
    static const float doubleRadius = 2*maxAtomRadius;
    Vector3D v1 = a->coord3d;

    foreach (const SharedAtom& neighbor, atoms) {
        if (neighbor == a) {
            continue;
        } 
        Vector3D v2 = neighbor->coord3d;
        if ( ( qAbs(v1.x - v2.x) <= doubleRadius) && ( qAbs(v1.y - v2.y) <= doubleRadius  ) && ( qAbs(v1.z - v2.z) <= doubleRadius ) ) {
            neighbors.append(neighbor);
        }
   }

    return neighbors;
}

GeodesicSphere MolecularSurface::getAtomSurfaceDots( const SharedAtom& a )
{
    QVector<Vector3D> surfaceDots;
    float radius = TOLERANCE + PDBFormat::getAtomCovalentRadius(a->atomicNumber);
    //Calculate sphere surface dots
    GeodesicSphere sphere(a->coord3d, radius);
    return sphere;
}

bool MolecularSurface::vertexNeighboursOneOf( const Vector3D& v, const QList<SharedAtom>& atoms )
{
    foreach (const SharedAtom& a, atoms) {
        float r = PDBFormat::getAtomCovalentRadius(a->atomicNumber) + TOLERANCE;
        Vector3D v2 = a->coord3d;
        //qDebug("testing if vertex (%f,%f,%f) neighbors atom (%f,%f,%f) of radius %f", v.x, v.y, v.z, v2.x, v2.y, v2.z, radius);
        if ( sqr(v.x - a->coord3d.x) + sqr(v.y - a->coord3d.y) + sqr(v.z - a->coord3d.z ) <= r*r ) {
                return true;
        }
    }
    
    return false;
}


MolecularSurfaceCalcTask::MolecularSurfaceCalcTask( const QString& surfaceTypeName, const QList<SharedAtom>& _atoms )
    :Task(tr("Molecular surface calculation"), TaskFlag_None), typeName(surfaceTypeName), atoms(_atoms)
{
    molSurface = NULL;
    
}

void MolecularSurfaceCalcTask::run()
{
   MolecularSurfaceFactory *factory=
            AppContext::getMolecularSurfaceFactoryRegistry()->getSurfaceFactory(typeName);
   molSurface = factory->createInstance();
   molSurface->calculate(atoms);
}

std::auto_ptr<MolecularSurface> MolecularSurfaceCalcTask::getCalculatedSurface()
{
    assert(molSurface != NULL);
    MolecularSurface* returnValue = molSurface;
    molSurface = NULL;
       
    return std::auto_ptr<MolecularSurface>(returnValue);
}
} // namespace

