/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MAlignmentObject.h"

#include <core_api/DNAAlphabet.h>
#include <datatype/DNASequence.h>
#include <util_algorithm/MSAUtils.h>

namespace GB2 {

GObject* MAlignmentObject::clone() const {
	MAlignmentObject* cln = new MAlignmentObject(msa, getGHintsMap());
    cln->setIndexInfo(getIndexInfo());
    return cln;
}


void MAlignmentObject::insertGap(int seqNum, int pos, int nGaps) {
	assert(!isStateLocked());
	MAlignment maBefore = msa;
	msa.insertChars(seqNum, pos, MAlignment_GapChar, nGaps);

	setModified(true);
    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::insertGap(int pos, int nGaps) {
    assert(nGaps > 0);
    assert(!isStateLocked());

    MAlignment maBefore = msa;
    QByteArray gap(nGaps, MAlignment_GapChar);
    for (int i=0, n = msa.getNumRows(); i < n; i++) {
		msa.insertChars(i, pos, MAlignment_GapChar, nGaps);
    }

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);
}

int MAlignmentObject::deleteGap(int seqNum, int pos, int maxGaps) {
    assert(!isStateLocked());

    MAlignment maBefore = msa;

	int n = 0, max = qBound(0, maxGaps, msa.getLength() - pos);
	while (n < max) {
		char c = msa.charAt(seqNum, pos + n);
		if (c != MAlignment_GapChar) { //not a gap
			break;
		}
		n++;
	}
	if (n == 0) {
		return 0;
	}
	msa.removeChars(seqNum, pos, n);
    
	setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);

    return n;
}

int MAlignmentObject::deleteGap(int pos, int maxGaps) {
    assert(!isStateLocked());

    MAlignment maBefore = msa;
    //find min gaps in all sequences starting from pos
    int minGaps = maxGaps;
	int max = qBound(0, maxGaps, msa.getLength() - pos);
    foreach(const MAlignmentRow& row, msa.getRows()) {
        int nGaps = 0;
        for (int i = pos; i < pos + max; i++, nGaps++) {
            if (row.chatAt(i) != MAlignment_GapChar) { 
                break;
            }
        }
        minGaps = qMin(minGaps, nGaps);
        if (minGaps == 0) {
            break;
        }
    }
    if (minGaps == 0) {
        return  0;
    }
    int nDeleted = minGaps;
    for (int i = 0, n = msa.getNumRows(); i < n; i++) {
		msa.removeChars(i, pos, nDeleted);
    }
	msa.setLength(msa.getLength() - nDeleted);

    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceListChanged = false;
    emit si_alignmentChanged(maBefore, mi);

    return nDeleted;
}


void MAlignmentObject::addRow(const DNASequence& seq, int seqIdx) {
    assert(!isStateLocked());
    MAlignment maBefore = msa;

    DNAAlphabet* newAlphabet = DNAAlphabet::deriveCommonAlphabet(seq.alphabet, getAlphabet());
    assert(newAlphabet != NULL);
    msa.setAlphabet(newAlphabet);
	
	MAlignmentRow row(seq.getName(), seq.seq, 0);
    msa.addRow(row, seqIdx);
    
    setModified(true);

    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::removeRow(int seqNum) {
    assert(!isStateLocked());

    MAlignment maBefore = msa;
    msa.removeRow(seqNum);
    setModified(true);

    MAlignmentModInfo mi;
    mi.sequenceContentChanged = false;
    emit si_alignmentChanged(maBefore, mi);
    
}


void MAlignmentObject::setMAlignment(const MAlignment& newMa) {
    assert(!isStateLocked());

    MAlignment maBefore = msa;

    msa = newMa;
    msa.setName( getGObjectName() );
    
    setModified(true);

    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::setGObjectName(const QString& newName) {
    msa.setName( newName );
    GObject::setGObjectName(newName);
}

void MAlignmentObject::removeRegion(int startPos, int startRow, int nBases, int nRows, bool removeEmptyRows) {
    assert(!isStateLocked());
    MAlignment maBefore = msa;

	msa.removeRegion(startPos, startRow, nBases, nRows, removeEmptyRows);
    
    setModified(true);
    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);

    
}

void MAlignmentObject::renameRow( int seqNum, const QString& newName ) {
    assert(seqNum >= 0 && seqNum < msa.getNumRows());
    assert(!isStateLocked());
    assert(!newName.isEmpty());

    const QString& curName = msa.getRow(seqNum).getName();
    if (curName == newName) {
        return;
    }

    MAlignment maBefore = msa;
    msa.renameRow(seqNum, newName);
    setModified(true);

    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);
}

void MAlignmentObject::crop( LRegion window, const QSet<QString>& rowNames )
{
    assert(!isStateLocked());
    MAlignment maBefore = msa;

    msa.crop(window, rowNames);

    setModified(true);
    MAlignmentModInfo mi;
    emit si_alignmentChanged(maBefore, mi);    
}



}//namespace


