/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSFFormat.h"

#include <document_format/DocumentFormatUtils.h>
#include <core_api/Task.h>
#include <core_api/IOAdapter.h>
#include <core_api/L10n.h>
#include <gobjects/GObjectTypes.h>
#include <gobjects/MAlignmentObject.h>
#include <util_text/TextUtils.h>
#include <util_algorithm/MSAUtils.h>

namespace GB2 {

const int MSFFormat::CHECK_SUM_MOD = 10000;
const QByteArray MSFFormat::MSF_FIELD = "MSF:";
const QByteArray MSFFormat::CHECK_FIELD = "Check:";
const QByteArray MSFFormat::LEN_FIELD = "Len:";
const QByteArray MSFFormat::NAME_FIELD = "Name:";
const QByteArray MSFFormat::TYPE_FIELD = "Type:";
const QByteArray MSFFormat::WEIGHT_FIELD = "Weight:";
const QByteArray MSFFormat::TYPE_VALUE = "P";
const double MSFFormat::WEIGHT_VALUE = 1.0;
const QByteArray MSFFormat::END_OF_HEADER_LINE = "..";
const QByteArray MSFFormat::SECTION_SEPARATOR = "//";
const int MSFFormat::CHARS_IN_ROW = 50;
const int MSFFormat::CHARS_IN_WORD = 10;

/* TRANSLATOR GB2::MSFFormat */    

//TODO: recheck if it does support streaming! Fix isObjectOpSupported if not!

MSFFormat::MSFFormat(QObject* p) : DocumentFormat(p, DocumentFormatFlags_SW, QStringList("msf")) {
	formatName = tr("MSF");
    supportedObjectTypes+=GObjectTypes::MULTIPLE_ALIGNMENT;
}

static bool getNextLine(IOAdapter* io, QByteArray& line) {
	static int READ_BUFF_SIZE = 1024;
	QByteArray readBuffer(READ_BUFF_SIZE, '\0');
	char* buff = readBuffer.data();

	qint64 len;
	bool eolFound = false, eof = false;
	while (!eolFound) {
		len = io->readLine(buff, READ_BUFF_SIZE, &eolFound);
		if (len < READ_BUFF_SIZE && !eolFound) {
			eolFound = eof = true;
		}
		line += readBuffer;
	}
	if (len != READ_BUFF_SIZE) {
		line.resize(line.size() + len - READ_BUFF_SIZE);
	}
	line = line.simplified();
	return eof;
}

static QByteArray getField(const QByteArray& line, const QByteArray& name) {
	int p = line.indexOf(name);
	if (p >= 0) {
		p += name.length();
		if (line[p] == ' ')
			++p;
		int q = line.indexOf(' ', p);
		if (q >= 0)
			return line.mid(p, q - p);
		else
			return line.mid(p);
	}
	return QByteArray();
}

int MSFFormat::getCheckSum(const QByteArray& seq) {
	int sum = 0;
	static int CHECK_SUM_COUNTER_MOD = 57;
	for (int i = 0; i < seq.length(); ++i) {
		char ch = seq[i];
		if (ch >= 'a' && ch <= 'z') {
			ch = ch + 'A' - 'a';
		}
		sum = (sum + ((i % CHECK_SUM_COUNTER_MOD) + 1) * ch) % MSFFormat::CHECK_SUM_MOD;
	}
	return sum;
}

void MSFFormat::load(IOAdapter* io, QList<GObject*>& objects, TaskStateInfo& ti) {
	MAlignment al(io->getURL().baseFileName());

	//skip comments
	int checkSum = -1;
	while (!ti.cancelFlag && checkSum < 0) {
		QByteArray line;
		if (getNextLine(io, line)) {
			ti.setError(MSFFormat::tr("Incorrect format"));
			return;
		}
		if (line.endsWith(END_OF_HEADER_LINE)) {
			bool ok;
			checkSum = getField(line, CHECK_FIELD).toInt(&ok);
			if (!ok || checkSum < 0)
				checkSum = CHECK_SUM_MOD;
		}
		ti.progress = io->getProgress();
	}

	//read info
	int sum = 0;
	QMap <QString, int> seqs;
	while (!ti.cancelFlag) {
		QByteArray line;
		if (getNextLine(io, line)) {
			ti.setError(MSFFormat::tr("Unexpected end of file"));
			return;
		}
		if (line.startsWith(SECTION_SEPARATOR))
			break;

		bool ok = false;
		QString name = QString::fromLocal8Bit(getField(line, NAME_FIELD).data());
		if (name.isEmpty()) {
			continue;
		}
		int check = getField(line, CHECK_FIELD).toInt(&ok);
		if (!ok || check < 0) {
			sum = check = CHECK_SUM_MOD;
		}

		seqs.insert(name, check);
		al.addRow(MAlignmentRow(name));
		if (sum < CHECK_SUM_MOD) {
			sum = (sum + check) % CHECK_SUM_MOD;
		}

		ti.progress = io->getProgress();
	}
	if (checkSum < CHECK_SUM_MOD && sum < CHECK_SUM_MOD && sum != checkSum) {
		ti.setError(MSFFormat::tr("Check sum test failed"));
		return;
	}

	//read data
    bool eof = false;
	while (!eof && !ti.cancelFlag) {
		QByteArray line;
		eof = getNextLine(io, line);
		if (line.isEmpty()) {
            continue;
		}

		int i = 0, n = al.getNumRows();
		for (; i < n; i++) {
			const MAlignmentRow& row = al.getRow(i);
			QByteArray t = row.getName().toLocal8Bit();
			if (line.startsWith(t) && line[t.length()] == ' ') {
				break;
			}
		}
		if (i == n) {
			continue;
		}
		for (int q, p = line.indexOf(' ') + 1; p > 0; p = q + 1) {
			q = line.indexOf(' ', p);
			QByteArray subSeq = (q < 0) ? line.mid(p) : line.mid(p, q - p);
			al.appendChars(i, subSeq.constData(), subSeq.length());
		}

		ti.progress = io->getProgress();
	}

	//checksum
	for (int i=0; i<al.getNumRows(); i++) {
		const MAlignmentRow& row = al.getRow(i);
		int expectedCheckSum = seqs[row.getName()];
		int sequenceCheckSum = getCheckSum(row.toByteArray(al.getLength()));
		if ( expectedCheckSum < CHECK_SUM_MOD &&  sequenceCheckSum != expectedCheckSum) {
			ti.setError(MSFFormat::tr("Check sum test failed"));
			return;
		}
		al.replaceChars(i, '.', MAlignment_GapChar);
		al.replaceChars(i, '~', MAlignment_GapChar);
	}

	DocumentFormatUtils::assignAlphabet(al);
	if (al.getAlphabet() == NULL) {
		ti.setError(MSFFormat::tr("Alphabet unknown"));
		return;
	}

	MAlignmentObject* obj = new MAlignmentObject(al);
	objects.append(obj);
}

Document* MSFFormat::loadDocument(IOAdapter* io, TaskStateInfo& ti, const QVariantMap& fs, DocumentLoadMode mode) {
	QList <GObject*> objs;
	load(io, objs, ti);

	if (ti.hasErrors()) {
		qDeleteAll(objs);
		return NULL;
	}
	return new Document(this, io->getFactory(), io->getURL(), objs, fs);
}

static bool writeBlock(IOAdapter *io, Document* d, TaskStateInfo& ti, const QByteArray& buf) {
	int len = io->writeBlock(buf);
	if (len != buf.length()) {
		ti.setError(L10N::errorWritingFile(d->getURL()));
		return true;
	}
	return false;
}

void MSFFormat::storeDocument( Document* d, TaskStateInfo& ti, IOAdapter* io ) {
	const MAlignmentObject* obj = NULL;
	if((d->getObjects().size() != 1) || ((obj = qobject_cast<const MAlignmentObject*>(d->getObjects().first())) == NULL)) {
		ti.setError("No data to write;");
		return;
	}
	const MAlignment& ma = obj->getMAlignment();

	//precalculate seq writing params
	int maxNameLen = 0, maLen = ma.getLength(), checkSum = 0;
	static int maxCheckSumLen = 4;
	QMap <QString, int> checkSums;
	foreach(const MAlignmentRow& row , ma.getRows()) {
		QByteArray sequence = row.toByteArray(maLen).replace(MAlignment_GapChar, '.');
		int seqCheckSum = getCheckSum(sequence);
		checkSums.insert(row.getName(), seqCheckSum);
		checkSum = (checkSum + seqCheckSum) % CHECK_SUM_MOD;
		maxNameLen = qMax(maxNameLen, row.getName().length());
	}
	int maxLengthLen = QString::number(maLen).length();

	//write first line
	QByteArray line = "  " + MSF_FIELD;
	line += " " + QByteArray::number(maLen);
	line += "  " + TYPE_FIELD;
	line += " " + TYPE_VALUE;
	line += "  " + QDateTime::currentDateTime().toString("dd.MM.yyyy hh:mm");
	line += "  " + CHECK_FIELD;
	line += " " + QByteArray::number(checkSum);
	line += "  " + END_OF_HEADER_LINE + "\n\n";
	if (writeBlock(io, d, ti, line))
		return;

	//write info
	foreach(const MAlignmentRow& row, ma.getRows()) {
		QByteArray line = " " + NAME_FIELD;
		line += " " + QString("%1").arg(row.getName(), -maxNameLen).toLocal8Bit();
		line += "  " + LEN_FIELD;
		line += " " + QString("%1").arg(maLen, -maxLengthLen);
		line += "  " + CHECK_FIELD;
		line += " " + QString("%1").arg(checkSums[row.getName()], -maxCheckSumLen);
		line += "  " + WEIGHT_FIELD;
		line += " " + QByteArray::number(WEIGHT_VALUE) + "\n";
		if (writeBlock(io, d, ti, line)) {
			return;
		}
	}
	if (writeBlock(io, d, ti, "\n" + SECTION_SEPARATOR + "\n\n")) {
		return;
	}

	for (int i = 0; !ti.cancelFlag && i < maLen; i += CHARS_IN_ROW) {
		/* write numbers */ {
			QByteArray line(maxNameLen + 2, ' ');
			QString t = QString("%1").arg(i + 1);
			QString s = QString("%1").arg(i + CHARS_IN_ROW < maLen ? i + CHARS_IN_ROW : maLen);
			int r = maLen - i < CHARS_IN_ROW ? maLen % CHARS_IN_ROW : CHARS_IN_ROW;
			r += (r - 1) / CHARS_IN_WORD - (t.length() + s.length());
			line += t;
			if (r > 0) {
				line += QByteArray(r, ' ');
				line += s;
			}
			line += '\n';
			if (writeBlock(io, d, ti, line)) {
				return;
			}
		}

		//write sequence
		foreach(const MAlignmentRow& row, ma.getRows()) {
			QByteArray line = QString("%1").arg(row.getName(), -(maxNameLen + 1)).toLocal8Bit();
			for (int j = 0; j < CHARS_IN_ROW && i + j < maLen; j += CHARS_IN_WORD) {
				line += ' ';
				int nChars = qMin(CHARS_IN_WORD, maLen - (i + j));
				line += row.mid(i + j, nChars).toByteArray(nChars).replace(MAlignment_GapChar, '.');
			}
			line += '\n';
			if (writeBlock(io, d, ti, line)) {
				return;
			}
		}
		if (writeBlock(io, d, ti, "\n")) {
			return;
		}
	}
}


bool MSFFormat::checkRawData(const QByteArray& rawData) const {
    const char* data = rawData.constData();
    int size = rawData.size();

    //MSF documents may contain unlimited number of comment lines in header ->
    //it is impossible to determine if file has MSF format by some predefined
    //amount of raw data read from it.

    //TODO: develop more advanced technique here!
    bool hasBinaryData = TextUtils::contains(TextUtils::BINARY, data, size);
    return hasBinaryData ? false : true;

}

} //namespace GB2
