/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#ifndef _GB2_LOCAL_TASK_H_
#define _GB2_LOCAL_TASK_H_

#include <QtCore/QMap>
#include <QtCore/QVariant>

#include <core_api/Task.h>
#include "Serializable.h"

#ifdef NEED_DLL_EXPORT
#define DLL_EXPORT_IF_NEEDED GB2_COREAPI_EXPORT
#else
#define DLL_EXPORT_IF_NEEDED 
#endif // NEED_DLL_EXPORT

namespace GB2 {

/*
 * Base class for settings of all LocalTasks
 */
class GB2_COREAPI_EXPORT LocalTaskSettings : public Serializable {
public:
    virtual ~LocalTaskSettings(){}
    
}; // LocalTaskSettings

/*
 * Base class for results of all LocalTasks
 */
class GB2_COREAPI_EXPORT LocalTaskResult : public Serializable {
public:
    virtual ~LocalTaskResult(){}
    
}; // LocalTaskResult

/*
 * LocalTask - task, that can be run on local machine by a remote request.
 * Task will be invoked by remote request if only LocalTaskFactory for this
 * task is registered in LocalTaskFactoryRegistry
 */
class GB2_COREAPI_EXPORT LocalTask : public Task {
    Q_OBJECT
public:
    LocalTask(const QString &name, TaskFlags flags);
    virtual ~LocalTask(){}
    virtual const LocalTaskResult * getResult() const = 0;
    
}; // LocalTask

/*
 * Base class for LocalTaskFactories
 */
class GB2_COREAPI_EXPORT LocalTaskFactory {
public:
    virtual ~LocalTaskFactory(){}
    virtual LocalTask * createInstance( const QVariant & settings ) const = 0;
    virtual LocalTaskResult * createResults( const QVariant & serializedResults ) const = 0;

    virtual QString getId()const = 0;
    
}; // LocalTaskFactory

/*
 * Simple realization of LocalTaskFactory
 */
template <class TSettings, class TTask, class TResult >
class DLL_EXPORT_IF_NEEDED SimpleLocalTaskFactoryImpl : public LocalTaskFactory {
public:
    static const QString ID;

public:
    SimpleLocalTaskFactoryImpl() {
    }
    
    virtual ~SimpleLocalTaskFactoryImpl() {
    }
    
    virtual LocalTask * createInstance( const QVariant & serializedSettings ) const {
        TSettings * settings = new TSettings();
        if( !settings->deserialize( serializedSettings ) ) {
            return NULL;
        }
        return new TTask( settings );
    }
    
    virtual LocalTaskResult * createResults( const QVariant & serializedResults ) const {
        TResult * results = new TResult();
        if( !results->deserialize( serializedResults ) ) {
            delete results;
            return NULL;
        }
        return results;
    }
    
    virtual QString getId() const {
        return ID; 
    }
    
}; // SimpleLocalTaskFactoryImpl

/*
 * Registry of LocalTaskFactories. Can be accessed by AppContext::getLocalTaskFactoryRegistry()
 */
class GB2_COREAPI_EXPORT LocalTaskFactoryRegistry {
public:
    LocalTaskFactoryRegistry();
    void registerLocalTaskFactory( LocalTaskFactory * factory );
    bool unregisterLocalTaskFactory( const QString &id );
    bool isLocalTaskFactoryRegistered( const QString &id ) const;
    const LocalTaskFactory * getLocalTaskFactory( const QString &id ) const;
    QStringList getLocalTaskFactoryIds() const;
    
private:
    LocalTaskFactoryRegistry( const LocalTaskFactoryRegistry & );
    LocalTaskFactoryRegistry & operator=( const LocalTaskFactoryRegistry & );
    
private:
    QMap< QString, LocalTaskFactory* > factories;
    
}; // LocalTaskFactoryRegistry

} // GB2

#endif // _GB2_LOCAL_TASK_H_
