/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#ifndef _GB2_LREGION_H_
#define _GB2_LREGION_H_

#include "core_api.h"

namespace GB2 {

class GB2_COREAPI_EXPORT LRegion {
public:
	LRegion() : startPos(0), len(0) {}
	LRegion(int _startPos, int _len ) : startPos(_startPos), len(_len) {}
    
	int endPos() const { assert(len>=0); return startPos+len; }

	bool operator== ( const LRegion & r ) const { return r.startPos == startPos && r.len == len; }

	bool operator!= ( const LRegion & r ) const { return r.startPos != startPos || r.len != len; }

    bool operator<(const LRegion &r) const {return startPos < r.startPos;}

	bool intersects(const LRegion& r) const;

	bool intersectsWithAny(const QList<LRegion>& rs) const;

	LRegion intersect(const LRegion& r) const;
	
	static QList<LRegion> intersect(const LRegion& r, const QList<LRegion>& regions);
		
	bool contains(int pos) const {return pos>=startPos && pos < endPos();}
	
	bool contains(const LRegion& r) const {return r.startPos >= startPos && r.endPos() <= endPos();}
	
	bool isEmpty() const {return len == 0; }

    int center() const { return (startPos + endPos()) / 2 ;}

	static bool intersects(int start1, int end1, int start2, int end2) {return !(start1 >= end2 || start2 >= end1);}

	static bool hasIntersectedRegions(const QList<LRegion>& rs);

    static LRegion join(const LRegion& r1, const LRegion& r2);

    static int minDistance(const LRegion& r1, const LRegion& r2);

    static LRegion containingRegion(const LRegion& r1, const LRegion& r2);

    static LRegion containingRegion(const QList<LRegion>& regions);
    
	int startPos, len;
private:
    static bool registerMeta;
};

inline bool LRegion::intersects(const LRegion& r) const {
    int sd = startPos - r.startPos;
    return (sd >= 0) ? (sd < r.len) : (-sd < len);
}

inline LRegion LRegion::intersect(const LRegion& r) const {
	int newStart = qMax(startPos, r.startPos);
	int newEnd = qMin(endPos(), r.endPos());
	if (newStart > newEnd) {
		return LRegion();
	} 
	return LRegion(newStart, newEnd - newStart);
}


inline bool LRegion::intersectsWithAny(const QList<LRegion>& rs) const {
	bool res = false;
	foreach(const LRegion& r, rs) {
		res = intersects(startPos, endPos(), r.startPos, r.endPos());
		if (res) {
			break;
		}
	}
	return res;
}

inline QList<LRegion> LRegion::intersect(const LRegion& r, const QList<LRegion>& regions) {
	QList<LRegion> res;
	foreach(const LRegion& tmp, regions) {
		LRegion tmpRes = r.intersect(tmp);
		if (!tmpRes.isEmpty()) {
			res.append(tmpRes);
		}
	}
	return res;
}


inline bool LRegion::hasIntersectedRegions(const QList<LRegion>& rs) {
	for (int i=0, n = rs.size(); i<n; i++) {
		const LRegion& r1 = rs[i];
		for (int j=i+1; j<n; j++) {
			const LRegion& r2 = rs[j];
			if (r2.intersects(r1)) {
				return true;
			}
		}
	}
	return false;
}

inline LRegion LRegion::join(const LRegion& r1, const LRegion& r2) {
    assert(r1.intersects(r2));
    int newStart = qMin(r1.startPos, r2.startPos);
    int newEnd = qMax(r1.endPos(), r2.endPos());
    return LRegion(newStart, newEnd - newStart);
}

inline int LRegion::minDistance(const LRegion& r1, const LRegion& r2) {
    if (r1.intersects(r2)) {
        return 0;
    }
    if (r1.endPos() <= r2.startPos) {
        return r2.startPos - r1.endPos() + 1;        
    }
    assert(r2.endPos() <= r1.startPos);
    return r1.startPos - r2.endPos() + 1;
}


inline LRegion LRegion::containingRegion(const LRegion& r1, const LRegion& r2) {
    if (r1.intersects(r2)) {
        return join(r1, r2);
    }
    if (r1.startPos < r2.startPos) {
        return LRegion(r1.startPos, r2.endPos() - r1.startPos);
    }
    return LRegion(r2.startPos, r1.endPos() - r2.startPos);
}

inline LRegion LRegion::containingRegion(const QList<LRegion>& regions) {
    assert(!regions.isEmpty());

    LRegion res = regions.first();
    foreach(const LRegion& r, regions) {
        res = containingRegion(res, r);
    }
    return res;
}

QDataStream &operator<<(QDataStream &out, const LRegion &myObj);
QDataStream &operator>>(QDataStream &in, LRegion &myObj);

}//namespace

Q_DECLARE_TYPEINFO(GB2::LRegion, Q_PRIMITIVE_TYPE);
Q_DECLARE_METATYPE( GB2::LRegion)
Q_DECLARE_METATYPE( QList< GB2::LRegion >)

#endif
