/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <core_api/AppGlobals.h>
#include <core_api/AppContext.h>

#include "CMDLineRegistry.h"
#include "CMDLineHelpProvider.h"
#include "CMDLineCoreOptions.h"

#include <cstdio>

namespace GB2 {

/***************************************************
 * CMDLineRegistry
 ***************************************************/

const QString SINGLE_DASH   = "-";
const QString DOUBLE_DASH   = "--";
const QString EQUALS        = "=";

static bool isDoubleDashParameter(const QString& val) {
    return val.startsWith( DOUBLE_DASH ) && val.length() > 2 && val.at(2).isLetter();
}

static bool isSingleDashParameter(const QString& val) {
    return val.startsWith( SINGLE_DASH ) && val.length() > 1 && val.at(1).isLetter();
}

static bool tryParseDoubleDashParameter(const QString& argument, QString& paramName, QString &paramValue) {
    if (!isDoubleDashParameter(argument)) {
        return false;
    }
    int nameEndIdx = argument.indexOf( EQUALS );
    if (nameEndIdx == -1) {
        paramName = argument.mid(2);
    } else {
        paramName = argument.mid(2, nameEndIdx - 2);
        paramValue = argument.mid(nameEndIdx+1);
    }
    return true;
}

static bool tryParseSingleDashParameter(const QString& argument, const QString& nextArgument, QString& paramName, QString &paramValue) {
    if (!isSingleDashParameter(argument)) {
        return false;
    }
    paramName = argument.mid(1);
    if (!isDoubleDashParameter(nextArgument) && !isSingleDashParameter(nextArgument)) {
        paramValue = nextArgument;    
    }    
    return true;
}

CMDLineRegistry::CMDLineRegistry(const QStringList& arguments) {
    int sz = arguments.size();
    for( int i = 0; i < sz; i++ ) {
        const QString& argument = arguments.at( i );
        StringPair pair;
        if ( !tryParseDoubleDashParameter(argument, pair.first, pair.second) ) {
            QString nextArgument;
            if (i < sz - 1) {
                nextArgument = arguments.at(i + 1);
            }
            if ( tryParseSingleDashParameter(argument, nextArgument, pair.first, pair.second) ) {
                if (!pair.second.isEmpty()) {
                    i++;
                }
            } else {
                pair.second = argument;
            }
        }
        params << pair;
    }
}

CMDLineRegistry::~CMDLineRegistry() {
    qDeleteAll( helpProviders );
}

const QList<StringPair> & CMDLineRegistry::getParameters() const {
    return params;
}

QStringList CMDLineRegistry::getOrderedParameterNames() const {
    QStringList res;
    QList<StringPair>::const_iterator it = params.constBegin();
    while( it != params.constEnd() ) {
        res << it->first;
        ++it;
    }
    return res;
}

bool CMDLineRegistry::hasParameter( const QString & paramName, int startWithIdx ) const {
    int sz = params.size();
    for( int i = qMax(0, startWithIdx); i < sz; ++i ) {
        const StringPair& param = params[i];
        if( param.first == paramName ) {
            return true;
        }
    }
    return false;
}

QString CMDLineRegistry::getParameterValue( const QString & paramName, int startWithIdx ) const {
    int sz = params.size();
    for( int i = qMax(0, startWithIdx); i < sz; ++i ) {
        const StringPair& param = params[i];
        if( param.first == paramName ) {
            return param.second;
        }
    }
    return QString();
}

static bool providerNameComparator(const CMDLineHelpProvider* p1, const CMDLineHelpProvider* p2) {
    return p1->getHelpSectionName().compare(p2->getHelpSectionName()) > 0;
}

void CMDLineRegistry::registerCMDLineHelpProvider(CMDLineHelpProvider* provider) {
    helpProviders.append(provider);
    qStableSort(helpProviders.begin(), helpProviders.end(), providerNameComparator); 
}

void CMDLineRegistry::unregisterCMDLineHelpProvider(CMDLineHelpProvider* provider) {
    helpProviders.removeOne(provider);
}

void CMDLineRegistry::dumpProgramNameAndUsage() const {
    fprintf( stdout, "\n%s", QString( "Console version of UGENE %1.%2.%3\n" ).
        arg( UGENE_VERSION_1 ).arg( UGENE_VERSION_2 ).arg( UGENE_VERSION_3 ).toLocal8Bit().constData() );
    fprintf( stdout, 
        "Usage: ugene [paramValue1 paramValue2 ...] [--paramName=paramValue1 paramValue2 ...] [-paramName paramValue1 paramValue2 ...]\n" );
}

void CMDLineRegistry::dumpSectionName( const QString & name ) const {
    fprintf(stdout, "   %-20s", name.toLocal8Bit().constData() );
}

void CMDLineRegistry::dumpSectionContent( const QString & content ) const {
    fprintf(stdout, "\t%s", content.toLocal8Bit().constData() );
}

void CMDLineRegistry::dumpSectionIndent() const {
    fprintf(stdout, "%20s", " " );
}

void CMDLineRegistry::dumpHelp() const {
    dumpProgramNameAndUsage();
    fprintf( stdout, "\nOptions: \n" );
    
    QString prevSectionName;
    foreach (CMDLineHelpProvider* hProvider, helpProviders) {
        const QString& sectionName = hProvider->getHelpSectionName();
        if ( sectionName != prevSectionName) {
            dumpSectionName( sectionName );
            prevSectionName = sectionName;
        } else {
            dumpSectionIndent();
        }
        dumpSectionContent( hProvider->getHelpSectionContent() );
        fprintf(stdout, "\n");
    }
    fprintf( stdout, "\n" );
}

void CMDLineRegistry::sl_dumpHelp() {
    QString paramName = getParameterValue( CMDLineCoreOptions::HELP );
    if( paramName.isEmpty() ) {
        dumpHelp();
        return;
    }
    
    dumpProgramNameAndUsage();
    
    int ind = 0;
    int sz = helpProviders.size();
    for( ind = 0; ind < sz; ++ind ) {
        if( helpProviders.at(ind)->getHelpSectionName() == paramName ) {
            break;
        }
    }
    
    dumpSectionName( helpProviders.at(ind)->getHelpSectionName() );
    dumpSectionContent( helpProviders.at(ind)->getHelpSectionContent() );
    for( int i = ind + 1; i < sz; ++i ) {
        if( helpProviders.at(i)->getHelpSectionName() != paramName ) {
            break;
        }
        fprintf(stdout, "\n");
        dumpSectionIndent();
        dumpSectionContent( helpProviders.at(i)->getHelpSectionContent() );
    }
    fprintf( stdout, "\n" );
}

void CMDLineRegistry::dumpParameters() const {
    QList<StringPair>::const_iterator it = params.constBegin();
    while( it != params.constEnd() ) {
        fprintf( stdout, "key: \"%s\" and value: \"%s\"\n", it->first.toLocal8Bit().constData(), it->second.toLocal8Bit().constData() );
        ++it;
    }
}

} //namespace
