/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <QtGui/QMenu>
#include <QtCore/QCoreApplication>
#include <QtCore/QDir>
#include <QtCore/QDirIterator>
#include <QtScript/QScriptEngine>
#include <QtGui/QMessageBox>
 
#include <core_api/AppContext.h>
#include <core_api/DNAAlphabet.h>

#include <util_algorithm/GAutoDeleteList.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <util_ov_annotated_dna/ADVUtils.h>

#include <selection/DNASequenceSelection.h>
#include <gobjects/DNASequenceObject.h>
#include <util_gui/GUIUtils.h>

#include <test_framework/xmltest/XMLTestFormat.h>
#include <test_framework/GTest.h>
#include <test_framework/GTestFrameworkComponents.h>

#include "RemoteRequestPlugin.h"
#include "RemoteRequestTask.h"
#include "SendSelectionDialogImpl.h"
#include "ScriptEngineContext.h"
#include "RemoteRequestPluginTests.h"
#include "RemoteRequestWorker.h"

namespace GB2
{

extern "C" Q_DECL_EXPORT Plugin * GB2_PLUGIN_INIT_FUNC() {
    RemoteRequestPlugin * plug = new RemoteRequestPlugin();
    return plug;
}

RemoteRequestPlugin::RemoteRequestPlugin() : Plugin( tr("remote_request_plugin"), tr("remote_request_desc") ), ctx(NULL) {
    if (AppContext::getMainWindow()) {
        ctx = new RemoteRequestViewContext( this/*, &scripts */);
        ctx->init();
    }
    GTestFormatRegistry* tfr = AppContext::getTestFramework()->getTestFormatRegistry();
    XMLTestFormat *xmlTestFormat = qobject_cast<XMLTestFormat*>(tfr->findFormat("XML"));
    assert(xmlTestFormat!=NULL);

    GAutoDeleteList<XMLTestFactory>* l = new GAutoDeleteList<XMLTestFactory>(this);
    l->qlist = RemoteRequestPluginTests::createTestFactories();

    foreach(XMLTestFactory* f, l->qlist) { 
        bool res = xmlTestFormat->registerTestFactory(f);
        Q_UNUSED(res);
        assert(res);
    }

    LocalWorkflow::RemoteRequestWorkerFactory::init();
}
RemoteRequestViewContext::RemoteRequestViewContext( QObject * p/*, ScriptsRegistry * _scripts */) : 
GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)/*, scripts(_scripts)*/
{
}

void RemoteRequestViewContext::initViewContext( GObjectView* view ) {
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(view);
    ADVGlobalAction * a = new ADVGlobalAction(av, QIcon(":/remote_request/images/remote_db_request.png"), tr("send_selection"), 60);
    connect( a, SIGNAL(triggered()), SLOT(sl_showDialog()) );
}


namespace {
    void filterReadyScripts( QList<Script *> * all_scripts ) {
        assert( all_scripts );
        foreach( Script * scr, *all_scripts ) {
            if( !scr->isReady() ) {
                all_scripts->removeAll( scr );
            }
        }
    }
}

void RemoteRequestViewContext::sl_showDialog() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);

    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();

    ServiceRegistry * sr = AppContext::getServiceRegistry();
    ScriptRegistryService * srs = qobject_cast<ScriptRegistryService*>( sr->findServices(Service_ScriptRegistry)[0] );
    assert( srs );

    //check scripts presence
    QList<Script *> scripts = srs->getScriptsByType( ScriptTypes::SCRIPT_TYPE_HTTP_ANNOTATOR );
    filterReadyScripts( &scripts );
    if( scripts.isEmpty() ) {
        QMessageBox::warning( av->getWidget(), tr("Warning"), tr("no_ready_scripts") );
        return;
    }
    
    //check selection non-emptiness
    DNASequenceSelection* s = seqCtx->getSequenceSelection();
    if (s->isEmpty()) {
        QMessageBox::warning( av->getWidget(), tr("Warning"), tr("no_sequence_selected") );
        return;
    }
    bool isAminoSeq = seqCtx->getAlphabet()->isAmino();
    SendSelectionDialogImpl dlg( scripts, seqCtx->getSequenceObject(), isAminoSeq, av->getWidget() );
    if( QDialog::Accepted == dlg.exec() ) {
        if( isAminoSeq && SendSelectionAlphabet_Nucleo == dlg.alpha ) {
            QMessageBox::warning( av->getWidget(), tr("Warning"), tr("cannot_send_amino_query") );
            return;
        }

        Script * scr = dlg.chosen;
		DNATranslation * complT = (dlg.strand != SendSelectionStrand_Direct ? seqCtx->getComplementTT() : 0);
        DNATranslation * aminoT = (dlg.alpha == SendSelectionAlphabet_Amino ? seqCtx->getAminoTT() : 0);

        //prepare query
        const QByteArray& sequence = seqCtx->getSequenceData();
        const QList<LRegion>& regions =  s->getSelectedRegions();
        foreach(const LRegion& r, regions) {
            QByteArray query( sequence.constData() + r.startPos, r.len );
  
            //Check selection size
            int sel_size = query.size() / ( aminoT ? 3 : 1 );                                
            if( dlg.maxQueryLen && sel_size > dlg.maxQueryLen ) {
                if( QMessageBox::Ok != QMessageBox::question( av->getWidget(), 
                    tr("Warning"), tr("too_large_selection_size"), QMessageBox::Ok, QMessageBox::Cancel )) 
                {
                    return;
                }
                else {
                    int from = dlg.maxQueryLen * (aminoT ? 3 : 1);
                    int howmuch = query.size() - from;
                    query.remove( from, howmuch );
                }
            }

            Task * t = new RemoteRequestToAnnotationsTask( scr, dlg.maxResultLen, dlg.minResultLen, dlg.strand, complT, aminoT, query, 
                                              r.startPos, dlg.getAnnotationObject(), dlg.getGroupName() );
            AppContext::getTaskScheduler()->registerTopLevelTask( t );
        }
    }
}

QList<XMLTestFactory*> RemoteRequestPluginTests::createTestFactories() {
    QList<XMLTestFactory*> res;
    res.append(GTest_RemoteRequest::createFactory());
    return res;
}

} //namespace
