/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "FindEnzymesDialog.h"
#include "FindEnzymesTask.h"
#include "EnzymesIO.h"
#include <limits>

#include <core_api/GObjectReference.h>
#include <core_api/AppContext.h>
#include <core_api/Settings.h>
#include <core_api/Log.h>
#include <core_api/Timer.h>

#include <util_gui/CreateAnnotationWidgetController.h>
#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <selection/DNASequenceSelection.h>
#include <gobjects/DNASequenceObject.h>

#include <QtCore/QCoreApplication>
#include <QtCore/QFileInfo>
#include <QtCore/QDir>
#include <QtGui/QVBoxLayout>
#include <QtGui/QTreeWidget>
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>
#include <QtGui/QHeaderView>

//TODO: group by TYPE, ORGANIZM
//TODO: check whole group (tristate mode)

namespace GB2 {

#define DATA_DIR_KEY            QString("enzymes")
#define DATA_FILE_KEY           QString("plugin_enzymes/lastFile")
#define LAST_SELECTION_KEY      QString("plugin_enzymes/selection")
#define SEP                     (",")
#define DEFAULT_ENZYMES_FILE    QString("rebase_v901_t2_com.bairoch.gz")
#define ANY_VALUE -1

static LogCategory log(ULOG_ENZYME_PLUGIN);

QList<SEnzymeData>  FindEnzymesDialog::loadedEnzymes;
QSet<QString>       FindEnzymesDialog::lastSelection;


FindEnzymesDialog::FindEnzymesDialog(ADVSequenceObjectContext* sctx) 
: QDialog(sctx->getAnnotatedDNAView()->getWidget()), seqCtx(sctx) {
    setupUi(this);
    ignoreItemChecks = false;

    splitter->setStretchFactor(0, 5);
    splitter->setStretchFactor(1, 1);

    tree->setSortingEnabled(true);
    tree->sortByColumn(0, Qt::AscendingOrder);
    tree->setUniformRowHeights(true);
    tree->setColumnWidth(0, 110); //id
    tree->setColumnWidth(1, 75); //accession
    tree->setColumnWidth(2, 50); //type
    
    sbRangeStart->setMaximum(seqCtx->getSequenceLen());
    sbRangeEnd->setMaximum(seqCtx->getSequenceLen());
    sbRangeEnd->setValue(seqCtx->getSequenceLen());
    bool hasSelection = !seqCtx->getSequenceSelection()->isEmpty();
    rbSelectionRange->setEnabled(hasSelection);
    rbSelectionRange->setChecked(hasSelection);
    totalEnzymes = 0;

    maxHitSB->setMaximum(INT_MAX);
    minHitSB->setMaximum(INT_MAX);

    maxHitSB->setMinimum(ANY_VALUE);
    minHitSB->setMinimum(ANY_VALUE);

    minHitSB->setSpecialValueText(tr("Any"));
    maxHitSB->setSpecialValueText(tr("Any"));

    maxHitSB->setValue(ANY_VALUE);
    minHitSB->setValue(ANY_VALUE);

    connect(enzymesFileButton, SIGNAL(clicked()), SLOT(sl_selectFile()));
    connect(selectAllButton, SIGNAL(clicked()), SLOT(sl_selectAll()));
    connect(selectNoneButton, SIGNAL(clicked()), SLOT(sl_selectNone()));
    connect(invertSelectionButton, SIGNAL(clicked()), SLOT(sl_inverseSelection()));
    connect(enzymeInfo, SIGNAL(clicked()), SLOT(sl_openDBPage()));

    CreateAnnotationModel acm;
    acm.sequenceObjectRef = GObjectReference(sctx->getSequenceGObject());
    acm.hideAnnotationName = true;
    acm.hideLocation = true;
    acm.data->name = "enzyme";
    acm.sequenceLen = sctx->getSequenceObject()->getSequenceLen();
    ac = new CreateAnnotationWidgetController(acm, this);
    QWidget* caw = ac->getWidget();    
    QVBoxLayout* l = new QVBoxLayout();
    l->setMargin(0);
    l->addWidget(caw);
    annotationsWidget->setLayout(l);
    annotationsWidget->setMinimumSize(caw->layout()->minimumSize());
    
    if (loadedEnzymes.isEmpty()) {
        QString lastUsedFile = AppContext::getSettings()->getValue(DATA_FILE_KEY).toString();
        loadFile(lastUsedFile);
    } else {
        setEnzymesList(loadedEnzymes);
    }
}

FindEnzymesDialog::~FindEnzymesDialog() {
    QStringList sl(lastSelection.toList());
    AppContext::getSettings()->setValue(LAST_SELECTION_KEY, sl.join(SEP));
}

void FindEnzymesDialog::initSettings() {
    QString dir = DialogUtils::getLastOpenFileDir(DATA_DIR_KEY);
    if (dir.isEmpty() || !QDir(dir).exists()) {
        dir = QDir::searchPaths( PATH_PREFIX_DATA ).first() + "/enzymes/";
        DialogUtils::setLastOpenFileDir(dir, DATA_DIR_KEY);
    }
    QString lastEnzFile = AppContext::getSettings()->getValue(DATA_FILE_KEY).toString();
    if (lastEnzFile.isEmpty() || !QFile::exists(lastEnzFile)) {
        lastEnzFile = dir + DEFAULT_ENZYMES_FILE;
        AppContext::getSettings()->setValue(DATA_FILE_KEY, lastEnzFile);
    }
    QString selStr = AppContext::getSettings()->getValue(LAST_SELECTION_KEY).toString();
    lastSelection = selStr.split(SEP).toSet();
}

void FindEnzymesDialog::loadFile(const QString& url) {
    TaskStateInfo ti;
    QList<SEnzymeData> enzymes;

    if (!QFileInfo(url).exists()) {
        ti.setError(  tr("File not exists: %1").arg(url) );
    } else {
        GTIMER(c1,t1,"FindEnzymesDialog::loadFile [EnzymesIO::readEnzymes]");
        enzymes = EnzymesIO::readEnzymes(url, ti);
    }
    if (ti.hasErrors()) {
        if (isVisible()) {
            QMessageBox::critical(NULL, tr("Error"), ti.getError());
        } else  {
            log.error(ti.getError());
        }
        return;
    }
    if (!enzymes.isEmpty()) {
        if (AppContext::getSettings()->getValue(DATA_FILE_KEY).toString() != url) {
            lastSelection.clear();
        }
        AppContext::getSettings()->setValue(DATA_FILE_KEY, url);
    }

    setEnzymesList(enzymes);
}

void FindEnzymesDialog::setEnzymesList(const QList<SEnzymeData>& enzymes) {
    tree->setSortingEnabled(false);
    tree->disconnect(this);
    tree->clear();
    totalEnzymes = 0;
    
    GTIMER(c2,t2,"FindEnzymesDialog::loadFile [refill data tree]");
    foreach (const SEnzymeData& enz, enzymes) {
        EnzymeTreeItem* item = new EnzymeTreeItem(enz);
        if (lastSelection.contains(enz->id)) {
            item->setCheckState(0, Qt::Checked);
        }
        totalEnzymes++;
        EnzymeGroupTreeItem* gi = findGroupItem(enz->id.isEmpty() ? QString(" ") : enz->id.left(1), true);
        gi->addChild(item);
        
    }
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        gi->updateVisual();
    }
    if (tree->topLevelItemCount() > 0 && tree->topLevelItem(0)->childCount() < 10) {
        tree->topLevelItem(0)->setExpanded(true);
    }
    t2.stop();

    GTIMER(c3,t3,"FindEnzymesDialog::loadFile [sort tree]");
    tree->setSortingEnabled(true);
    t3.stop();

    connect(tree, SIGNAL(itemChanged(QTreeWidgetItem*,int)), SLOT(sl_itemChanged(QTreeWidgetItem*, int)));


//     GTIMER(c4,t4,"FindEnzymesDialog::loadFile [resize tree]");
//     tree->header()->resizeSections(QHeaderView::ResizeToContents);
//     t4.stop();

    updateStatus();
    loadedEnzymes = enzymes;
}

EnzymeGroupTreeItem* FindEnzymesDialog::findGroupItem(const QString& s, bool create) {
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++) {
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        if (gi->s == s) {
            return gi;
        }
    }
    if (create) {
        EnzymeGroupTreeItem* gi = new EnzymeGroupTreeItem(s);
        tree->addTopLevelItem(gi);
        return gi;
    }
    return NULL;
}

void FindEnzymesDialog::updateStatus() {
    int nChecked = 0;
    QStringList checkedNamesList;
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++){
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        nChecked+= gi->checkedEnzymes.size();
        foreach(const EnzymeTreeItem* ci, gi->checkedEnzymes) {
            checkedNamesList.append(ci->enzyme->id);
        }
    }
    checkedNamesList.sort();
    checkedEnzymesEdit->setPlainText(checkedNamesList.join(","));

    statusLabel->setText(tr("Number of enzymes: %1, selected %2").arg(totalEnzymes).arg(nChecked));
}

void FindEnzymesDialog::sl_selectFile() {
    LastOpenDirHelper dir(DATA_DIR_KEY);
    dir.url = QFileDialog::getOpenFileName(this, tr("Select enzyme database file"), dir.dir, EnzymesIO::getFileDialogFilter());
    if (!dir.url.isEmpty()) {
        loadFile(dir.url);
    }
}

void FindEnzymesDialog::sl_selectAll() {
    ignoreItemChecks =  true;
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++){
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        for (int j=0, m = gi->childCount(); j < m; j++) {
            EnzymeTreeItem* item = (EnzymeTreeItem*)gi->child(j);
            item->setCheckState(0, Qt::Checked);
        }
        gi->updateVisual();
    }
    ignoreItemChecks = false;
    updateStatus();
}

void FindEnzymesDialog::sl_selectNone() {
    ignoreItemChecks = true;
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++){
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        for (int j=0, m = gi->childCount(); j < m; j++) {
            EnzymeTreeItem* item = (EnzymeTreeItem*)gi->child(j);
            item->setCheckState(0, Qt::Unchecked);
        }
        gi->updateVisual();
    }
    ignoreItemChecks = false;
    updateStatus();
}

void FindEnzymesDialog::sl_inverseSelection() {
    ignoreItemChecks = true;
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++){
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        for (int j=0, m = gi->childCount(); j < m; j++) {
            EnzymeTreeItem* item = (EnzymeTreeItem*)gi->child(j);
            Qt::CheckState oldState = item->checkState(0);
            item->setCheckState(0, oldState == Qt::Checked ? Qt::Unchecked : Qt::Checked);
        }
        gi->updateVisual();
    }
    ignoreItemChecks = false;
    updateStatus();
}

#define MAX_ENZYMES_TO_FIND 100*1000

void FindEnzymesDialog::accept() {
    QList<SEnzymeData> selectedEnzymes;
    lastSelection.clear();
    for(int i=0, n = tree->topLevelItemCount(); i<n; i++){
        EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)tree->topLevelItem(i);
        for (int j=0, m = gi->childCount(); j < m; j++) {
            EnzymeTreeItem* item = (EnzymeTreeItem*)gi->child(j);
            if (item->checkState(0) == Qt::Checked) {
                selectedEnzymes.append(item->enzyme);
                lastSelection.insert(item->enzyme->id);
            }
        }
    }
    
    if (selectedEnzymes.isEmpty()) {
        QMessageBox::critical(this, tr("Error!"), tr("No enzymes selected!"));
        return;
    }
    
    QString err = ac->validate();
    if (!err.isEmpty()) {
        QMessageBox::critical(this, tr("Error!"), err);
        return;
    }
    
    int maxHitVal = maxHitSB->value(), minHitVal = minHitSB->value();
    if(maxHitVal == ANY_VALUE){
        maxHitVal = INT_MAX;
    }
    if (minHitVal == ANY_VALUE){
        minHitVal = 1;
    }

    if (minHitVal > maxHitVal) {
        QMessageBox::critical(this, tr("Error!"), tr("Minimum hit value must be lesser or equal then maximum!"));
        return;
    }

    LRegion range;
    if (rbSequenceRange->isEnabled()) {
        range = LRegion(0, seqCtx->getSequenceLen());
    } else if (rbSelectionRange->isEnabled() && !seqCtx->getSequenceSelection()->getSelectedRegions().isEmpty()) {
        range = seqCtx->getSequenceSelection()->getSelectedRegions().first();
    } else {
        range.startPos = qMin(sbRangeStart->value(), sbRangeEnd->value());
        range.len = qMax(sbRangeStart->value(), sbRangeEnd->value()) - range.startPos;
        if (range.len == 0) {
            QMessageBox::critical(this, tr("Error!"), tr("Invalid range!"));
            sbRangeStart->setFocus();
            return;
        }
    }
    ac->prepareAnnotationObject();
    const CreateAnnotationModel& m = ac->getModel();
    QString group = m.groupName;
    AnnotationTableObject* obj = m.getAnnotationObject();
    assert(!group.isEmpty() && obj!=NULL);
    
    FindEnzymesToAnnotationsTask* task = new FindEnzymesToAnnotationsTask(obj, group, seqCtx->getSequenceObject()->getDNASequence(),
                                                    range, selectedEnzymes, MAX_ENZYMES_TO_FIND, minHitVal, maxHitVal); 
    AppContext::getTaskScheduler()->registerTopLevelTask(task);
  
    QDialog::accept();
}

void FindEnzymesDialog::sl_openDBPage() {
    QTreeWidgetItem* ci = tree->currentItem();
    EnzymeTreeItem* item = ci == NULL || ci->parent() == 0 ? NULL : (EnzymeTreeItem*)tree->currentItem();
    if (item == NULL) {
        QMessageBox::critical(this, tr("Error!"), tr("No enzyme selected!"));
        return;
    }
    QString suffix = item->enzyme->accession;
    if (suffix.isEmpty()){
        suffix = item->enzyme->id;
    } else if (suffix.startsWith("RB")) {
        suffix = suffix.mid(2);
        
    }
    GUIUtils::runWebBrowser("http://rebase.neb.com/cgi-bin/reb_get.pl?enzname="+suffix);
}

void FindEnzymesDialog::sl_itemChanged(QTreeWidgetItem* item, int col) {
    if (item->parent() == NULL  || col != 0 || ignoreItemChecks) {
        return;
    }
    EnzymeTreeItem* ei = (EnzymeTreeItem*)item;
    EnzymeGroupTreeItem* gi = (EnzymeGroupTreeItem*)ei->parent();
    gi->updateVisual();
    updateStatus();
}

//////////////////////////////////////////////////////////////////////////
// Tree item
EnzymeTreeItem::EnzymeTreeItem(const SEnzymeData& ed) 
: enzyme(ed)
{
    setText(0, enzyme->id);  
    setCheckState(0, Qt::Unchecked);
    setText(1, enzyme->accession);
    setText(2, enzyme->type);
    setText(3, enzyme->seq);
    setData(3, Qt::ToolTipRole, enzyme->seq);
    setText(4, enzyme->organizm);//todo: show cut sites
    setData(4, Qt::ToolTipRole, enzyme->organizm);
}

bool EnzymeTreeItem::operator<(const QTreeWidgetItem & other) const {
    int col = treeWidget()->sortColumn();
    const EnzymeTreeItem& ei = (const EnzymeTreeItem&)other;
    if (col == 0) {
        bool eq = enzyme->id == ei.enzyme->id;
        if (!eq) {
            return enzyme->id < ei.enzyme->id;
        }
        return this < &ei;
    } 
    return text(col) < ei.text(col);
}

EnzymeGroupTreeItem::EnzymeGroupTreeItem(const QString& _s) : s(_s){
    updateVisual();
}

void EnzymeGroupTreeItem::updateVisual() {
    int numChilds = childCount();
    checkedEnzymes.clear();
    for (int i=0; i < numChilds; i++) {
        EnzymeTreeItem* item = (EnzymeTreeItem*)child(i);
        if (item->checkState(0) == Qt::Checked) {
            checkedEnzymes.insert(item);
        }
    }
    QString text0 = s + " (" + QString::number(checkedEnzymes.size()) + ", " + QString::number(numChilds) + ")";
    setText(0, text0);
    
    if (numChilds > 0) {
        QString text4 = ((EnzymeTreeItem*)child(0))->enzyme->id;
        if (childCount() > 1) {
            text4+=" .. "+((EnzymeTreeItem*)child(numChilds-1))->enzyme->id;
        }
        setText(4, text4);
    }
}

bool EnzymeGroupTreeItem::operator<(const QTreeWidgetItem & other) const {
    if (other.parent() != NULL) {
        return true;
    }
    int col = treeWidget()->sortColumn();
    return text(col) <other.text(col);
}

} //namespace
