/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "SelectionUtils.h"
#include "DocumentSelection.h"
#include "GObjectSelection.h"
#include "SelectionTypes.h"

#include <core_api/GObject.h>
#include <core_api/DocumentModel.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/UnloadedObject.h>

#include <core_api/SelectionModel.h>

namespace GB2 {

QSet<GObject*> SelectionUtils::findObjects(GObjectType t, const MultiGSelection* ms, UnloadedObjectFilter f) {
	QSet<GObject*> res;
	foreach(const GSelection* s, ms->getSelections()) {
		QSet<GObject*> tmp = findObjects(t, s, f);
		res+=tmp;
	}
	return res;
}

QSet<GObject*> SelectionUtils::findObjects(GObjectType t, const GSelection* s, UnloadedObjectFilter f) {
	QSet<GObject*> res;
	GSelectionType stype = s->getSelectionType();
	if (stype == GSelectionTypes::DOCUMENTS) {
		const DocumentSelection* ds = qobject_cast<const DocumentSelection*>(s);
		foreach(Document* d, ds->getSelectedDocuments()) {
			const QList<GObject*>& objs= d->getObjects();
            QSet<GObject*> tmp = GObjectUtils::select(objs, t, f).toSet();
			res+=tmp;
		}
	} else  if (stype == GSelectionTypes::GOBJECTS) {
		const GObjectSelection* os = qobject_cast<const GObjectSelection*>(s);
		const QList<GObject*>& objs= os->getSelectedObjects();
		res = GObjectUtils::select(objs, t, f).toSet();
	}
	return res;
}


QSet<Document*>	SelectionUtils::findDocumentsWithObjects(GObjectType t, const MultiGSelection* ms, UnloadedObjectFilter f, bool deriveDocsFromObjectSelection) {
	QSet<Document*> res;
	foreach(const GSelection* s, ms->getSelections()) {
		res+=findDocumentsWithObjects(t, s, f, deriveDocsFromObjectSelection);
	}
	return res;
}

QSet<Document*>	SelectionUtils::findDocumentsWithObjects(GObjectType t, const GSelection* s, UnloadedObjectFilter f, bool deriveDocsFromObjectSelection) {
	QSet<Document*> res;
	GSelectionType st = s->getSelectionType();
	if (st == GSelectionTypes::DOCUMENTS) {
		DocumentFormatConstraints c;
		c.supportedObjectTypes.append(t);
		const DocumentSelection* ds = qobject_cast<const DocumentSelection*>(s);
		const QList<Document*>& docs = ds->getSelectedDocuments();
		foreach(Document* d, docs) {
			if (!d->getObjects().isEmpty()) {
                QList<GObject*> objs = d->findGObjectByType(t, f);
				if (!objs.isEmpty()) {
					res+=d;
				}
			} else if (f == UOF_LoadedAndUnloaded && !d->isLoaded()) { //document is unloaded
				DocumentFormat* df = d->getDocumentFormat();
				if (df->checkConstraints(c)) {
					res+=d;
				}
			}
		}
	} else if (st == GSelectionTypes::GOBJECTS && deriveDocsFromObjectSelection) {
		QSet<GObject*> objects = findObjects(t, s, f);
		foreach(GObject* o, objects) {
			res+=o->getDocument();
		}
	}
	return res;
}

bool SelectionUtils::isDocumentInSelection(const Document* doc, const MultiGSelection& ms, bool deriveDocsFromObjectSelection) {
	foreach(const GSelection* s, ms.getSelections()) {
		GSelectionType st = s->getSelectionType();
		if (st == GSelectionTypes::DOCUMENTS) {
			const DocumentSelection* ds = qobject_cast<const DocumentSelection*>(s);
			const QList<Document*>& docs = ds->getSelectedDocuments();
			if (docs.contains((Document* const &)doc)) {//TODO? why cast
				return true;
			}
		} else if (st == GSelectionTypes::GOBJECTS && deriveDocsFromObjectSelection) {
			const GObjectSelection* os = qobject_cast<const GObjectSelection*>(s);
			const QList<GObject*>& objects = os->getSelectedObjects();
			foreach(GObject* o, objects) {
				if (o->getDocument() == doc) {
					return true;
				}
			}
		}
	}
	return false;
}

QList<Document*> SelectionUtils::getSelectedDocs(const MultiGSelection& ms) {
	foreach(const GSelection* s, ms.getSelections()) {
		GSelectionType st = s->getSelectionType();
		if (st == GSelectionTypes::DOCUMENTS) {
			const DocumentSelection* ds = qobject_cast<const DocumentSelection*>(s);
			return  ds->getSelectedDocuments();
		}
	}
	return QList<Document*>();
}

QList<GObject*> SelectionUtils::getSelectedObjects(const MultiGSelection& ms) {
	foreach(const GSelection* s, ms.getSelections()) {
		GSelectionType st = s->getSelectionType();
		if (st == GSelectionTypes::GOBJECTS) {
			const GObjectSelection* os = qobject_cast<const GObjectSelection*>(s);
			return os->getSelectedObjects();
		}
	}
	return QList<GObject*>();
}


LRegion SelectionUtils::normalizeRegionBy3(LRegion reg, int seqLen, bool direct) {
    assert(reg.len > 0);
    if ( seqLen < 3 ){
        return reg;
    }
    int d = reg.len % 3;
    if (d == 1) {
        if (direct) {
            reg.len-=1;
        } else {
            reg.startPos+=1;
            reg.len-=1;
        }
    } else if (d == 2) {
        if (direct) {
            reg.len+=(reg.len+1 < seqLen) ? +1 : -2;
        } else {
            int prevStart = reg.startPos;
            reg.startPos+=(reg.startPos > 0) ? -1 : +2;
            reg.len+=prevStart - reg.startPos;
        }
    }
    return reg;
}



}//namespace
