/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "LogView.h"
#include "LogCache.h"

#include "../app_settings/logview_settings/LogSettingsGUIController.h"

#include <core_api/MainWindow.h>
#include <core_api/AppContext.h>
#include <core_api/Timer.h>
#include <core_api/Counter.h>

#include <QtCore/QDate>
#include <QtCore/QThread>
#include <QtGui/QAction>
#include <QtGui/QMenu>
#include <QtGui/QApplication>

namespace GB2 {

static LogCategory logPerf(ULOG_CAT_PERFORMANCE);


static void checkThread() {
#ifdef _DEBUG
    QThread* appThread = QApplication::instance()->thread();
    QThread* thisThread = QThread::currentThread();
    assert (appThread == thisThread);
#endif
}


#define MAX_VISIBLE_MESSAGES 1000

LogViewDockWidget::LogViewDockWidget(LogCache* _c)  {
    connected = false;

    setObjectName(DOCK_LOG_VIEW);
    setWindowTitle(tr("Log"));
    setWindowIcon(QIcon(":ugene/images/book_open.png"));

    cache = _c;

    settings.reinitAll();

    AppContext::getAppSettingsGUI()->registerPage(new LogSettingsPageController(this), APP_SETTINGS_USER_APPS);

    showSettingsAction = new QAction(tr("log_settings"), this);
    showSettingsAction->setIcon(QIcon(":ugene/images/log_settings.png"));
    connect(showSettingsAction, SIGNAL(triggered()), SLOT(sl_openSettingsDialog()));
    
    dumpCountersAction = new QAction(tr("Dump performance counters"), this);
    connect(dumpCountersAction, SIGNAL(triggered()), SLOT(sl_dumpCounters()));

    addSeparatorAction = new QAction(tr("Append separator"), this);
    connect(addSeparatorAction, SIGNAL(triggered()), SLOT(sl_addSeparator()));
    clearAction = new QAction(tr("Clear log"), this);
    connect(clearAction, SIGNAL(triggered()), SLOT(sl_clear()));

    QVBoxLayout* l = new QVBoxLayout();
    l->setSpacing(0);
    l->setMargin(0);
    l->setContentsMargins(0, 0, 0, 0);
    setLayout(l);

    edit = new QPlainTextEdit(); 
    edit->setUndoRedoEnabled(false);
    edit->setReadOnly(true);
    edit->setLineWrapMode(QPlainTextEdit::WidgetWidth);
    edit->setContextMenuPolicy(Qt::CustomContextMenu);
    edit->setTextInteractionFlags(Qt::NoTextInteraction|Qt::TextSelectableByMouse|Qt::TextSelectableByKeyboard);
    edit->setMaximumBlockCount(MAX_VISIBLE_MESSAGES);
    
    l->addWidget(edit);

    
    QObject::connect(edit, SIGNAL(customContextMenuRequested(const QPoint &)),this,SLOT(popupMenu(const QPoint &)));
    resetView();
}

void LogViewDockWidget::sl_openSettingsDialog() {
    AppContext::getAppSettingsGUI()->showSettingsDialog(APP_SETTINGS_GUI_LOG);
}



void LogViewDockWidget::popupMenu(const QPoint& pos) {
    Q_UNUSED(pos);

    QMenu popup;
    QAction* copyAction = popup.addAction(tr("logview_copy"), edit, SLOT(copy()));
    copyAction->setShortcut(QKeySequence(Qt::CTRL | Qt::Key_C));
    copyAction->setEnabled(edit->textCursor().hasSelection());
    popup.addAction(dumpCountersAction);
    popup.addAction(addSeparatorAction);
    popup.addAction(clearAction);
    popup.addAction(showSettingsAction);
    popup.exec(QCursor::pos());
}

void LogViewDockWidget::resetView() {
    QTime startTime = QTime::currentTime();
    
    edit->clear();
    
    QList<LogMessage*> messagesToShow;
    for (int i= cache->messages.size(); --i>=0;) {
        LogMessage* m = cache->messages[i];
        if (isShown(*m)) {
            messagesToShow.prepend(m);
            if (messagesToShow.count() == MAX_VISIBLE_MESSAGES) {
                break;    
            }
        }
    }
    foreach(LogMessage* m, messagesToShow) {
        edit->appendHtml(prepareText(*m));
    }
    edit->moveCursor(QTextCursor::End);
    edit->moveCursor(QTextCursor::StartOfLine);
    edit->ensureCursorVisible();

    QTime endTime = QTime::currentTime();
    logPerf.trace(QString("Log view update time %1 millis").arg(startTime.msecsTo(endTime)));
}

void LogViewDockWidget::sl_onMessage(const LogMessage& msg) {
    checkThread();
    if (isVisible()) {
        addMessage(msg);
    }
}

void LogViewDockWidget::showEvent(QShowEvent *e) {
    Q_UNUSED(e);
    assert(!connected);
    connect(LogServer::getInstance(), SIGNAL(si_message(const LogMessage&)), SLOT(sl_onMessage(const LogMessage&)));
    connected = true;
    resetView();
}

void LogViewDockWidget::hideEvent(QHideEvent *e) {
    Q_UNUSED(e);
    
    //do not use any resources when hidden
    LogServer::getInstance()->disconnect(this);
    connected = false;
    edit->clear(); 
}

bool LogViewDockWidget::isShown(const LogMessage& msg)  {
    if (!settings.activeLevelGlobalFlag[msg.level]) {
        return false;
    }

    const LogCategorySettings& cs = settings.getLogCategorySettings(msg.category);
    if (!cs.activeLevelFlag[msg.level]) {
        return false;
    }
    return true;
}

void LogViewDockWidget::setSettings(const LogSettings& s) {
    if (settings == s) {
        return;
    }
    settings = s;
    settings.save();
    resetView();
}


QString LogViewDockWidget::prepareText(const LogMessage& msg) const {
    QString color = settings.levelColors[msg.level];
    QString date = settings.showDate ? "["+GTimer::createDateTime(msg.time).toString("hh:mm") + "]" : QString();
    QString category = settings.showCategory ? "["+ msg.category + "]" : QString();
    QString level = settings.showLevel ? "["+ LogCategories::getLocalizedLevelName(msg.level)+"] " : QString();
    QString spacing = date.isEmpty() && category.isEmpty() && level.isEmpty() ? QString() : QString(" ");
    QString text = "<font color="+color+">" + date + category + level + spacing + msg.text+"</font><br/>";

    return text;
}

void LogViewDockWidget::addMessage(const LogMessage& msg) {
    if (!isShown(msg)) {
        return;
    }
    addText(prepareText(msg));
}

void LogViewDockWidget::addText(const QString& txt) {
    edit->appendHtml(txt);
    edit->moveCursor(QTextCursor::End);
    edit->moveCursor(QTextCursor::StartOfLine);
    edit->ensureCursorVisible();
}

void LogViewDockWidget::sl_logSettingsChanged() {
    if (isVisible()) {
        resetView();
    }
}

void LogViewDockWidget::sl_dumpCounters() {
    addText("Counters report start    ***********************\n");
    foreach(GCounter* c, GCounter::allCounters()) {
        double val = c->scaledTotal();
        QString text = c->name + " " + QString::number(val) + " " + c->suffix;
        addText(text);
    }
    addText("Counters report end    ***********************\n");
}

void LogViewDockWidget::sl_addSeparator() {
    addText("\n==================================================\n");
}

void LogViewDockWidget::sl_clear() {
    cache->messages.clear();
    edit->clear(); 
}
}//namespace

