/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include <core_api/AppContext.h>
#include <util_tasks/CreateAnnotationTask.h>
#include "Primer3Task.h"

namespace GB2 {

Primer3Task::Primer3Task(primer_args *pa, seq_args *sa): Task(tr("Pick primers task"), TaskFlag_None), pa(pa), sa(sa) {}
void Primer3Task::run(){         
    primers = runPrimer3( pa, sa, &stateInfo.cancelFlag, &stateInfo.progress );
}

Task::ReportResult Primer3Task::report() {
    return ReportResult_Finished;
}

pair_array_t Primer3Task::getBestPairs() {
    return primers.best_pairs;
}

primer_rec *Primer3Task::getLeftPrimers() {
    return primers.left;
}

primer_rec *Primer3Task::getRightPrimers() {
    return primers.right;
}

primer_rec *Primer3Task::getOligos() {
    return primers.intl;
}

void Primer3Task::cleanup() {
    if (NULL != sa->internal_input) free(sa->internal_input);
    if (NULL != sa->left_input) free(sa->left_input);
    if (NULL != sa->right_input) free(sa->right_input);    
    if (NULL != sa->sequence) free(sa->sequence);
    if (NULL != sa->quality)  free(sa->quality);
    if (NULL != sa->trimmed_seq) free(sa->trimmed_seq);

    /* edited by T. Koressaar for lowercase masking */
    if (NULL != sa->trimmed_orig_seq) free(sa->trimmed_orig_seq);

    if (NULL != sa->upcased_seq) free(sa->upcased_seq);
    if (NULL != sa->upcased_seq_r) free(sa->upcased_seq_r);
    if (NULL != sa->sequence_name) free(sa->sequence_name);
    if (NULL != sa->sequence_file) free(sa->sequence_file);    
    if (NULL != sa->error.data) free(sa->error.data);
    if (NULL != sa->warning.data) free(sa->warning.data); 

    if (primers.left != NULL) free(primers.left);
    if (primers.right != NULL) free(primers.right);
    if (primers.intl != NULL) free(primers.intl);
    if (primers.best_pairs.num_pairs > 0)        
        free(primers.best_pairs.pairs); 
    free(sa);
    free(pa);
}

//////////////////////////////////////////////////////////////////////////
////Primer3ToAnnotationsTask

Primer3ToAnnotationsTask::Primer3ToAnnotationsTask( primer_args * pa_, seq_args * sa_, 
AnnotationTableObject * aobj_, const QString & groupName_, const QString & annName_ ) :
Task(tr("Search primers to annotations"), TaskFlags_NR_FOSCOE), pa(pa_), sa(sa_), aobj(aobj_), groupName(groupName_), annName(annName_)
{
    searchTask = new Primer3Task( pa, sa );
    addSubTask(searchTask);
}

QList<Task *> Primer3ToAnnotationsTask::onSubTaskFinished( Task* subTask ) {
    assert( searchTask == subTask );
    if( searchTask == subTask && !subTask->hasErrors() && !subTask->isCanceled() ) {
        processResults();
    } 
    return QList<Task *>();
}

void Primer3ToAnnotationsTask::processResults() {
    assert( searchTask );
    assert( sa );

    if( sa->error.data ) {
        stateInfo.setError(sa->error.data);
    }
    pair_array_t currentBestPairs = searchTask->getBestPairs();
    int pairsNum = currentBestPairs.num_pairs;
    if( pairsNum > 0 ) {
        int print_lib_sim = (pa->repeat_lib.repeat_file || pa->io_mishyb_library.repeat_file);
        
        for (int i = 0; i < pairsNum; i++) {    
            QList<SharedAnnotationData> annotations;
            annotations << oligoToAnnotation("primer", currentBestPairs.pairs[i].left, false, pa->repeat_lib, print_lib_sim);
            annotations << oligoToAnnotation("primer", currentBestPairs.pairs[i].right, true, pa->repeat_lib, print_lib_sim);
            //oligoToAnnotation("primer", currentBestPairs.pairs[i].intl, false, pa->io_mishyb_library, print_lib_sim);
            CreateAnnotationsTask * t = new CreateAnnotationsTask(aobj, groupName+"/pair "+QString::number(i+1), annotations);    
            AppContext::getTaskScheduler()->registerTopLevelTask(t);              
        }  
    }

}

SharedAnnotationData Primer3ToAnnotationsTask::oligoToAnnotation(const char *title, const primer_rec *o, bool complement, 
                                                                 const seq_lib seqlib, int print_lib_sim)
{    
    SharedAnnotationData sd (new AnnotationData());        
    sd->name=QString::fromAscii(title);
    int start = o->start + sa->incl_s + pa->first_base_index;
    int len = o->length;
    if (complement == true)
        start = start - len + 1;    
    sd->complement = complement; 
    sd->location.append(LRegion(start - 1, len));       
    sd->qualifiers.append(Qualifier("tm",QString::number(o->temp)));    
    sd->qualifiers.append(Qualifier("gc%",QString::number(o->gc_content)));    
    sd->qualifiers.append(Qualifier("any",QString::number(0.01 * o->self_any)));    
    sd->qualifiers.append(Qualifier("3'",QString::number(0.01 * o->self_end)));       
    return sd;   
}

}