/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "Primer3DialogController.h"
#include "Primer3Task.h"
#include <util_gui/DialogUtils.h>
#include <core_api/AppContext.h>
#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVConstants.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVUtils.h>
#include <core_api/LRegion.h>

#include <gobjects/DNASequenceObject.h>
#include <selection/DNASequenceSelection.h>
#include <util_gui/CreateAnnotationWidgetController.h>

#include <document_format/GenbankFeatures.h>
#include <util_tasks/CreateAnnotationTask.h>
#include <QtGui/QMessageBox>

namespace GB2 {

Primer3DialogController::Primer3DialogController(ADVSequenceObjectContext* _ctx) 
: QDialog(_ctx->getAnnotatedDNAView()->getWidget()), ctx(_ctx)
{
    setupUi(this);                    
    //removing unnecessary widget. It duplicates sbRangeStart and sbRangeEnd.
    edit_INCLUDED_REGION->setVisible(false);
    label_INCLUDED_REGION->setVisible(false);
    
    seqLen = ctx->getSequenceLen();
    initialSelection = ctx->getSequenceSelection()->isEmpty() ? LRegion() : ctx->getSequenceSelection()->getSelectedRegions().first();

    sbRangeStart->setMinimum(1);
    sbRangeStart->setMaximum(seqLen);

    sbRangeEnd->setMinimum(1);
    sbRangeEnd->setMaximum(seqLen);          

    connect(AppContext::getTaskScheduler(), SIGNAL(si_stateChanged(Task*)), SLOT(sl_onTaskFinished(Task*)));    
    addAnnotationWidget();
    connectGUI();    
    sl_pbResetClicked();   
    sl_pbSelectionRange();
}

Primer3DialogController::~Primer3DialogController() {
}

void Primer3DialogController::connectGUI()
{
    connect(pbPick, SIGNAL(clicked()), this, SLOT(sl_pbPickClicked()));
    connect(pbReset, SIGNAL(clicked()), this, SLOT(sl_pbResetClicked()));    
    connect(pbSelectionRange, SIGNAL(clicked()), this, SLOT(sl_pbSelectionRange()));
    connect(pbSequenceRange, SIGNAL(clicked()), this, SLOT(sl_pbSequenceRange()));
    connect(sbRangeStart, SIGNAL(valueChanged(int)), this, SLOT(sl_sbRangeStartChanged(int)));
    connect(sbRangeEnd, SIGNAL(valueChanged(int)), this, SLOT(sl_sbRangeEndChanged(int)));
}

void Primer3DialogController::addAnnotationWidget() {   
    acm.data->name = "top primers";
    acm.sequenceObjectRef = GObjectReference(ctx->getSequenceGObject());
    acm.hideAnnotationName = false;
    acm.hideLocation = true;
    ac = new CreateAnnotationWidgetController(acm, this);
    QWidget* caw = ac->getWidget();    
    QVBoxLayout* l = new QVBoxLayout();
    l->setMargin(0);
    l->addWidget(caw);
    annotationWidget->setLayout(l);
    annotationWidget->setMinimumSize(caw->layout()->minimumSize());
}

void Primer3DialogController::sl_sbRangeStartChanged( int val )
{
    if (val > sbRangeEnd->value()) {
        sbRangeEnd->setValue(val);
    }

}

void Primer3DialogController::sl_sbRangeEndChanged( int val )
{
    if (val < sbRangeStart->value()) {
        sbRangeStart->setValue(val);
    }
}

void Primer3DialogController::sl_pbSelectionRange()
{   
    sbRangeStart->setValue(initialSelection.isEmpty() ? 1 : initialSelection.startPos + 1);
    sbRangeEnd->setValue(initialSelection.isEmpty() ? seqLen : initialSelection.endPos());
}

void Primer3DialogController::sl_pbSequenceRange()
{   
    sbRangeStart->setValue(1);
    sbRangeEnd->setValue(seqLen);
}

void Primer3DialogController::sl_pbPickClicked(){                        
    if (readForm()) {
        ac->prepareAnnotationObject();
        const CreateAnnotationModel& m = ac->getModel();      

        Primer3ToAnnotationsTask * primer3Task = new Primer3ToAnnotationsTask( pa,sa, m.getAnnotationObject(), m.groupName, "" );
        AppContext::getTaskScheduler()->registerTopLevelTask(primer3Task);  
        accept();
    } else {    
        free(sa);
        free(pa);
    }
}

void Primer3DialogController::sl_pbResetClicked(){  
    //Main
    edit_EXCLUDED_REGION->clear();
    edit_TARGET->clear();
    edit_INCLUDED_REGION->clear();
    edit_PRIMER_START_CODON_POSITION_EDIT->clear();
    combobox_PRIMER_MISPRIMING_LIBRARY->setCurrentIndex(0);
    edit_PRIMER_PRODUCT_SIZE_RANGE->setText("150-250 100-300 301-400 401-500 501-600 601-700 701-850 851-1000");
    edit_PRIMER_NUM_RETURN->setText("5");//pa->num_return = NUM_RETURN;
    edit_PRIMER_MAX_END_STABILITY->setText("9.0");//a->max_end_stability = DEFAULT_MAX_END_STABILITY;
    edit_PRIMER_MAX_MISPRIMING->setText("12.0");//pa->repeat_compl = REPEAT_SIMILARITY;
    edit_PRIMER_PAIR_MAX_MISPRIMING->setText("24.0");//pa->pair_repeat_compl = PAIR_REPEAT_SIMILARITY;        
    edit_PRIMER_MAX_TEMPLATE_MISPRIMING->setText("12.0");//a->max_template_mispriming = MAX_TEMPLATE_MISPRIMING;
    edit_PRIMER_PAIR_MAX_TEMPLATE_MISPRIMING->setText("24.0");//a->pair_max_template_mispriming = PAIR_MAX_TEMPLATE_MISPRIMING;
    checkbox_PICK_LEFT->setChecked(true);
    checkbox_PICK_HYBRO->setChecked(false);
    checkbox_PICK_RIGHT->setChecked(true);
    edit_PICK_LEFT->setEnabled(false);//true
    edit_PICK_HYBRO->setEnabled(false);
    edit_PICK_RIGHT->setEnabled(false);//true
    edit_PICK_LEFT->clear();
    edit_PICK_HYBRO->clear();
    edit_PICK_RIGHT->clear();

    //Tab
    edit_PRIMER_OPT_SIZE->setText("20");//a->primer_opt_size  = OPT_SIZE;
    edit_PRIMER_MIN_SIZE->setText("18");//a->primer_min_size  = MIN_SIZE;
    edit_PRIMER_MAX_SIZE->setText("27");//a->primer_max_size  = MAX_SIZE;
    edit_PRIMER_OPT_TM->setText("60.0");//a->opt_tm           = OPT_TM;
    edit_PRIMER_MIN_TM->setText("57.0");//a->min_tm           = MIN_TM;
    edit_PRIMER_MAX_TM->setText("63.0");//a->max_tm           = MAX_TM;
    edit_PRIMER_PRODUCT_OPT_TM->clear();//a->product_opt_tm    = PRIMER_PRODUCT_OPT_TM;           
    edit_PRIMER_PRODUCT_MIN_TM->clear();//a->product_min_tm    = PR_DEFAULT_PRODUCT_MIN_TM;    
    edit_PRIMER_PRODUCT_MAX_TM->clear();//a->product_max_tm    = PR_DEFAULT_PRODUCT_MAX_TM;
    edit_PRIMER_MIN_GC_PERCENT->setText("20.0");//a->min_gc           = MIN_GC;
    edit_PRIMER_OPT_GC_PERCENT->clear();//a->opt_gc_content   = DEFAULT_OPT_GC_PERCENT;
    edit_PRIMER_MAX_GC_PERCENT->setText("80.0");//a->max_gc           = MAX_GC;
    edit_PRIMER_MAX_DIFF_TM->setText("100.0");//a->max_diff_tm      = MAX_DIFF_TM;
    combobox_PRIMER_TM_SANTALUCIA->setCurrentIndex(0);//a->tm_santalucia    = TM_SANTALUCIA; /* added by T.Koressaar */       
    edit_PRIMER_SELF_ANY->setText("8.0");//a->self_any         = SELF_ANY;
    edit_PRIMER_SELF_END->setText("3.0");//a->self_end         = SELF_END;
    edit_PRIMER_NUM_NS_ACCEPTED->setText("0");//a->num_ns_accepted  = NUM_NS_ACCEPTED;
    edit_PRIMER_MAX_POLY_X->setText("5");//a->max_poly_x       = MAX_POLY_X;    
    edit_PRIMER_INSIDE_PENALTY->clear();//a->inside_penalty    = PR_DEFAULT_INSIDE_PENALTY;
    edit_PRIMER_OUTSIDE_PENALTY->setText("0");//a->outside_penalty   = PR_DEFAULT_OUTSIDE_PENALTY;    
    edit_PRIMER_FIRST_BASE_INDEX->setText("1");//a->first_base_index  = FIRST_BASE_INDEX;
    edit_PRIMER_GC_CLAMP->setText("0");//a->gc_clamp         = GC_CLAMP;
    edit_PRIMER_SALT_CONC->setText("50.0");//a->salt_conc        = SALT_CONC;
    combobox_PRIMER_SALT_CORRECTIONS->setCurrentIndex(0);//a->salt_corrections = SALT_CORRECTIONS; /* added by T.Koressaar */
    edit_PRIMER_DIVALENT_CONC->setText("0.0");//a->divalent_conc    = DIVALENT_CONC;
    edit_PRIMER_DNTP_CONC->setText("0.0");//a->dntp_conc        = DNTP_CONC;
    edit_PRIMER_DNA_CONC->setText("50.0");//a->dna_conc         = DNA_CONC; 
    checbox_PRIMER_LIBERAL_BASE->setChecked(true);
    checkbox_SHOW_DEBUGGING->setChecked(false);
    checkbox_PRIMER_LIB_AMBIGUITY_CODES_CONSENSUS->setChecked(true);
    checkbox_PRIMER_LOWERCASE_MASKING->setChecked(false);
        
    //a->pair_compl_any   = PAIR_COMPL_ANY;
    //a->pair_compl_end   = PAIR_COMPL_END;
    //a->file_flag        = FILE_FLAG;
    //a->explain_flag     = EXPLAIN_FLAG;        
    //a->liberal_base      = LIBERAL_BASE;
    //a->primer_task       = PRIMER_TASK;        
    //a->pr_min[0]         = 100;
    //a->pr_max[0]         = 300;
    //a->num_intervals     = 1;    
    //a->min_quality       = MIN_QUALITY;
    //a->min_end_quality   = MIN_QUALITY;
    //a->quality_range_min = QUALITY_RANGE_MIN;
    //a->quality_range_max = QUALITY_RANGE_MAX;        
    //a->lowercase_masking = LOWERCASE_MASKING; /* added by T.Koressaar */    
    //a->product_opt_size  = PRIMER_PRODUCT_OPT_SIZE;    
    //a->io_primer_opt_size = INTERNAL_OLIGO_OPT_SIZE;
    //a->io_primer_min_size = INTERNAL_OLIGO_MIN_SIZE;
    //a->io_primer_max_size = INTERNAL_OLIGO_MAX_SIZE;
    //a->io_opt_tm          = INTERNAL_OLIGO_OPT_TM;
    //a->io_min_tm          = INTERNAL_OLIGO_MIN_TM;
    //a->io_max_tm          = INTERNAL_OLIGO_MAX_TM;
    //a->io_min_gc          = INTERNAL_OLIGO_MIN_GC;
    //a->io_opt_gc_content  = DEFAULT_OPT_GC_PERCENT;
    //a->io_max_gc          = INTERNAL_OLIGO_MAX_GC;
    //a->io_max_poly_x      = INTERNAL_OLIGO_MAX_POLY_X;
    //a->io_salt_conc       = INTERNAL_OLIGO_SALT_CONC;
    //a->io_divalent_conc   = INTERNAL_OLIGO_DIVALENT_CONC;
    //a->io_dntp_conc       = INTERNAL_OLIGO_DNTP_CONC;
    //a->io_dna_conc        = INTERNAL_OLIGO_DNA_CONC;
    //a->io_num_ns_accepted = INTERNAL_OLIGO_NUM_NS;
    //a->io_self_any        = INTERNAL_OLIGO_SELF_ANY;
    //a->io_self_end        = INTERNAL_OLIGO_SELF_END;
    //a->io_repeat_compl    = INTERNAL_OLIGO_REPEAT_SIMILARITY;
    //a->io_min_quality     = MIN_QUALITY;
    //a->io_min_end_quality = MIN_QUALITY;
    //a->io_max_template_mishyb
    //                      = IO_MAX_TEMPLATE_MISHYB;

    //a->primer_weights.temp_gt       = PRIMER_WT_TM_GT;
    //a->primer_weights.temp_lt       = PRIMER_WT_TM_LT;
    //a->primer_weights.length_gt     = PRIMER_WT_SIZE_GT;
    //a->primer_weights.length_lt     = PRIMER_WT_SIZE_LT;
    //a->primer_weights.gc_content_gt = PRIMER_WT_GC_PERCENT_GT;
    //a->primer_weights.gc_content_lt = PRIMER_WT_GC_PERCENT_LT;
    //a->primer_weights.compl_any     = PRIMER_WT_COMPL_ANY;
    //a->primer_weights.compl_end     = PRIMER_WT_COMPL_END;
    //a->primer_weights.num_ns        = PRIMER_WT_NUM_NS;
    //a->primer_weights.repeat_sim    = PRIMER_WT_REP_SIM;
    //a->primer_weights.seq_quality   = PRIMER_WT_SEQ_QUAL;
    //a->primer_weights.end_quality   = PRIMER_WT_END_QUAL;
    //a->primer_weights.pos_penalty   = PRIMER_WT_POS_PENALTY;
    //a->primer_weights.end_stability = PRIMER_WT_END_STABILITY;

    //a->io_weights.temp_gt     = IO_WT_TM_GT;
    //a->io_weights.temp_lt     = IO_WT_TM_LT;
    //a->io_weights.length_gt   = IO_WT_SIZE_GT;
    //a->io_weights.length_lt   = IO_WT_SIZE_LT;
    //a->io_weights.gc_content_gt = IO_WT_GC_PERCENT_GT;
    //a->io_weights.gc_content_lt = IO_WT_GC_PERCENT_LT;
    //a->io_weights.compl_any   = IO_WT_COMPL_ANY;
    //a->io_weights.compl_end   = IO_WT_COMPL_END;
    //a->io_weights.num_ns      = IO_WT_NUM_NS;
    //a->io_weights.repeat_sim  = IO_WT_REP_SIM;
    //a->io_weights.seq_quality = IO_WT_SEQ_QUAL;
    //a->io_weights.end_quality = IO_WT_END_QUAL;

    //a->pr_pair_weights.primer_quality  = PAIR_WT_PRIMER_PENALTY;
    //a->pr_pair_weights.io_quality      = PAIR_WT_IO_PENALTY;
    //a->pr_pair_weights.diff_tm         = PAIR_WT_DIFF_TM;
    //a->pr_pair_weights.compl_any       = PAIR_WT_COMPL_ANY;
    //a->pr_pair_weights.compl_end       = PAIR_WT_COMPL_END;
    //a->pr_pair_weights.repeat_sim      = PAIR_WT_REP_SIM;
    //a->pr_pair_weights.product_tm_lt   = PAIR_WT_PRODUCT_TM_LT;
    //a->pr_pair_weights.product_tm_gt   = PAIR_WT_PRODUCT_TM_GT;
    //a->pr_pair_weights.product_size_lt = PAIR_WT_PRODUCT_SIZE_LT;
    //a->pr_pair_weights.product_size_gt = PAIR_WT_PRODUCT_SIZE_GT;

    ///* a->short_match = 0; not used */

    //a->lib_ambiguity_codes_consensus   = LIB_AMBIGUITY_CODES_CONSENSUS;
}

bool Primer3DialogController::readForm(){ 
    QLineEdit *in;
    char *datum;

    pa = (primer_args *)pr_safe_malloc(sizeof(*pa));
    sa = (seq_args *)pr_safe_malloc(sizeof(*sa));  

    pr_set_default_global_args(pa);

    memset(&sa->error, 0, sizeof(sa->error));
    memset(&pa->glob_err, 0, sizeof(pa->glob_err));
    memset(sa, 0, sizeof(*sa)); 

    sa->sequence = QStringToChar(ctx->getSequenceData().mid(sbRangeStart->value()-1, sbRangeEnd->value() - sbRangeStart->value() + 1));

    sa->start_codon_pos = PR_DEFAULT_START_CODON_POS;
    sa->incl_l = -1; /* Indicates logical NULL. */

    //Main   
    in = edit_EXCLUDED_REGION;
    if (!in->text().isEmpty()) {
        datum = QStringToChar(in->text());
        parse_interval_list("EXCLUDED_REGION", datum, &sa->num_excl, sa->excl, &sa->error);//edit_EXCLUDED_REGION->clear();
        free(datum);
        if (sa->error.data != NULL) {
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " is incorrect")));            
            return false;
        }
    }
    in = edit_TARGET;
    if (!in->text().isEmpty()) {
        datum = QStringToChar(in->text());
        parse_interval_list("TARGET", datum, &sa->num_targets, sa->tar, &sa->error);//edit_TARGET->clear();        
        free(datum);
        if (sa->error.data != NULL) {
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " is incorrect")));            
            return false;
        }
    }
    in = edit_INCLUDED_REGION;
    if (!in->text().isEmpty()) {
        datum = QStringToChar(in->text());
        const char *p = parse_int_pair("INCLUDED_REGION", datum, ',', &sa->incl_s, &sa->incl_l, &sa->error); //edit_INCLUDED_REGION->clear();            
        if (NULL != p) {
            while (' ' == *p || '\t' == *p) p++;
            if (*p != '\n' && *p != '\0')
                tag_syntax_error("INCLUDED_REGION", datum, &sa->error);                    
        }        
        free(datum);
        if (sa->error.data != NULL) {
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " is incorrect")));            
            return false;
        }
    }        
    //combobox_PRIMER_MISPRIMING_LIBRARY->setCurrentIndex(0);
    in = edit_PRIMER_PRODUCT_SIZE_RANGE;
    if (!in->text().isEmpty()) {
        datum = QStringToChar(in->text());
        parse_product_size("PRIMER_PRODUCT_SIZE_RANGE", datum, pa, &pa->glob_err);//edit_PRIMER_PRODUCT_SIZE_RANGE->setText("150-250 100-300 301-400 401-500 501-600 601-700 701-850 851-1000");    
        free(datum);
        if (pa->glob_err.data != NULL) {            
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " is incorrect")));            
            return false;
        }
    }

    if (!QStringToInt(edit_PRIMER_START_CODON_POSITION_EDIT, &sa->start_codon_pos)) return false; //edit_RIMER_START_CODON_POSITION_EDIT->clear();
    if (!QStringToInt(edit_PRIMER_NUM_RETURN, &pa->num_return)) return false; //edit_PRIMER_NUM_RETURN->setText("5");//pa->num_return = NUM_RETURN;        
    if (!QStringToDouble(edit_PRIMER_MAX_END_STABILITY, &pa->max_end_stability)) return false; //edit_PRIMER_MAX_END_STABILITY->setText("9.0");//a->max_end_stability = DEFAULT_MAX_END_STABILITY;
    if (!QStringToShortAlign(edit_PRIMER_MAX_MISPRIMING, &pa->repeat_compl)) return false; //edit_PRIMER_MAX_MISPRIMING->setText("12.00");//pa->repeat_compl = REPEAT_SIMILARITY;
    if (!QStringToShortAlign(edit_PRIMER_PAIR_MAX_MISPRIMING, &pa->pair_repeat_compl)) return false; //edit_PRIMER_PAIR_MAX_MISPRIMING->setText("24.00");//pa->pair_repeat_compl = PAIR_REPEAT_SIMILARITY;        
    if (!QStringToShortAlign(edit_PRIMER_MAX_TEMPLATE_MISPRIMING, &pa->max_template_mispriming)) return false; //edit_PRIMER_MAX_TEMPLATE_MISPRIMING->setText("12.00");//a->max_template_mispriming = MAX_TEMPLATE_MISPRIMING;
    if (!QStringToShortAlign(edit_PRIMER_PAIR_MAX_TEMPLATE_MISPRIMING, &pa->pair_max_template_mispriming)) return false; //edit_PRIMER_PAIR_MAX_TEMPLATE_MISPRIMING->setText("24.00");//a->pair_max_template_mispriming = PAIR_MAX_TEMPLATE_MISPRIMING;
    //checkbox_PICK_LEFT->setChecked(true);
    //checkbox_PICK_HYBRO->setChecked(false);
    //checkbox_PICK_RIGHT->setChecked(true);
    //edit_PICK_LEFT->setEnabled(true);
    //edit_PICK_HYBRO->setEnabled(false);
    //edit_PICK_RIGHT->setEnabled(true);
    //edit_PICK_LEFT->clear();
    //edit_PICK_HYBRO->clear();
    //edit_PICK_RIGHT->clear();
    int sum = (int)checkbox_PICK_LEFT->isChecked() + (int)checkbox_PICK_HYBRO->isChecked() + (int)checkbox_PICK_RIGHT->isChecked();
    if (sum == 3) pa->primer_task = pick_pcr_primers_and_hyb_probe;
    if ((sum == 2) && checkbox_PICK_HYBRO->isChecked()) {
        //if (checkbox_PICK_LEFT->isChecked()); //WARNING: Assuming you want to pick a right primer because you are picking a left primer and internal oligo
        ///if (checkbox_PICK_RIGHT->isChecked()); //WARNING: Assuming you want to pick a right primer because you are picking a left primer and internal oligo
        pa->primer_task = pick_pcr_primers_and_hyb_probe;
    }
    if ((sum == 2) && !checkbox_PICK_HYBRO->isChecked()) pa->primer_task = pick_pcr_primers;
    if (sum == 1) {
        if (checkbox_PICK_LEFT->isChecked()) pa->primer_task = pick_left_only;
        if (checkbox_PICK_HYBRO->isChecked()) pa->primer_task = pick_right_only;
        if (checkbox_PICK_RIGHT->isChecked()) pa->primer_task = pick_right_only;       
    }
    if (sum == 0) {
        //WARNING: assuming you want to pick PCR primers
        pa->primer_task = pick_pcr_primers;
    }
    pa->primer_task = pick_pcr_primers;

    ////Tab
    if (!QStringToInt(edit_PRIMER_OPT_SIZE, &pa->primer_opt_size)) return false; //edit_PRIMER_OPT_SIZE->setText("20");//a->primer_opt_size  = OPT_SIZE;
    if (!QStringToInt(edit_PRIMER_MIN_SIZE, &pa->primer_min_size)) return false; //edit_PRIMER_MIN_SIZE->setText("18");//a->primer_min_size  = MIN_SIZE;
    if (!QStringToInt(edit_PRIMER_MAX_SIZE, &pa->primer_max_size)) return false; //edit_PRIMER_MAX_SIZE->setText("27");//a->primer_max_size  = MAX_SIZE;
    if (!QStringToDouble(edit_PRIMER_OPT_TM, &pa->opt_tm)) return false; //edit_PRIMER_OPT_TM->setText("60.0");//a->opt_tm           = OPT_TM;
    if (!QStringToDouble(edit_PRIMER_MIN_TM, &pa->min_tm)) return false; //edit_PRIMER_MIN_TM->setText("57.0");//a->min_tm           = MIN_TM;
    if (!QStringToDouble(edit_PRIMER_MAX_TM, &pa->max_tm)) return false; //edit_PRIMER_MAX_TM->setText("63.0");//a->max_tm           = MAX_TM;
    if (!QStringToDouble(edit_PRIMER_PRODUCT_OPT_TM, &pa->product_opt_tm)) return false; //edit_PRIMER_PRODUCT_OPT_TM->clear();//a->product_opt_tm    = PRIMER_PRODUCT_OPT_TM;           
    if (!QStringToDouble(edit_PRIMER_PRODUCT_MIN_TM, &pa->product_min_tm)) return false; //edit_PRIMER_PRODUCT_MIN_TM->clear();//a->product_min_tm    = PR_DEFAULT_PRODUCT_MIN_TM;    
    if (!QStringToDouble(edit_PRIMER_PRODUCT_MAX_TM, &pa->product_max_tm)) return false; //edit_PRIMER_PRODUCT_MAX_TM->clear();//a->product_max_tm    = PR_DEFAULT_PRODUCT_MAX_TM;
    if (!QStringToDouble(edit_PRIMER_MIN_GC_PERCENT, &pa->min_gc)) return false; //edit_PRIMER_MIN_GC_PERCENT->setText("20.0");//a->min_gc           = MIN_GC;
    if (!QStringToDouble(edit_PRIMER_OPT_GC_PERCENT, &pa->opt_gc_content)) return false; //edit_PRIMER_OPT_GC_PERCENT->clear();//a->opt_gc_content   = DEFAULT_OPT_GC_PERCENT;
    if (!QStringToDouble(edit_PRIMER_MAX_GC_PERCENT, &pa->max_gc)) return false; //edit_PRIMER_MAX_GC_PERCENT->setText("20.0");//a->max_gc           = MAX_GC;
    if (!QStringToDouble(edit_PRIMER_MAX_DIFF_TM, &pa->max_diff_tm)) return false; //edit_PRIMER_MAX_DIFF_TM->setText("100.0");//a->max_diff_tm      = MAX_DIFF_TM;
    pa->tm_santalucia = combobox_PRIMER_TM_SANTALUCIA->currentIndex();//combobox_PRIMER_TM_SANTALUCIA->setCurrentIndex(0);//a->tm_santalucia    = TM_SANTALUCIA; /* added by T.Koressaar */       
    if (!QStringToShortAlign(edit_PRIMER_SELF_ANY, &pa->self_any)) return false; //edit_PRIMER_SELF_ANY->setText("8.0");//a->self_any         = SELF_ANY;
    if (!QStringToShortAlign(edit_PRIMER_SELF_END, &pa->self_end)) return false; //edit_PRIMER_SELF_END->setText("3.0");//a->self_end         = SELF_END;
    if (!QStringToInt(edit_PRIMER_NUM_NS_ACCEPTED, &pa->num_ns_accepted)) return false; //edit_PRIMER_NUM_NS_ACCEPTED->setText("0");//a->num_ns_accepted  = NUM_NS_ACCEPTED;
    if (!QStringToInt(edit_PRIMER_MAX_POLY_X, &pa->max_poly_x)) return false; //edit_PRIMER_MAX_POLY_X->setText("5");//a->max_poly_x       = MAX_POLY_X;    
    if (!QStringToDouble(edit_PRIMER_INSIDE_PENALTY, &pa->inside_penalty)) return false; //edit_PRIMER_INSIDE_PENALTY->clear();//a->inside_penalty    = PR_DEFAULT_INSIDE_PENALTY;
    if (!QStringToDouble(edit_PRIMER_OUTSIDE_PENALTY, &pa->outside_penalty)) return false; //edit_PRIMER_OUTSIDE_PENALTY->setText("0");//a->outside_penalty   = PR_DEFAULT_OUTSIDE_PENALTY;    
    if (!QStringToInt(edit_PRIMER_FIRST_BASE_INDEX, &pa->first_base_index)) return false; //edit_PRIMER_FIRST_BASE_INDEX->setText("1");//a->first_base_index  = FIRST_BASE_INDEX;
    if (!QStringToInt(edit_PRIMER_GC_CLAMP, &pa->gc_clamp)) return false; //edit_PRIMER_GC_CLAMP->setText("0");//a->gc_clamp         = GC_CLAMP;
    if (!QStringToDouble(edit_PRIMER_SALT_CONC, &pa->salt_conc)) return false; //edit_PRIMER_SALT_CONC->setText("50.0");//a->salt_conc        = SALT_CONC;
    pa->salt_corrections = combobox_PRIMER_SALT_CORRECTIONS->currentIndex();//combobox_PRIMER_SALT_CORRECTIONS->setCurrentIndex(0);//a->salt_corrections = SALT_CORRECTIONS; /* added by T.Koressaar */
    if (!QStringToDouble(edit_PRIMER_DIVALENT_CONC, &pa->divalent_conc)) return false; //edit_PRIMER_DIVALENT_CONC->setText("0.0");//a->divalent_conc    = DIVALENT_CONC;
    if (!QStringToDouble(edit_PRIMER_DNTP_CONC, &pa->dntp_conc)) return false; //edit_PRIMER_DNTP_CONC->setText("0.0");//a->dntp_conc        = DNTP_CONC;
    if (!QStringToDouble(edit_PRIMER_DNA_CONC, &pa->dna_conc)) return false; //edit_PRIMER_DNA_CONC->setText("50.0");//a->dna_conc         = DNA_CONC; 
    pa->liberal_base = (int)checbox_PRIMER_LIBERAL_BASE->isChecked();//checbox_PRIMER_LIBERAL_BASE->setChecked(true);
    //checkbox_SHOW_DEBUGGING->setChecked(false);
    pa->lib_ambiguity_codes_consensus = (int)checkbox_PRIMER_LIB_AMBIGUITY_CODES_CONSENSUS->isChecked();//checkbox_PRIMER_LIB_AMBIGUITY_CODES_CONSENSUS->setChecked(true);
    pa->lowercase_masking = (int) checkbox_PRIMER_LOWERCASE_MASKING->isChecked();//checkbox_PRIMER_LOWERCASE_MASKING->setChecked(false);

    if (sa->incl_l == -1) {
        sa->incl_l = strlen(sa->sequence);
        sa->incl_s = pa->first_base_index;
    }

    if(pa->min_quality != 0 && pa->min_end_quality < pa->min_quality)
        pa->min_end_quality = pa->min_quality;

    /* Adjust base indexes in sa. */
    sa->incl_s -= pa->first_base_index;
    sa->start_codon_pos -= pa->first_base_index;
    adjust_base_index_interval_list(sa->tar, sa->num_targets,
        pa->first_base_index);
    adjust_base_index_interval_list(sa->excl, sa->num_excl,
        pa->first_base_index);
    adjust_base_index_interval_list(sa->excl_internal,
        sa->num_internal_excl,
        pa->first_base_index);

    return true;
}

void Primer3DialogController::reject() {
    QDialog::reject();
}

char *Primer3DialogController::QStringToChar(QString in) {
    QByteArray ba = in.toAscii(); 
    char *out = (char *)pr_safe_malloc(ba.size() + 1);
    strcpy(out, ba.data());    
    return out;
}

bool Primer3DialogController::QStringToChar(QLineEdit *in, char *out) {
    QString text = in->text();
    if (!text.isEmpty()) {
        QByteArray ba = text.toAscii(); 
        out = (char *)pr_safe_malloc(ba.size() + 1);
        strcpy(out, ba.data());         
    }
    return true;
}

bool Primer3DialogController::QStringToInt(QLineEdit *in, int *out) {   
    QString text = in->text();
    if (!text.isEmpty()) {  
        bool ok;
        int res = text.toInt(&ok);
        if (ok) {
            *out = res;  
        } else {            
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " must be integer")));            
            return false;
        }
    }
    return true;
}

bool Primer3DialogController::QStringToDouble(QLineEdit *in, double *out) {   
    QString text = in->text();
    if (!text.isEmpty()) {  
        bool ok;
        double res = text.toDouble(&ok);
        if (ok) {
            *out = res;
        } else {
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " must be double")));            
            return false;
        }
    }
    return true;
}

bool Primer3DialogController::QStringToShortAlign(QLineEdit *in, short *out) {   
    QString text = in->text();
    if (!text.isEmpty()) { 
        bool ok;
        double res = (text.toDouble(&ok)*100);
        if ((!ok) || (res > SHRT_MAX)) {
            QMessageBox::critical(this, windowTitle(),  tr(QStringToChar("The field " + in->objectName().mid(5) + " * 100 must be short")));            
            return false;
        } else {            
            *out = (short)res;
        }        
    }
    return true;
}

}
