/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "BaseDocWorker.h"

#include "CoreLib.h"
#include <workflow/WorkflowEnv.h>
#include <workflow_support/WorkflowUtils.h>
#include <workflow_library/BioActorLibrary.h>
#include <util_tasks/LoadDocumentTask.h>
#include <util_tasks/SaveDocumentTask.h>
#include <util_tasks/FailTask.h>
#include <util_tasks/MultiTask.h>

#include <core_api/AppContext.h>
#include <core_api/ProjectModel.h>
#include <core_api/Log.h>
#include <core_api/DocumentModel.h>
#include <core_api/IOAdapter.h>
#include <util_gui/DialogUtils.h>

namespace GB2 {
namespace LocalWorkflow {

static LogCategory log(ULOG_CAT_WD);

LocalDocReader::LocalDocReader(Actor* a, const QString& tid, const DocumentFormatId& fid) : BaseWorker(a), ch(NULL), fid(fid), 
done(false), attachDoc2Proj(false) {
    mtype = WorkflowEnv::getDataTypeRegistry()->getById(tid);
}

void LocalDocReader::init() {
    QStringList urls = DesignerUtils::expandToUrls(actor->getParameter("URL")->value.toString());
    Project* p = AppContext::getProject();
    foreach(QString url, urls) {
        Document* doc = NULL;
        bool newDoc = true;
        if (p) {
            doc = p->findDocumentByURL(url);
            if (doc && doc->getDocumentFormatId() == fid) {
                newDoc = false;
            } else {
                doc = NULL;
            }
        }
        if (!doc) {
            DocumentFormat* format = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
            assert(format);
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
            doc = new Document(format, iof, url);
        }
        //TODO lock document???
        docs.insert(doc, newDoc);
    }

    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool LocalDocReader::isReady() {
    return !isDone();
}

Task* LocalDocReader::tick() {
    if (!docs.isEmpty()) {
        Document* doc = docs.begin().key();
        if (!doc->isLoaded()) {
            return new LoadUnloadedDocumentTask(doc);
        } else {
            doc2data(doc);
            while (!cache.isEmpty()) {
                ch->put(cache.takeFirst());
            }
            if (docs.take(doc)) {
                doc->unload();
                delete doc;
            }
        }
    } 
    if (docs.isEmpty()) {
        done = true;
        ch->setEnded();
    }
    return NULL;
}

bool LocalDocReader::isDone() {
    return done && cache.isEmpty();
}

void LocalDocReader::cleanup() {
    QMapIterator<Document*, bool> it(docs);
    while (it.hasNext())
    {
        it.next();
        if (it.value()) {
            if (it.key()->isLoaded()) {
                it.key()->unload();
            }
            delete it.key();
        }
    }
}

LocalDocWriter::LocalDocWriter(Actor* a, const DocumentFormatId& fid) : BaseWorker(a), ch(NULL),
done(false), append(false), fileMode(SaveDoc_Roll) {
    format = AppContext::getDocumentFormatRegistry()->getFormatById(fid);
}


bool LocalDocWriter::isDone() {
    return done;
}

void LocalDocWriter::cleanup() {
}

void LocalDocWriter::init() {
    url = actor->getParameter(CoreLib::URL_ATTR_ID)->value.toString();
    fileMode = actor->getParameter(BioActorLibrary::FILE_MODE_ATTR_ID)->value.toUInt();
    fileMode |= SaveDoc_DestroyAfter;
    Attribute* a = actor->getParameter(CoreLib::APPEND_ATTR_ID);
    if (a) {
        append = a->value.toBool();
    }
    assert(ports.size() == 1);
    ch = ports.values().first();
}

bool LocalDocWriter::isReady() {
    return ch->hasMessage() || ch->isEnded() && !done;
}

Task* LocalDocWriter::tick() {
    while (ch->hasMessage()) {
        Document* doc = NULL;
        QString anUrl = url;
        QVariantMap data = ch->get().getData().toMap();
        if (anUrl.isEmpty()) {
            anUrl = data.value(CoreLib::URL_SLOT_ID).toString();
        }
        if (anUrl.isEmpty()) {
            QString err = tr("Unspecified URL to write %1").arg(format->getFormatName());
            if (failFast) {
                return new FailTask(err);
            } else {
                log.error(err);
                return NULL;
            }
        }
        doc = docs.value(anUrl);
        if (!doc) {
            IOAdapterFactory* iof = AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById(BaseIOAdapters::url2io(url));
            doc = new Document(format, iof, anUrl);
            doc->setLoaded(true);
            docs.insert(anUrl, doc);
        }
        data2doc(doc, data);
        if (!append) {
            break;
        }
    }
    done = ch->isEnded();
    if (append && !done) {
        return NULL;
    }
    return processDocs();
}

Task* LocalDocWriter::processDocs()
{
    assert(!docs.isEmpty());
    QList<Task*> tlist;
    QMapIterator<QString, Document*> it(docs);
    while (it.hasNext())
    {
        it.next();
        Document* doc = it.value();
        QString anUrl = it.key();

        int count = ++counter[anUrl];
        if (!append && count != 1) {
            anUrl = DialogUtils::prepareFileName(anUrl, count, format->getSupportedDocumentFileExtensions());
        } else {
            assert(count == 1);
            anUrl = DialogUtils::ensureFileExt(anUrl, format->getSupportedDocumentFileExtensions());
        }
        doc->setURL(anUrl);
        log.info(tr("Writing to %1 [%2]").arg(anUrl).arg(format->getFormatName()));
        tlist << new SaveDocumentTask(doc, fileMode);
    }
    docs.clear();

    return tlist.size() == 1 ? tlist.first() : new MultiTask(tr("Save documents"), tlist);
}

} // Workflow namespace
} // GB2 namespace
