/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "TestRunnerPlugin.h"
#include "TestViewController.h"
#include "GTestScriptWrapper.h"

#include <core_api/MainWindow.h>
#include <core_api/Log.h>
#include <core_api/Settings.h>

#include <test_framework/GTest.h>

#include <QtGui/QMenu>
#include <QtXml/QtXml>
#include <QtCore/QProcess>

#define SETTINGS_ROOT QString("test_runner/")


namespace GB2 {

static LogCategory log(ULOG_CAT_PLUGIN_TEST_RUNNER);


extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    if (AppContext::getMainWindow()) {
    TestRunnerPlugin* plug = new TestRunnerPlugin();
    return plug;
    }
    return NULL;
}


TestRunnerPlugin::TestRunnerPlugin() : Plugin(tr("test_runner_plug_name"), tr("test_runner_desc")) {
    TestRunnerScriptModule* sm=new TestRunnerScriptModule();
    sm->moduleName=QString("QTest");
    AppContext::getScriptModuleRegistry()->registerGScriptModule(sm);
    services.push_back(new TestRunnerService());
}
//////////////////////////////////////////////////////////////////////////
// service
TestRunnerService::TestRunnerService() 
: Service(Service_TestRunner, tr("sname"), tr("sdesc"))
{
    windowAction = NULL;
    view = NULL;
}

TestRunnerService::~TestRunnerService() {
    assert(suites.isEmpty());
}

void TestRunnerService::serviceStateChangedCallback(ServiceState oldState, bool enabledStateChanged) {
    Q_UNUSED(oldState);

    if (!enabledStateChanged) {
        return;
    }
    //QStringList ugene_env = QProcess::systemEnvironment();
    //bool test_runner_enabled = -1 != ugene_env.indexOf( QRegExp(QString(ENV_UGENE_DEV)+"*", Qt::CaseInsensitive, QRegExp::Wildcard) );

    if (isEnabled()) {
        assert(view==NULL);
        assert(windowAction == NULL);

        env = new GTestEnvironment();
        readSavedSuites();
        readBuiltInVars();

        windowAction = new QAction(tr("Test runner"), this);
        windowAction->setObjectName("action__testrunner");
        connect(windowAction, SIGNAL(triggered()), SLOT(sl_showWindow()));
        AppContext::getMainWindow()->getMenuManager()->getTopLevelMenu(MWMENU_TOOLS)->addAction(windowAction);
    } else {
        assert(windowAction!=NULL);
        delete windowAction;
        windowAction = NULL;
        
        saveSuites();
        saveEnv();
        deallocateSuites();
        delete env;
        env = NULL;


        if (view!=NULL) {
            view->killAllChildForms();
            AppContext::getMainWindow()->getMDIManager()->closeMDIWindow(view);
            assert(view == NULL);
        }
    }
}


void TestRunnerService::sl_showWindow() {
    assert(isEnabled());
    if (view==NULL) {
        view = new TestViewController(this);
        view->installEventFilter(this);
        AppContext::getMainWindow()->getMDIManager()->addMDIWindow(view);
    }
    AppContext::getMainWindow()->getMDIManager()->activateWindow(view);
}


bool TestRunnerService::eventFilter(QObject *obj, QEvent *event) {
    if (event->type() == QEvent::Close && obj == view) {
        view->killAllChildForms();
        view = NULL;
    }
    return QObject::eventFilter(obj, event);
}


void TestRunnerService::addTestSuite(GTestSuite *ts) {
    assert(!findTestSuiteByURL(ts->getURL()));
    assert(!suites.contains(ts));
    suites.append(ts);
    
    GTestEnvironment * tsEnv = ts->getEnv();
    const QStringList & tsEnvKeys = tsEnv->getVars().keys();
    QStringList tsEnvResultedKeys;
    //skipping non-empty variables
    foreach( const QString & key, tsEnvKeys ) {
        if( tsEnv->getVar(key).isEmpty() ) {
            tsEnvResultedKeys.push_back( key );
        }
    }
    readEnvForKeys(tsEnvResultedKeys);
    updateDefaultEnvValues(ts);
    
    emit si_testSuiteAdded(ts);
}

void TestRunnerService::updateDefaultEnvValues(GTestSuite* ts) {
    QMap<QString, QString> vars = env->getVars();
    if (vars.contains("COMMON_DATA_DIR") && vars.value("COMMON_DATA_DIR").isEmpty()) {
        env->setVar("COMMON_DATA_DIR", QFileInfo(ts->getURL()).absoluteDir().absolutePath() + "/_common_data");
    }

    if (vars.contains("TEMP_DATA_DIR") && vars.value("TEMP_DATA_DIR").isEmpty()) {
        env->setVar("TEMP_DATA_DIR", QFileInfo(ts->getURL()).absoluteDir().absolutePath()+"/_tmp");
    }
}

void TestRunnerService::removeTestSuite(GTestSuite* ts) {
    assert(suites.contains(ts));
    suites.removeOne(ts);

    //todo: cleanup vars, but leave built-in
    saveEnv();

    emit si_testSuiteRemoved(ts);
}


GTestSuite* TestRunnerService::findTestSuiteByURL(const QString& url) {
    foreach(GTestSuite* t, suites) {
        if (t->getURL() == url) {
            return t;
        }
    }
    return NULL;
}

void TestRunnerService::readBuiltInVars() {
    QStringList biVars;
    biVars<<NUM_THREADS_VAR;
    readEnvForKeys(biVars);

    QMap<QString, QString> vars = env->getVars();
    if (!vars.contains(NUM_THREADS_VAR) || vars.value(NUM_THREADS_VAR).isEmpty()) {
        env->setVar(NUM_THREADS_VAR, "5");
    }
    if (!vars.contains(TIME_OUT_VAR) || vars.value(TIME_OUT_VAR).isEmpty()) {
        env->setVar(TIME_OUT_VAR, "0");
    }
}

void TestRunnerService::readSavedSuites() {
    //TODO: do it in in service startup task!!!

    QStringList suiteUrls = AppContext::getSettings()->getValue(SETTINGS_ROOT + "suites", QStringList()).toStringList();
    QString err;
    QMap<QString, QString> env;
    QString url;
    foreach(const QString& url, suiteUrls) {
        GTestSuite* ts = GTestSuite::readTestSuite(url, err);
        if (ts == NULL) {
            log.error(tr("error_reading_ts_%1_error_%2").arg(url).arg(err));
        } else {
            addTestSuite(ts);
        }
    }
}

void TestRunnerService::saveSuites() {
    QStringList list;
    foreach(GTestSuite* s, suites) {
        list.append(s->getURL());
    }
    AppContext::getSettings()->setValue(SETTINGS_ROOT + "suites", list);
}

void TestRunnerService::deallocateSuites() {
    foreach(GTestSuite* s, suites) {
        emit si_testSuiteRemoved(s);
        delete s;
    }
    suites.clear();
}

void TestRunnerService::readEnvForKeys(QStringList keys) {
    foreach(const QString& k, keys) {
        QString val = env->getVar(k);
        if (val.isEmpty()) {
            val = AppContext::getSettings()->getValue(SETTINGS_ROOT + "env/"+ k, QString()).toString();
            env->setVar(k, val);
        }
    }
}

void TestRunnerService::saveEnv() {
    foreach(const QString& k, env->getVars().keys()) {
        QString val = env->getVar(k);
        if (!val.isEmpty()) {
            AppContext::getSettings()->setValue(SETTINGS_ROOT + "env/"+ k, val);
        } else {
            AppContext::getSettings()->remove(SETTINGS_ROOT + "env/"+ k);
        }
    }
}

void TestRunnerService::sl_refresh()
{
    saveSuites();
    deallocateSuites();
    readSavedSuites();
}
//////////////////////////////////////////////////////////////////////////
// Script Module
void TestRunnerScriptModule::setup(QScriptEngine *engine) const{
    GTestScriptWrapper::setQTest(engine);
}
}//namespace
