/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "DNAExportPlugin.h"

#include "ExportDialogController.h"
#include "ExportMSA2SequenceDialog.h"
#include "DNAExportTask.h"

#include <core_api/MainWindow.h>
#include <core_api/ProjectView.h>
#include <core_api/AppContext.h>
#include <core_api/MainWindow.h>
#include <core_api/DocumentFormats.h>
#include <core_api/DNATranslation.h>
#include <util_algorithm/MSAUtils.h>

#include <util_gui/DialogUtils.h>
#include <util_gui/GUIUtils.h>
#include <util_text/TextUtils.h>

#include <util_ov_annotated_dna/AnnotatedDNAView.h>
#include <util_ov_annotated_dna/ADVSequenceObjectContext.h>
#include <util_ov_annotated_dna/ADVConstants.h>

#include <selection/GObjectSelection.h>
#include <selection/DocumentSelection.h>
#include <selection/SelectionUtils.h>
#include <selection/AnnotationSelection.h>
#include <selection/DNASequenceSelection.h>

#include <gobjects/GObjectTypes.h>
#include <gobjects/AnnotationTableObject.h>
#include <gobjects/DNASequenceObject.h>
#include <gobjects/GObjectUtils.h>
#include <gobjects/MAlignmentObject.h>


#include <QtGui/QMenu>
#include <QtGui/QMessageBox>
#include <QtGui/QFileDialog>

namespace GB2 {

extern "C" Q_DECL_EXPORT Plugin* GB2_PLUGIN_INIT_FUNC() {
    if (AppContext::getMainWindow()) {
        DNAExportPlugin* plug = new DNAExportPlugin();
        return plug;
    }
    return NULL;
}

DNAExportPlugin::DNAExportPlugin() : Plugin(tr("dna_export_name"), tr("dna_export_desc")) {
    services.push_back(new DNAExportService());
}

//////////////////////////////////////////////////////////////////////////
// Service
DNAExportService::DNAExportService() 
: Service(Service_DNAExport, tr("dna_export_service_name"), tr("dna_export_service_desc"), QList<ServiceType>() << Service_ProjectView)
{
    exportSequencesToFastaAction = NULL;
    exportSequencesToClustalAction = NULL;
    exportAlignmentToFastaAction = NULL;
    viewContext = NULL;
}

void DNAExportService::serviceStateChangedCallback(ServiceState oldState, bool enabledStateChanged) {
    Q_UNUSED(oldState);

    if (!enabledStateChanged) {
        return;
    }
    if (isEnabled()) {
        exportSequencesToFastaAction = new QAction(tr("Export to FASTA"), this);
        connect(exportSequencesToFastaAction, SIGNAL(triggered()), SLOT(sl_saveSequencesToFasta()));

        exportSequencesToClustalAction = new QAction(tr("Export to CLUSTAL"), this);
        connect(exportSequencesToClustalAction, SIGNAL(triggered()), SLOT(sl_saveSequencesToClustal()));

        exportAlignmentToFastaAction = new QAction(tr("Export alignment to FASTA"), this);
        connect(exportAlignmentToFastaAction, SIGNAL(triggered()), SLOT(sl_saveAlignmentToFasta()));

        ProjectView* pv = AppContext::getProjectView();
        assert(pv!=NULL);
        connect(pv, SIGNAL(si_onDocTreePopupMenuRequested(QMenu&)), SLOT(sl_addToProjectViewMenu(QMenu&)));
        viewContext = new DNAExportViewContext(this);
        viewContext->init();
    } else {
        ProjectView* pv = AppContext::getProjectView();
        assert(pv!=NULL);
        pv->disconnect(this);
        delete exportSequencesToFastaAction; exportSequencesToFastaAction = NULL;
        delete exportSequencesToClustalAction; exportSequencesToClustalAction = NULL;
        delete exportAlignmentToFastaAction; exportAlignmentToFastaAction = NULL;
        delete viewContext; viewContext = NULL;
    }
}

void DNAExportService::sl_addToProjectViewMenu(QMenu& m) {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);
    QMenu* sub = NULL;

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, &ms, UOF_LoadedOnly);
    if (!set.isEmpty()) {
        sub = new QMenu(tr("Export"));
        sub->addAction(exportSequencesToFastaAction);
        sub->addAction(exportSequencesToClustalAction);
    } else {
        set = SelectionUtils::findObjects(GObjectTypes::MULTIPLE_ALIGNMENT, &ms, UOF_LoadedOnly);
        if (set.size() == 1) {
            sub = new QMenu(tr("Export"));
            sub->addAction(exportAlignmentToFastaAction);
        }
    }

    if (sub!=NULL) {
        QAction* beforeAction = GUIUtils::findActionAfter(m.actions(), ACTION_PROJECT__ADD_MENU);
        m.insertMenu(beforeAction, sub);
    }
}

static bool hasComplementForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findComplementTT(so) == NULL) {
            return false;
        } 
    }
    return true;
}

static bool hasAminoForAll(const QSet<GObject*>& set) {
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        if (o == NULL || GObjectUtils::findAminoTT(so, false) == NULL) {
            return false;
        } 
    }
    return true;
}

void DNAExportService::sl_saveSequencesToFasta() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, &ms, UOF_LoadedOnly);
    if (set.isEmpty()) {
        QMessageBox::critical(NULL, tr("Error!"), tr("No sequence objects selected"));
        return;
    }
    bool allowMerge = set.size() > 1;
    bool allowComplement = hasComplementForAll(set);
    bool allowTranslate = hasAminoForAll(set);
        
    ExportDialogController d(allowMerge, allowComplement, allowTranslate, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.allAminoStrands = d.translateAllFrames;
    s.strand = d.strand;
    foreach(GObject* o, set) {
        DNASequenceObject* so = qobject_cast<DNASequenceObject*>(o);
        assert(so!=NULL);
        s.names.append(so->getGObjectName());
        s.alphabets.append(so->getAlphabet());
        s.sequences.append(so->getSequence());
        s.complTranslations.append(GObjectUtils::findComplementTT(so));
        s.aminoTranslations.append(d.translate ? GObjectUtils::findAminoTT(so, false) : NULL);
        //FIXME meta info is lost
    }
    
    Task* t = new DNAExportSequenceTask(s);
//    t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void DNAExportService::sl_saveSequencesToClustal() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::DNA_SEQUENCE, &ms, UOF_LoadedOnly);
    if (set.isEmpty()) {
        QMessageBox::critical(NULL, tr("Error!"), tr("No sequence objects selected"));
        return;
    }

    QString err;
    MAlignment ma = MSAUtils::seq2ma(set.toList(), err);
    if (!err.isEmpty()) {
        QMessageBox::critical(NULL, tr("error"), err);
        return;
    }
    
    QString filter = DialogUtils::prepareDocumentsFileFilter(BaseDocumentFormats::CLUSTAL_ALN, false);
    LastOpenDirHelper lod;
    lod.url = QFileDialog::getSaveFileName(NULL, tr("select_file_title"), lod, filter);
    if (lod.url.isEmpty()) {
        return;
    }

    Task* t = new DNAExportAlignmentTask(ma, lod.url);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void DNAExportService::sl_saveAlignmentToFasta() {
    ProjectView* pv = AppContext::getProjectView();
    assert(pv!=NULL);

    MultiGSelection ms; ms.addSelection(pv->getGObjectSelection()); ms.addSelection(pv->getDocumentSelection());
    QSet<GObject*> set = SelectionUtils::findObjects(GObjectTypes::MULTIPLE_ALIGNMENT, &ms, UOF_LoadedOnly);
    if (set.size()!=1) {
        QMessageBox::critical(NULL, tr("Error!"), tr("Select one alignment object to export"));
        return;
    }
    GObject* obj = set.toList().first();
    MAlignment ma = qobject_cast<MAlignmentObject*>(obj)->getMAlignment();
    ExportMSA2SequenceDialog d;
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    ExportAlignment2Sequence* exportTask = new ExportAlignment2Sequence(ma, d.url, d.trimGaps);
    AppContext::getTaskScheduler()->registerTopLevelTask(exportTask);
}

//////////////////////////////////////////////////////////////////////////
// DNAExportViewContext

DNAExportViewContext::DNAExportViewContext(QObject* p) 
: GObjectViewWindowContext(p, ANNOTATED_DNA_VIEW_FACTORY_ID)
{
}

void DNAExportViewContext::initViewContext(GObjectView* v) {

    GObjectViewAction *a1 = new GObjectViewAction(this, v, tr("export_sequence_selection"));
    connect(a1, SIGNAL(triggered()), SLOT(sl_saveSequenceToFasta()));

    GObjectViewAction *a2 = new GObjectViewAction(this, v, tr("export_annotation_selection"));
    connect(a2, SIGNAL(triggered()), SLOT(sl_saveAnnotationsToFasta()));

    addViewAction(a1);
    addViewAction(a2);
}

void DNAExportViewContext::buildMenu(GObjectView* v, QMenu* m) {
    QMenu* sub = GUIUtils::findSubMenu(m, ADV_MENU_EXPORT);
    QList<GObjectViewAction*> list = getViewActions(v);
    assert(!list.isEmpty());
    foreach(GObjectViewAction* a, list) {
        sub->addAction(a);
    }
}


void DNAExportViewContext::sl_saveAnnotationsToFasta() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);
    AnnotationSelection* as = av->getAnnotationsSelection();
    AnnotationGroupSelection* ags = av->getAnnotationsGroupSelection();

    QSet<Annotation*> annotations;

    const QList<AnnotationSelectionData>& aData = as->getSelection();
    foreach(const AnnotationSelectionData& ad, aData) {
        annotations.insert(ad.annotation);
    }

    const QList<AnnotationGroup*>& groups =  ags->getSelection();
    foreach(AnnotationGroup* g, groups) {
        g->findAllAnnotationsInGroupSubTree(annotations);
    }

    if (annotations.isEmpty()) {
        QMessageBox::warning(av->getWidget(), tr("warning"), tr("no_annotation_selected"));
        return;
    }
    bool merge = true;
    QSetIterator<Annotation*> iter(annotations);
    const bool strand = iter.next()->isOnComplementStrand();
    while (iter.hasNext()) {
        if (iter.next()->isOnComplementStrand() != strand) {
            merge = false;
            break;
        }
    }
    bool allowComplement = false;
    bool allowTranslation = true;
    ExportDialogController d(merge, allowComplement, allowTranslation, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    QHash<QString, TriState> usedNames;
    foreach(Annotation* a, annotations) {
        const QString& name = a->getAnnotationName();
        usedNames[name] = usedNames.contains(name) ? TriState_Yes : TriState_No;
    }
    
    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.strand = d.strand;
    s.allAminoStrands = d.translateAllFrames;

    QList<Annotation*> sortedAnnotations = annotations.toList();
    qStableSort(sortedAnnotations.begin(), sortedAnnotations.end(), strand ? annotationGreaterThanByRegion : annotationLessThanByRegion);

    foreach(Annotation* a, sortedAnnotations) {
        ADVSequenceObjectContext* seqCtx = av->getSequenceContext(a->getGObject());
        if (seqCtx == NULL) {
            continue;
        } 
        DNATranslation* complTT = a->isOnComplementStrand() ? seqCtx->getComplementTT() : NULL;
        DNATranslation* aminoTT = d.translate ? seqCtx->getAminoTT() : NULL;
        LRegion seqReg(0, seqCtx->getSequenceLen());
        const QByteArray& sequence = seqCtx->getSequenceData();

        const QList<LRegion>& location = a->getLocation();
        bool multi = location.size() > 1;
        for (int i=0; i < location.size(); i++) {
            QString prefix = a->getAnnotationName();
            if (multi) {
                prefix+=QString(" part %1 of %2").arg(QString::number(i+1)).arg(QString::number(location.size()));
            }
            QString name = prefix;
            for (int j=0; usedNames.contains(name) && usedNames.value(name) == TriState_Yes; j++) {
                name = prefix + "|" + QString::number(j);
            }
            usedNames[name] = TriState_Yes;
            s.names.append(name);

            LRegion reg = location[i].intersect(seqReg);
            QByteArray partSeq = sequence.mid(reg.startPos, reg.len);//TODO: mid() creates a copy -> optimize!!
            if (complTT!=NULL) {
                complTT->translate(partSeq.data(), partSeq.length());
                TextUtils::reverse(partSeq.data(), partSeq.length()); //todo: do it not in the main thread, but in task!!
            }
            s.sequences.append(partSeq);
            s.alphabets.append(seqCtx->getAlphabet());
            s.complTranslations.append(complTT);
            s.aminoTranslations.append(aminoTT);
        }
    }
    assert(s.names.size() > 0);
    Task* t = new DNAExportSequenceTask(s);
    //t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}

void DNAExportViewContext::sl_saveSequenceToFasta() {
    QAction* a = (QAction*)sender();
    GObjectViewAction* viewAction = qobject_cast<GObjectViewAction*>(a);
    AnnotatedDNAView* av = qobject_cast<AnnotatedDNAView*>(viewAction->getObjectView());
    assert(av);
    ADVSequenceObjectContext* seqCtx = av->getSequenceInFocus();
    DNASequenceSelection* sel  = NULL;
    if (seqCtx!=NULL) {
        //TODO: support multi-export..
        sel = seqCtx->getSequenceSelection();
    }
    if (sel == NULL || sel->isEmpty()) {
        QMessageBox::warning(av->getWidget(), tr("warning"), tr("no_sequence_selected"));
        return;
    }

    const QList<LRegion>& regions =  sel->getSelectedRegions();
    bool merge = regions.size() > 1;
    bool complement = seqCtx->getComplementTT()!=NULL;
    bool amino = seqCtx->getAminoTT()!=NULL;
    ExportDialogController d(merge, complement, amino, BaseDocumentFormats::PLAIN_FASTA);
    int rc = d.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    assert(d.file.length() > 0);

    const QByteArray& sequence = seqCtx->getSequenceData();
    DNAAlphabet* al = seqCtx->getAlphabet();
    DNAExportTaskSettings s;
    s.fileName = d.file;
    s.merge = d.merge;
    s.mergeGap = d.mergeGap;
    s.strand = d.strand;
    s.allAminoStrands = d.translateAllFrames;
    foreach(const LRegion& r, regions) {
        QString prefix = QString("region [%1 %2]").arg(QString::number(r.startPos+1)).arg(QString::number(r.endPos()));
        QString name = prefix;
        for (int i=0; s.names.contains(name); i++) {
            name = prefix + "|" + QString::number(i);
        }
        s.names.append(name);
        s.alphabets.append(al);
        s.sequences.append(QByteArray(sequence.constData() + r.startPos, r.len)); //todo: optimize to avoid copying!!
        s.complTranslations.append(seqCtx->getComplementTT());
        s.aminoTranslations.append(d.translate ? seqCtx->getAminoTT() : NULL);
    }
    Task* t = new DNAExportSequenceTask(s);
    //t->setUserNotificationRequired(true);
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}



}//namespace
