/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "WorkflowTests.h"
#include "WorkflowIOTasks.h"

#include <workflow/WorkflowEnv.h>
#include <workflow_support/WorkflowUtils.h>
#include <workflow_support/WorkflowRunTask.h>

#include <QtGui/QListWidgetItem>

namespace GB2 {

QList<XMLTestFactory*> GB2::WorkflowTests::createTestFactories()
{
    QList<XMLTestFactory*> res;
    res.append(GTest_LoadWorkflow::createFactory());
    res.append(GTest_SaveWorkflow::createFactory());
    res.append(GTest_LoadBrokenWorkflow::createFactory());
    res.append(GTest_RunWorkflow::createFactory());
    res.append(GTest_ValidateWorkflow::createFactory());
    res.append(GTest_ConfigureWorkflow::createFactory());
    
    return res;
}

#define DOC_ATTR   "doc"
#define NAME_ATTR   "name"

void GTest_LoadWorkflow::init(XMLTestFormat*, const QDomElement& el) {
    contextAdded = false;
    loadTask = NULL;
    docContextName = el.attribute("index");

    QString url = el.attribute("url");
    if (url.isEmpty()) {
        failMissingValue("url");
        return;
    }

    QString dir = el.attribute("dir");
    if (dir == "temp"){
        tempFile=true;
        url = env->getVar("TEMP_DATA_DIR") + "/" + url;
    }
    else{
        tempFile=false;
        url = env->getVar("COMMON_DATA_DIR") + "/" + url;
    }
    Schema* sh = new Schema();
    sh->deepCopy = true;
    loadTask = new LoadWorkflowTask(sh, NULL, url);
    addSubTask(loadTask);
}

Task::ReportResult GTest_LoadWorkflow::report() {
    if (loadTask!=NULL && loadTask->hasErrors()) {
        stateInfo.setError( loadTask->getError());
    } else if (!docContextName.isEmpty()) {
        SchemaHandle* sh = new SchemaHandle(loadTask->getSchema());
        sh->remapping = loadTask->getRemap();
        addContext(docContextName, sh);
        contextAdded = true;
    }
    return ReportResult_Finished;
}

void GTest_LoadWorkflow::cleanup() {
    if (contextAdded) {
        delete getContext(docContextName);
        removeContext(docContextName);
    }
    if(tempFile){
        QFile::remove(url);
    }
}

void GTest_SaveWorkflow::init(XMLTestFormat*, const QDomElement& el) {
    saveTask = NULL;

    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        failMissingValue(DOC_ATTR);
        return;
    }
    url = el.attribute("url");
    if (url.isEmpty()) {
        failMissingValue("url");
        return;
    }
    url = env->getVar("TEMP_DATA_DIR") + "/" + url;
}

void GTest_SaveWorkflow::prepare() {
    SchemaHandle* doc = getContext<SchemaHandle>(this, docContextName);
    if (doc == NULL) {
        stateInfo.setError( QString("document not found %1").arg(docContextName));
        return ;
    }
    Metadata m; m.url = url;
    saveTask = new SaveWorkflowTask(doc->schema, m);
    addSubTask(saveTask);
}


void GTest_LoadBrokenWorkflow::init(XMLTestFormat*, const QDomElement& el) {
    loadTask = NULL;
    QString url = el.attribute("url");
    if (url.isEmpty()) {
        failMissingValue("url");
        return;
    }

    QString dir = el.attribute("dir");
    if (dir == "temp"){
        url = env->getVar("TEMP_DATA_DIR") + "/" + url;
    }
    else{
        url = env->getVar("COMMON_DATA_DIR") + "/" + url;
    }

    Schema* sh = new Schema();
    sh->deepCopy = true;
    loadTask = new LoadWorkflowTask(sh, NULL, url);
    addSubTask(loadTask);
}

Task::ReportResult GTest_LoadBrokenWorkflow::report() {
    if (!loadTask->hasErrors()) {
        stateInfo.setError(QString("file read successfully [%1]").arg(loadTask->getUrl()));
    }
    return ReportResult_Finished;
}

void GTest_LoadBrokenWorkflow::cleanup() {
    if (loadTask) {
        delete loadTask->getSchema();
    }
}

void GTest_RunWorkflow::init(XMLTestFormat *, const QDomElement& el) {
    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        failMissingValue(DOC_ATTR);
        return;
    }
}

void GTest_RunWorkflow::prepare()
{
    SchemaHandle* doc = getContext<SchemaHandle>(this, docContextName);
    if (doc == NULL) {
        stateInfo.setError( QString("document not found %1").arg(docContextName));
        return;
    }
    Schema& sh = *doc->schema;
    if (sh.domain.isEmpty()) {
        sh.domain = WorkflowEnv::getDomainRegistry()->getAllIds().value(0);
    }
    addSubTask(new WorkflowRunTask(sh, sh.iterations));
}

// Task::ReportResult GTest_RunWorkflow::report() {
//     propagateSubtaskError();
//     return ReportResult_Finished;
// }

void GTest_ValidateWorkflow::init(XMLTestFormat *, const QDomElement& el) {
    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        failMissingValue(DOC_ATTR);
        return;
    }

    QDomNodeList procNodes = el.elementsByTagName("error");
    for(int i=0; i<procNodes.size(); i++) {
        QDomElement it = procNodes.item(i).toElement();
        if (it.isNull()) {
            continue;
        }
        QString id = it.attribute("actor");
        if (id.isEmpty()) {
            failMissingValue("actor of element error");
            return;
        }
        RefMap map;
        map[ACTOR_REF] = id;
        if (it.hasAttribute("port")) {
            QString port = it.attribute("port");
            map[PORT_REF] = port;
        }
        if (it.hasAttribute("iteration")) {
            QString s = it.attribute("iteration");
            map[ITERATION_REF] = s;
        }
        expectedErrors.append(map);
    }
}

Task::ReportResult GTest_ValidateWorkflow::report()
{
    SchemaHandle* doc = getContext<SchemaHandle>(this, docContextName);
    if (doc == NULL) {
        stateInfo.setError(QString("document not found %1").arg(docContextName));
        return ReportResult_Finished;
    }
    if (doc->schema->iterations.isEmpty()) {
        doc->schema->iterations << Iteration("Empty");
    }
    QList<QListWidgetItem*> lst;
    const bool valid = DesignerUtils::validate(*doc->schema, &lst);
    Q_UNUSED(valid);
//     if (valid && !expectedErrors.isEmpty()) {
//         stateInfo.setError(QString("Validation result mismatch for %1").arg(docContextName));
//         return ReportResult_Finished;
//     }

    foreach(QListWidgetItem* it, lst) {
        QVariant id = it->data(ACTOR_REF);
        QVariant port = it->data(PORT_REF);
        QVariant iteration = it->data(ITERATION_REF);
        bool found = false;
        for (int i = 0; i < expectedErrors.size(); ++i) {
            const RefMap& map = expectedErrors.at(i);
            ActorId rid = doc->remapping.value(str2aid(map.value(ACTOR_REF).toString()));
            if (id == rid
                && port == map.value(PORT_REF)
                && iteration == map.value(ITERATION_REF)) 
            {
                expectedErrors.takeAt(i);
                found = true;
                break;
            }
        }
        if (!found) {
            stateInfo.setError("Unexpected error: " + it->text());
            break;
        }
    }
    if (!hasErrors() && expectedErrors.size() != 0) {
        QStringList sl;
        foreach(const RefMap& map, expectedErrors) {
            sl << QString("actor=%1 port=%2 iter=%3")
                .arg(map.value(ACTOR_REF).toString())
                .arg(map.value(PORT_REF).toString())
                .arg(map.value(ITERATION_REF).toString());
        }
        stateInfo.setError(QString("Expected errors not met: %1").arg(sl.join("/n")));
    }
    qDeleteAll(lst);

    return ReportResult_Finished;
}

// void GTest_ValidateWorkflow::cleanup() {
//     qDeleteAll(expectedErrors);
//     expectedErrors.clear();
// }

QVariant GTest_ConfigureWorkflow::getValue(const QDomElement& el) {
    if (el.hasAttribute("value")) {
        return el.attribute("value");
    }
    if (el.hasAttribute("file")) {
        return env->getVar("COMMON_DATA_DIR") + "/" + el.attribute("file");
    }
    if (el.hasAttribute("tmpfile")) {
        return env->getVar("TEMP_DATA_DIR") + "/" + el.attribute("tmpfile");
    }
    return QVariant();
}

void GTest_ConfigureWorkflow::init(XMLTestFormat *, const QDomElement& el) {
    docContextName = el.attribute(DOC_ATTR);
    if (docContextName.isEmpty()) {
        failMissingValue(DOC_ATTR);
        return;
    }

    QDomNodeList procNodes = el.elementsByTagName("param");
    for(int i=0; i<procNodes.size(); i++) {
        QDomElement it = procNodes.item(i).toElement();
        if (it.isNull()) {
            continue;
        }
        ActorId id = str2aid(it.attribute("actor"));
        if (id.isEmpty()) {
            failMissingValue("actor of element param");
            return;
        }
        int iteration = -1;
        if (it.hasAttribute("iteration")) {
            QString s = it.attribute("iteration");
            bool ok;
            iteration = s.toInt(&ok);
            assert(ok);
        }

        QString name = it.attribute("name");
        if (name.isEmpty()) {
            failMissingValue("name of element param");
            return;
        }
        QVariant value = getValue(it);
        CfgMap& map = iteration != -1 ? iparams[iteration] : dparams;
        map[id].insert(name, value);
    }
}

Task::ReportResult GTest_ConfigureWorkflow::report()
{
    SchemaHandle* doc = getContext<SchemaHandle>(this, docContextName);
    if (doc == NULL) {
        stateInfo.setError(QString("document not found %1").arg(docContextName));
        return ReportResult_Finished;
    }

    Schema* sh = doc->schema;

    if (!dparams.isEmpty()) {
        QMapIterator<ActorId, QVariantMap> it(dparams);
        while (it.hasNext())
        {
            it.next();
            ActorId id = doc->remapping.value(it.key());
            Actor* a = sh->actorById(id);
            if (!a) {
                stateInfo.setError(QString("actor not found %1").arg(it.key()));
                return ReportResult_Finished;
            }
            a->setParameters(it.value());
        }
    }
    QMapIterator<int, CfgMap> jt(iparams);
    while (jt.hasNext())
    {
        jt.next();
        int idx = sh->iterationById(jt.key());
        if (idx < 0) {
            stateInfo.setError(QString("Iteration not found: %1").arg(jt.key()));
            return ReportResult_Finished;
        }
        Iteration& iteration = sh->iterations[idx];
        QMapIterator<ActorId, QVariantMap> it(jt.value());
        while (it.hasNext())
        {
            it.next();
            ActorId id = doc->remapping.value(it.key());
            if (id.isEmpty()) {
                stateInfo.setError(QString("No such actor: %1").arg(it.key()));
                return ReportResult_Finished;
            }
            iteration.cfg.insert(id, it.value());
        }
    }
    return ReportResult_Finished;
}

} //namespace
