/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/

#include "MSAEditorBaseOffsetsCache.h"

#include <gobjects/MAlignmentObject.h>

namespace GB2 {

//one of 256 positions contains offset info
#define CACHE_FREQ 256

MSAEditorBaseOffsetCache::MSAEditorBaseOffsetCache(QObject* p, MAlignmentObject* obj) 
: QObject(p), aliObj(obj)
{
    connect(aliObj, SIGNAL(si_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)), 
        SLOT(sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo&)));
    
    objVersion = 1;
    globVersion = 0;
}


void MSAEditorBaseOffsetCache::sl_alignmentChanged(const MAlignment&, const MAlignmentModInfo& modInfo) {
    if (modInfo.sequenceContentChanged) {
        objVersion++;
    }
}

int MSAEditorBaseOffsetCache::getBaseCounts(int seqNum, int aliPos, bool inclAliPos) {
    const MAlignment& ma = aliObj->getMAlignment();
    const MAlignmentItem& mai = ma.alignedSeqs[seqNum];
    const char* seq = mai.sequence.constData();
    int aliLen = ma.getLength();
    int i = 0;
    int cnt = _getBaseCounts(seqNum, aliPos, i);
    int endPos = inclAliPos ? aliPos+1 : aliPos;
    for (; i < endPos; i++) {
        if (seq[i] != MAlignment_GapChar) {
            cnt++;
        }
    }
    return cnt;
}

int  MSAEditorBaseOffsetCache::_getBaseCounts(int seqNum, int aliPos, int& cachedEndPos) {
    updateCacheSize();
    updateCacheRow(seqNum);
    const RowCache& r = cache[seqNum];
    assert(globVersion = objVersion);
    assert(r.cacheVersion == objVersion);
    int cacheIdx = aliPos / CACHE_FREQ  - 1;
    if (cacheIdx < 0) {
        cachedEndPos = 0;
        return 0;
    }
    cachedEndPos = (cacheIdx + 1) * CACHE_FREQ;
    int res = r.rowOffsets[cacheIdx];
    return res;
}

void MSAEditorBaseOffsetCache::updateCacheSize() {
    if (objVersion == globVersion) {
        return;
    }
    int nSeq = aliObj->getMAlignment().getNumSequences();
    cache.resize(nSeq);
    globVersion = objVersion;
}

void MSAEditorBaseOffsetCache::updateCacheRow(int seqNum) {
    assert(globVersion == objVersion);
    RowCache& r = cache[seqNum];
    if (r.cacheVersion == objVersion) {
        return;
    }
    const MAlignment& ma = aliObj->getMAlignment();
    int aliLen = ma.getLength();
    int nPoints = aliLen / CACHE_FREQ;
    r.rowOffsets.resize(nPoints);
    const MAlignmentItem& mai = ma.alignedSeqs[seqNum];
    const char* seq = mai.sequence.constData();
    for (int i=0, cnt = 0; i < aliLen; i++) {
        if (i != 0 && i % CACHE_FREQ == 0) {
            r.rowOffsets[i / CACHE_FREQ - 1] = cnt;
        }
        if (seq[i] != MAlignment_GapChar) {
            cnt++;
        }
    }
    r.cacheVersion = objVersion;
}

}//namespace

