/*****************************************************************
* Unipro UGENE - Integrated Bioinformatics Suite
* Copyright (C) 2008,2009 Unipro, Russia (http://ugene.unipro.ru)
* All Rights Reserved
* 
*     This source code is distributed under the terms of the
*     GNU General Public License. See the files COPYING and LICENSE
*     for details.
*****************************************************************/


#include <QtGui/QMenu>
#include <QtGui/QFileDialog>
#include <QtGui/QMessageBox>

#include <core_api/MainWindow.h>
#include <core_api/AppContext.h>
#include <core_api/IOAdapter.h>
#include <util_gui/DialogUtils.h>

#include <util_tasks/CreateFileIndexTask.h>

#include "UIndexSupport.h"

namespace {

using namespace GB2;

const QString GZIP_END = ".gz";
const QString UIND_END = ".uind";

void updateModel( CreateFileIndexDialogModel& m ) {
    if( m.gzippedOutput && !m.outputUrl.endsWith( GZIP_END ) ) {
        m.outputUrl.append( GZIP_END );
    }
    int sz = m.outputUrl.size();
    bool outIsGzip = m.outputUrl.endsWith( GZIP_END );
    QString outRawName = ( outIsGzip )? m.outputUrl.left( sz - GZIP_END.size() ): m.outputUrl;
    if( !outRawName.endsWith( UIND_END ) ) {
        outRawName.append( UIND_END );
    }
    m.outputUrl = ( outIsGzip )? outRawName + GZIP_END: outRawName;
}

} // anonymous namespace

namespace GB2 {

UIndexSupport::UIndexSupport() {
    QAction* showDlgAction = new QAction( tr("Create index file"), this );
    showDlgAction->setIcon(QIcon(":core/images/table_add.png"));
    connect( showDlgAction, SIGNAL( triggered() ), SLOT( sl_showCreateFileIndexDialog() ) );
    AppContext::getMainWindow()->getMenuManager()->getTopLevelMenu( MWMENU_TOOLS )->addAction( showDlgAction );
}

void UIndexSupport::sl_showCreateFileIndexDialog() {
    CreateFileIndexDialogModel model;
    
    CreateFileIndexDialog dlg(QApplication::activeWindow());
    int rc = dlg.exec();
    if (rc == QDialog::Rejected) {
        return;
    }
    model = dlg.getModel();
    updateModel( model );
    IOAdapterId inputIO  = BaseIOAdapters::url2io( model.inputUrl );
    IOAdapterId outputIO = BaseIOAdapters::url2io( model.outputUrl );
    if( inputIO.isEmpty() || outputIO.isEmpty() ) {
        QMessageBox::critical( NULL, tr( "error" ), tr("Unknown protocol" ) );
        return;
    }
    
    QList< QString > inputUrls;
    QList< IOAdapterFactory* > inputFactories;
    inputUrls.append( model.inputUrl );
    inputFactories.append( AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( inputIO ) );
    
    CreateFileIndexTask* t = new CreateFileIndexTask( inputUrls, model.outputUrl,
        inputFactories, AppContext::getIOAdapterRegistry()->getIOAdapterFactoryById( outputIO ) );
    AppContext::getTaskScheduler()->registerTopLevelTask(t);
}


CreateFileIndexDialog::CreateFileIndexDialog(QWidget* p): QDialog(p) {
    setupUi( this );
    connect( inputFileOpenButton,  SIGNAL( clicked() ), SLOT( sl_openInputFile() ) );
    connect( outputFileOpenButton, SIGNAL( clicked() ), SLOT( sl_openOutputFile() ) );
}

void CreateFileIndexDialog::sl_openInputFile() {
    LastOpenDirHelper h;
    h.url = QFileDialog::getOpenFileName( this, tr( "Open file" ), h.dir );
    if (!h.url.isEmpty()) {
        inputFileEdit->setText(h.url);
        outputFileEdit->setText(h.url + UIND_END);
    }
}

void CreateFileIndexDialog::sl_openOutputFile() {
    LastOpenDirHelper h;
    h.url = QFileDialog::getSaveFileName( this, tr( "Select index file to create" ), h.dir );
    if (!h.url.isEmpty()) {
        outputFileEdit->setText(h.url);
    }
}

void CreateFileIndexDialog::accept() {
    model.inputUrl  = inputFileEdit->text();
    model.outputUrl = outputFileEdit->text();
    model.gzippedOutput = ( Qt::Checked == gzipOutputCheckBox->checkState() );
    
    if (model.inputUrl.isEmpty()) { 
        QMessageBox::critical(this, tr("Error!"), tr("Invalid input file name!"));
        inputFileEdit->setFocus();
        return;
    }
    if (model.outputUrl.isEmpty()) { 
        QMessageBox::critical(this, tr("Error!"), tr("Invalid output file name!"));
        outputFileEdit->setFocus();
        return;
    }
    QDialog::accept();
}


CreateFileIndexDialogModel CreateFileIndexDialog::getModel() const {
    return model;
}


} // GB2
